<?php

namespace Drupal\postoffice_compat\Email;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\contact\ContactFormInterface;
use Drupal\postoffice\Email\LocalizedEmailInterface;
use Drupal\postoffice\Email\SiteEmailInterface;
use Drupal\postoffice\Email\SiteEmailTrait;
use Drupal\postoffice\Email\TemplateAttachmentsInterface;
use Drupal\postoffice\Email\TemplateAttachmentsTrait;
use Drupal\postoffice\Email\ThemedEmailInterface;
use Drupal\postoffice\Email\UrlOptionsTrait;
use Symfony\Component\Mime\Email;

/**
 * Postoffice email for autoreply messages generated by the core contact module.
 */
class ContactAutoreplyEmail extends Email implements ContactFormEmailInterface, LocalizedEmailInterface, SiteEmailInterface, TemplateAttachmentsInterface, ThemedEmailInterface {

  use CompatEmailTrait;
  use ContactFormEmailTrait;
  use SiteEmailTrait;
  use TemplateAttachmentsTrait;
  use UrlOptionsTrait;

  /**
   * Language code for this message.
   */
  protected string $langcode;

  /**
   * Constructs a new email for messages generated by the core user module.
   */
  public function __construct(
    string $langcode,
    ContactFormInterface $contactForm,
    ?ImmutableConfig $siteConfig = NULL,
    ?LanguageManagerInterface $languageManager = NULL,
  ) {
    parent::__construct();
    $this->langcode = $langcode;
    $this->contactForm = $contactForm;
    $this->siteConfig = $siteConfig;
    $this->languageManager = $languageManager;
  }

  /**
   * Constructs a new contact autoreply email from a core message structure.
   */
  public static function createFromMessage(array $message): static {
    return (new static($message['langcode'], $message['params']['contact_form']))
      ->headersFromMessage($message)
      ->to($message['to'])
      ->subject($message['subject']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildThemedEmail(): ?array {
    return [
      '#theme' => 'postoffice_compat_email_contact_autoreply',
      '#body' => [
        '#type' => 'processed_text',
        '#text' => $this->getContactForm()->getReply(),
        '#format' => 'postoffice_compat_email_fallback',
        '#langcode' => $this->langcode,
      ],
      '#contact_form_id' => $this->contactForm->id(),
      '#email' => $this,
      '#langcode' => $this->langcode,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getLangcode(): string {
    return $this->langcode;
  }

}
