<?php

namespace Drupal\postoffice_compat\Email;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\postoffice\Email\LocalizedEmailInterface;
use Drupal\postoffice\Email\SiteEmailInterface;
use Drupal\postoffice\Email\SiteEmailTrait;
use Drupal\postoffice\Email\TemplateAttachmentsInterface;
use Drupal\postoffice\Email\TemplateAttachmentsTrait;
use Drupal\postoffice\Email\ThemedEmailInterface;
use Drupal\postoffice\Email\UrlOptionsTrait;
use Drupal\user\UserInterface;
use Symfony\Component\Mime\Email;

/**
 * Postoffice email for messages generated by the core user module.
 */
class UserEmail extends Email implements LocalizedEmailInterface, SiteEmailInterface, TemplateAttachmentsInterface, ThemedEmailInterface, UserEmailInterface {

  use CompatEmailTrait;
  use SiteEmailTrait;
  use TemplateAttachmentsTrait;
  use UrlOptionsTrait;

  /**
   * The drupal core email key.
   */
  protected string $key;

  /**
   * Language code for this message.
   */
  protected string $langcode;

  /**
   * The account this message is sent to.
   */
  protected UserInterface $account;

  /**
   * Constructs a new email for messages generated by the core user module.
   */
  public function __construct(
    string $key,
    string $langcode,
    UserInterface $account,
    ?ImmutableConfig $siteConfig = NULL,
    ?LanguageManagerInterface $languageManager = NULL,
  ) {
    parent::__construct();
    $this->key = $key;
    $this->langcode = $langcode;
    $this->account = $account;
    $this->siteConfig = $siteConfig;
    $this->languageManager = $languageManager;
  }

  /**
   * Constructs a new user email from a core message structure.
   */
  public static function createFromMessage(array $message): static {
    return (new static($message['key'], $message['langcode'], $message['params']['account']))
      ->headersFromMessage($message)
      ->to($message['to'])
      ->subject($message['subject'])
      ->html($message['body']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildThemedEmail(): ?array {
    return [
      '#theme' => 'postoffice_compat_email_user',
      '#body' => [
        '#markup' => Markup::create($this->getHtmlBody()),
      ],
      '#email' => $this,
      '#key' => $this->key,
      '#langcode' => $this->langcode,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getLangcode(): string {
    return $this->langcode;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountCancelUrl(): string {
    return user_cancel_url($this->account, ['langcode' => $this->getLangcode()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountOneTimeLoginUrl(): string {
    return user_pass_reset_url($this->account, ['langcode' => $this->getLangcode()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountCanonicalUrl(): string {
    return $this->account
      ->toUrl('canonical', $this->getUrlOptions())
      ->setAbsolute(TRUE)
      ->toString();
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountEditUrl(): string {
    return $this->account
      ->toUrl('edit-form', $this->getUrlOptions())
      ->setAbsolute(TRUE)
      ->toString();
  }

  /**
   * {@inheritdoc}
   */
  public function getAccount(): UserInterface {
    return $this->account;
  }

}
