<?php

namespace Drupal\postoffice_compat\Email;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\contact\MessageInterface;
use Drupal\postoffice\Email\LocalizedEmailInterface;
use Drupal\postoffice\Email\SiteEmailInterface;
use Drupal\postoffice\Email\SiteEmailTrait;
use Drupal\postoffice\Email\TemplateAttachmentsInterface;
use Drupal\postoffice\Email\TemplateAttachmentsTrait;
use Drupal\postoffice\Email\ThemedEmailInterface;
use Drupal\postoffice\Email\UrlOptionsTrait;
use Drupal\user\UserInterface;
use Symfony\Component\Mime\Email;

/**
 * Postoffice email for personal messages generated by the core contact module.
 */
class ContactPersonalEmail extends Email implements ContactMessageEmailInterface, ContactPersonalEmailInterface, LocalizedEmailInterface, SiteEmailInterface, TemplateAttachmentsInterface, ThemedEmailInterface {

  use CompatEmailTrait;
  use ContactMessageEmailTrait;
  use SiteEmailTrait;
  use TemplateAttachmentsTrait;
  use UrlOptionsTrait;

  /**
   * Language code for this message.
   */
  protected string $langcode;

  /**
   * Constructs a new email for messages generated by the core user module.
   */
  public function __construct(
    string $langcode,
    MessageInterface $contactMessage,
    UserInterface $sender,
    bool $isCopy,
    ?ImmutableConfig $siteConfig = NULL,
    ?LanguageManagerInterface $languageManager = NULL,
  ) {
    parent::__construct();
    $this->langcode = $langcode;
    $this->contactMessage = $contactMessage;
    $this->contactSender = $sender;
    $this->contactCopy = $isCopy;
    $this->siteConfig = $siteConfig;
    $this->languageManager = $languageManager;
  }

  /**
   * Constructs a new contact page email from a core message structure.
   */
  public static function createFromMessage(array $message, bool $isCopy): static {
    return (new static(
      $message['langcode'],
      $message['params']['contact_message'],
      $message['params']['sender'],
      $isCopy
    ))->headersFromMessage($message)
      ->to($message['to'])
      ->subject($message['subject']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildThemedEmail(): ?array {
    return [
      '#theme' => 'postoffice_compat_email_contact_personal',
      '#copy' => $this->contactCopy,
      '#email' => $this,
      '#langcode' => $this->langcode,
      '#message' => $this->viewContactMessage(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getLangcode(): string {
    return $this->langcode;
  }

  /**
   * {@inheritdoc}
   */
  public function getRecipientName(): string {
    $recipient = $this->contactMessage->getPersonalRecipient();
    return $recipient->getDisplayName();
  }

  /**
   * {@inheritdoc}
   */
  public function getRecipientEditUrl(): string {
    $recipient = $this->contactMessage->getPersonalRecipient();
    return $recipient->toUrl('edit-form', $this->getUrlOptions())->toString();
  }

}
