<?php

namespace Drupal\postoffice_commerce\Email;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\postoffice\Email\LocalizedEmailInterface;
use Drupal\postoffice\Email\SiteEmailInterface;
use Drupal\postoffice\Email\SiteEmailTrait;
use Drupal\postoffice\Email\TemplateAttachmentsInterface;
use Drupal\postoffice\Email\TemplateAttachmentsTrait;
use Drupal\postoffice\Email\ThemedEmailInterface;
use Drupal\postoffice\Email\UrlOptionsTrait;
use Drupal\postoffice_compat\Email\CompatEmailTrait;
use Symfony\Component\Mime\Email;

/**
 * Postoffice email for messages generated by Drupal Commerce.
 */
class CommerceEmail extends Email implements LocalizedEmailInterface, SiteEmailInterface, TemplateAttachmentsInterface, ThemedEmailInterface {

  use CompatEmailTrait;
  use SiteEmailTrait;
  use TemplateAttachmentsTrait;
  use UrlOptionsTrait;

  /**
   * The commerce email id.
   */
  protected string $key;

  /**
   * Language code for this message.
   */
  protected string $langcode;

  /**
   * Constructs a new email for messages generated by Drupal Commerce.
   */
  public function __construct(
    string $key,
    string $langcode,
    ?ImmutableConfig $siteConfig = NULL,
    ?LanguageManagerInterface $languageManager = NULL,
  ) {
    parent::__construct();
    $this->key = $key;
    $this->langcode = $langcode;
    $this->siteConfig = $siteConfig;
    $this->languageManager = $languageManager;
  }

  /**
   * Constructs a new commerce email from a core message structure.
   */
  public static function createFromMessage(array $message): static {
    return (new static($message['key'], $message['langcode']))
      ->headersFromMessage($message)
      ->from($message['from'])
      ->to($message['to'])
      ->subject($message['subject'])
      ->html($message['body']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildThemedEmail(): ?array {
    return [
      '#theme' => 'postoffice_commerce_email',
      '#body' => [
        '#markup' => Markup::create($this->getHtmlBody()),
      ],
      '#email' => $this,
      '#key' => $this->key,
      '#langcode' => $this->langcode,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getLangcode(): string {
    return $this->langcode;
  }

}
