<?php

namespace Drupal\postoffice_commerce\Email;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\postoffice\Email\LocalizedEmailInterface;
use Drupal\postoffice\Email\SiteEmailInterface;
use Drupal\postoffice\Email\SiteEmailTrait;
use Drupal\postoffice\Email\TemplateAttachmentsInterface;
use Drupal\postoffice\Email\TemplateAttachmentsTrait;
use Drupal\postoffice\Email\ThemedEmailInterface;
use Drupal\postoffice\Email\UrlOptionsTrait;
use Drupal\postoffice_compat\Email\CompatEmailTrait;
use Symfony\Component\Mime\Email;

/**
 * Postoffice email for messages generated by Drupal Commerce.
 */
class OrderReceiptEmail extends Email implements LocalizedEmailInterface, OrderEmailInterface, SiteEmailInterface, TemplateAttachmentsInterface, ThemedEmailInterface {

  use CompatEmailTrait;
  use SiteEmailTrait;
  use TemplateAttachmentsTrait;
  use UrlOptionsTrait;

  /**
   * Constructs a new email for messages generated by Drupal Commerce.
   */
  public function __construct(
    protected OrderInterface $order,
    ?ImmutableConfig $siteConfig = NULL,
    ?LanguageManagerInterface $languageManager = NULL,
  ) {
    parent::__construct();
    $this->siteConfig = $siteConfig;
    $this->languageManager = $languageManager;
  }

  /**
   * Constructs a new commerce email from a core message structure.
   */
  public static function createFromMessage(array $message): static {
    return (new static($message['params']['order']))
      ->headersFromMessage($message)
      ->from($message['from'])
      ->to($message['to'])
      ->subject($message['subject'])
      ->html($message['body']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildThemedEmail(): ?array {
    return [
      '#theme' => 'postoffice_commerce_order_receipt_email',
      '#body' => [
        '#markup' => Markup::create($this->getHtmlBody()),
      ],
      '#email' => $this,
      '#order_type' => $this->getOrder()->bundle(),
      '#langcode' => $this->getLangcode(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getLangcode(): string {
    $langcode = "";

    $customer = $this->getOrder()->getCustomer();
    if ($customer->isAuthenticated()) {
      $langcode = $customer->getPreferredLangcode();
    }

    return $langcode;
  }

  /**
   * {@inheritdoc}
   */
  public function getOrder(): OrderInterface {
    return $this->order;
  }

}
