<?php

namespace Drupal\postoffice_simplenews\Email;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Drupal\postoffice\Email\LocalizedEmailInterface;
use Drupal\postoffice\Email\SiteEmailInterface;
use Drupal\postoffice\Email\SiteEmailTrait;
use Drupal\postoffice\Email\TemplateAttachmentsInterface;
use Drupal\postoffice\Email\TemplateAttachmentsTrait;
use Drupal\postoffice\Email\ThemedEmailInterface;
use Drupal\postoffice\Email\UrlOptionsTrait;
use Drupal\postoffice_compat\Email\CompatEmailTrait;
use Drupal\simplenews\Mail\MailEntity;
use Drupal\simplenews\SubscriberInterface;
use Symfony\Component\Mime\Email;

/**
 * Postoffice email for messages generated by Simplenews.
 */
class IssueEmail extends Email implements IssueEmailInterface, LocalizedEmailInterface, SiteEmailInterface, TemplateAttachmentsInterface, ThemedEmailInterface {

  use CompatEmailTrait;
  use SiteEmailTrait;
  use SubscriberRouteParamsTrait;
  use TemplateAttachmentsTrait;
  use UrlOptionsTrait;

  /**
   * Simplenews newsletter mail.
   */
  protected MailEntity $news;

  /**
   * Constructs a new email for messages generated by Simplenews.
   */
  public function __construct(
    MailEntity $news,
    ?ImmutableConfig $siteConfig = NULL,
    ?LanguageManagerInterface $languageManager = NULL,
  ) {
    parent::__construct();
    $this->news = $news;
    $this->siteConfig = $siteConfig;
    $this->languageManager = $languageManager;
  }

  /**
   * Constructs a new simplenews email from a core message structure.
   */
  public static function createFromMessage(array $message): static {
    /** @var \Drupal\simplenews\Mail\MailInterface $news */
    $news = $message['params']['simplenews_mail'];
    $email = (new static($news))
      ->headersFromMessage($message)
      ->subject($message['subject'])
      ->to($message['to'])
      ->html($message['body']);

    if (!empty($message['plain'])) {
      $email->text($message['plain']);
    }

    return $email;
  }

  /**
   * {@inheritdoc}
   */
  public function buildThemedEmail(): ?array {
    return [
      '#theme' => 'postoffice_simplenews_issue_email',
      '#body' => [
        '#markup' => Markup::create($this->getHtmlBody()),
      ],
      '#email' => $this,
      '#newsletter_id' => $this->getNewsletterId(),
      '#langcode' => $this->getLangcode(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getLangcode(): string {
    return $this->getSubscriber()->getLangcode();
  }

  /**
   * {@inheritdoc}
   */
  public function getNewsletterId(): string {
    return $this->news->getNewsletter()->id();
  }

  /**
   * {@inheritdoc}
   */
  public function getSubscriber(): SubscriberInterface {
    return $this->news->getSubscriber();
  }

  /**
   * {@inheritdoc}
   */
  public function getUnsubscribeUrl(): string {
    $result = '#';

    if (!$this->isTest()) {
      $params = $this->generateSubscriberRouteParams('remove', [
        'newsletter_id' => $this->getNewsletterId(),
      ]);
      $result = Url::fromRoute('simplenews.subscriptions_remove', $params, $this->getUrlOptions())->toString();
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function isTest(): bool {
    return $this->news->getKey() === 'test';
  }

  /**
   * {@inheritdoc}
   */
  public function isExtra(): bool {
    return $this->news->getKey() === 'extra';
  }

}
