/**
 * @file
 * Primary selection JavaScript for the Primary Entity Reference module.
 */

(function primarySelectionIIFE($, Drupal) {
  /**
   * Highlights the primary item in the inline form.
   *
   * @param {jQuery} $inlineForm
   *   The inline form container.
   * @param {string} primaryValue
   *   The value of the primary item.
   */
  function highlightPrimaryItem($inlineForm, primaryValue) {
    // Remove existing primary highlighting.
    $inlineForm.find('.primary-item').removeClass('primary-item');

    // Add highlighting to the selected primary item.
    if (primaryValue !== undefined) {
      const $primaryItem = $inlineForm.find(`[data-delta="${primaryValue}"]`);
      if ($primaryItem.length) {
        $primaryItem.addClass('primary-item');
      }
    }
  }

  /**
   * Updates the primary selection options.
   *
   * @param {jQuery} $container
   *   The primary selection container.
   */
  function updatePrimaryOptions($container) {
    const $inlineForm = $container.closest(
      '.primary-entity-reference-inline-form',
    );
    const $radios = $container.find('input[type="radio"]');
    const checkedRadio = $radios.filter(':checked')[0];
    const currentValue = checkedRadio ? checkedRadio.value : null;

    // Count the number of items in the inline form.
    const itemCount = $inlineForm.find('[data-delta]').length;

    // Update radio options if needed.
    if (itemCount > $radios.length) {
      // Add new options for additional items.
      for (let i = $radios.length; i < itemCount; i++) {
        const $newRadio = $(
          `<input type="radio" name="${$radios.attr('name')}" value="${i}">`,
        );
        const $newLabel = $(
          `<label>${Drupal.t('Item @number', { '@number': i + 1 })}</label>`,
        );
        const $newItem = $('<div class="form-item"></div>')
          .append($newRadio)
          .append($newLabel);

        $container.find('.form-radios').append($newItem);
      }
    }

    // Ensure the current selection is still valid.
    if (currentValue && currentValue >= itemCount) {
      // If the current selection is no longer valid, select the first item.
      $container
        .find('input[type="radio"][value="0"]')
        .prop('checked', true)
        .trigger('change');
    }
  }

  /**
   * Primary selection behavior.
   */
  Drupal.behaviors.primaryEntityReferenceSelection = {
    attach(context) {
      // Find all primary selection containers.
      $('.primary-selection-container', context)
        .once('primary-selection')
        .each(function primarySelectionEach() {
          const $container = $(this);
          const $radios = $container.find('input[type="radio"]');
          const $inlineForm = $container.closest(
            '.primary-entity-reference-inline-form',
          );

          // Handle radio button changes.
          $radios.on('change', function primaryRadioChange() {
            const selectedValue = this.value;
            highlightPrimaryItem($inlineForm, selectedValue);
          });

          // Initial highlighting.
          const checkedRadio = $radios.filter(':checked')[0];
          if (checkedRadio) {
            highlightPrimaryItem($inlineForm, checkedRadio.value);
          }
        });
    },
  };

  /**
   * Form validation for primary selection.
   */
  Drupal.behaviors.primaryEntityReferenceValidation = {
    attach(context) {
      $('form', context)
        .once('primary-validation')
        .each(function primaryValidationEach() {
          const $form = $(this);

          $form.on('submit', function primaryFormSubmit(e) {
            const $primaryRadios = $form.find(
              '.primary-selection-container input[type="radio"]:checked',
            );

            // Check if primary selection is required and selected.
            if ($primaryRadios.length === 0) {
              e.preventDefault();

              // Show error message.
              let $errorContainer = $form.find('.primary-selection-error');
              if ($errorContainer.length === 0) {
                $errorContainer = $(
                  '<div class="primary-selection-error messages messages--error"></div>',
                );
                $form
                  .find('.primary-selection-container')
                  .after($errorContainer);
              }

              $errorContainer.html(
                `<div class="messages__content">${Drupal.t(
                  'Please select a primary item before saving.',
                )}</div>`,
              );

              // Scroll to the error using native scrollIntoView.
              $errorContainer[0].scrollIntoView({
                behavior: 'smooth',
                block: 'center',
              });

              return false;
            }

            // Remove error message if validation passes.
            $form.find('.primary-selection-error').remove();
          });
        });
    },
  };

  /**
   * Dynamic options update for primary selection.
   */
  Drupal.behaviors.primaryEntityReferenceDynamicOptions = {
    attach(context) {
      // Watch for changes in the inline form that might affect primary options.
      $(context)
        .once('dynamic-options')
        .on('iefAdd', function primaryIefAdd(e, data) {
          // Trigger a refresh of primary options when new items are added.
          const $container = $(data.container).closest(
            '.primary-selection-container',
          );
          if ($container.length) {
            // This would typically trigger a form rebuild or AJAX refresh.
            // For now, we'll just update the available options.
            updatePrimaryOptions($container);
          }
        });
    },
  };
})(jQuery, Drupal);
