<?php

declare(strict_types=1);

namespace Drupal\primary_entity_reference\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\StringTranslation\TranslatableMarkup;

// Only define this widget if Inline Entity Form is installed.
if (!class_exists('Drupal\inline_entity_form\Plugin\Field\FieldWidget\InlineEntityFormComplex')) {
  return;
}

use Drupal\inline_entity_form\Plugin\Field\FieldWidget\InlineEntityFormComplex;

/**
 * Plugin implementation of the 'primary_entity_reference_inline_form' widget.
 *
 * This widget requires the Inline Entity Form module to be installed.
 */
#[FieldWidget(
  id: 'primary_entity_reference_inline_form',
  label: new TranslatableMarkup('Inline entity form - Primary'),
  description: new TranslatableMarkup('An inline entity form widget with primary selection.'),
  field_types: ['primary_entity_reference'],
  multiple_values: TRUE,
)]
class PrimaryEntityReferenceInlineFormWidget extends InlineEntityFormComplex {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    $defaults = parent::defaultSettings();
    $defaults += [
      'primary_label' => 'Primary',
    ];

    return $defaults;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = parent::settingsForm($form, $form_state);

    $element['primary_selection_settings'] = [
      '#type'          => 'details',
      '#title'         => $this->t('Primary selection settings'),
      '#open'          => TRUE,
      '#weight'        => 10,
    ];

    $element['primary_selection_settings']['primary_label'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Primary label'),
      '#description'   => $this->t('The label to display for the primary selection.'),
      '#default_value' => $this->getSetting('primary_label'),
      '#required'      => TRUE,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    $summary[] = $this->t('Primary selection is required.');

    $primary_label = $this->getSetting('primary_label');
    $summary[] = $this->t('Primary label: @label', ['@label' => $primary_label]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    // Get the parent form element.
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Add the primary selection library.
    $element['#attached']['library'][] = 'primary_entity_reference/primary_selection';

    // Add a wrapper class for styling.
    $class = 'primary-entity-reference-inline-form';
    $element['#attributes']['class'][] = $class;

    // Get the IEF ID from the element (set by parent).
    $ief_id = $element['#ief_id'] ?? NULL;

    if (!$ief_id) {
      return $element;
    }

    // Get the IEF widget state to check for entities.
    $widget_state = $form_state->get(['inline_entity_form', $ief_id]);
    $entities = $widget_state['entities'] ?? [];

    // Add primary column to table fields.
    $primary_label = $this->getSetting('primary_label');
    if (!isset($element['entities']['#table_fields']['primary'])) {
      $element['entities']['#table_fields'] = [
        'primary' => [
          'type' => 'callback',
          'label' => $primary_label,
          'weight' => -100,
          'callback' => [get_class($this), 'renderPrimaryField'],
        ],
      ] + ($element['entities']['#table_fields'] ?? []);
    }

    // Determine the current primary selection.
    $current_primary = NULL;
    foreach ($items as $item_delta => $item) {
      if ($item->get('primary')->getValue()) {
        $current_primary = $item_delta;
        break;
      }
    }

    // If no primary is set and we have entities, default to the first one.
    if ($current_primary === NULL && !empty($entities)) {
      $current_primary = 0;
    }

    $field_name = $this->fieldDefinition->getName();

    // Add primary radio to each entity row.
    foreach ($entities as $key => $value) {
      if (!isset($element['entities'][$key])) {
        continue;
      }

      // Add primary radio button for this row.
      $element['entities'][$key]['primary'] = [
        '#type' => 'radio',
        '#title' => $this->t('Primary'),
        '#title_display' => 'invisible',
        '#return_value' => $key,
        '#default_value' => ($current_primary == $key) ? $key : NULL,
        '#parents' => [$field_name, 'primary'],
        '#required' => FALSE,
        '#attributes' => [
          'class' => ['primary-radio'],
        ],
      ];
    }

    // Add validation.
    if (!empty($entities)) {
      $element['#element_validate'][] = [get_class($this), 'validatePrimarySelection'];
    }

    return $element;
  }

  /**
   * Renders the primary radio button for a table cell.
   *
   * This callback is used by the IEF table rendering to display custom columns.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being rendered in this row.
   * @param array $variables
   *   The template variables including the form array.
   *
   * @return array
   *   A render array for the primary radio button.
   */
  public static function renderPrimaryField($entity, array $variables) {
    $form = $variables['form'];

    // Find the delta (key) for this entity in the form.
    foreach ($form as $key => $item) {
      if (is_array($item) && isset($item['#entity']) && $item['#entity'] === $entity) {
        // Return the radio button if it exists.
        if (isset($form[$key]['primary'])) {
          return $form[$key]['primary'];
        }
        break;
      }
    }

    // Fallback if radio not found.
    return ['#markup' => ''];
  }

  /**
   * Builds options for the primary selection radios.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The field items.
   * @param array $entities
   *   The entities array from the widget state.
   *
   * @return array
   *   An array of options for the radios.
   */
  public function buildPrimaryOptions(FieldItemListInterface $items, array $entities = []) {
    $options = [];

    // If we have entities in the widget state, use those.
    if (!empty($entities)) {
      foreach ($entities as $delta => $value) {
        $entity = $value['entity'];
        $label = $entity->label() ?: $this->t('Item @number', ['@number' => $delta + 1]);
        $options[$delta] = $label;
      }
    }
    else {
      // Fallback: build options from field values.
      $values = $items->getValue();
      if (empty($values)) {
        $options[0] = $this->t('First item');
        return $options;
      }

      $target_type = $this->getTargetEntityType();
      $storage = $this->entityTypeManager->getStorage($target_type);

      foreach ($values as $delta => $value) {
        if (isset($value['target_id'])) {
          $entity = $storage->load($value['target_id']);
          if ($entity) {
            $options[$delta] = $entity->label();
          }
          else {
            $options[$delta] = $this->t('Item @number', ['@number' => $delta + 1]);
          }
        }
        else {
          $options[$delta] = $this->t('Item @number', ['@number' => $delta + 1]);
        }
      }
    }

    return $options;
  }

  /**
   * Validates the primary selection.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $form
   *   The complete form.
   */
  public static function validatePrimarySelection(array $element, FormStateInterface $form_state, array $form) {
    // Get the widget value.
    $value = $form_state->getValue($element['#parents']);

    // Check if primary is selected.
    $primary = $value['primary'] ?? NULL;

    if ($primary === NULL || $primary === '') {
      $form_state->setError($element, t('Please select a primary item.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // Get the field name and retrieve the primary selection.
    $field_name = $this->fieldDefinition->getName();
    $field_parents = $form['#parents'] ?? [];
    $parents = array_merge($field_parents, [$field_name, 'primary']);

    $primary_delta = $form_state->getValue($parents);

    // Let the parent massage the values first.
    $values = parent::massageFormValues($values, $form, $form_state);

    // Reset all primary flags to 0.
    foreach ($values as $delta => $value) {
      $values[$delta]['primary'] = 0;
    }

    // Set the selected primary.
    if ($primary_delta !== NULL && isset($values[$primary_delta])) {
      $values[$primary_delta]['primary'] = 1;
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function extractFormValues(FieldItemListInterface $items, array $form, FormStateInterface $form_state) {
    parent::extractFormValues($items, $form, $form_state);

    // Get the field name and retrieve the primary selection.
    $field_name = $this->fieldDefinition->getName();
    $field_parents = $form['#parents'] ?? [];
    $parents = array_merge($field_parents, [$field_name, 'primary']);

    $primary_delta = $form_state->getValue($parents);

    // Update the primary flag on the field items.
    if ($primary_delta !== NULL) {
      foreach ($items as $delta => $item) {
        $item->set('primary', $delta == $primary_delta ? 1 : 0);
      }
    }
  }

  /**
   * Gets the target entity type ID.
   *
   * @return string
   *   The target entity type ID.
   */
  public function getTargetEntityType(): string {
    return $this->fieldDefinition->getSetting('target_type');
  }

  /**
   * Gets the target entity type label.
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The target entity type label.
   */
  public function getTargetEntityTypeLabel() {
    $entity_type = $this->entityTypeManager->getDefinition($this->getTargetEntityType());
    return $entity_type->getLabel();
  }

}
