<?php

declare(strict_types=1);

namespace Drupal\primary_entity_reference\Hook;

use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\field\FieldStorageConfigInterface;
use Drupal\views\FieldViewsDataProvider;

/**
 * Hook implementations for primary_entity_reference.
 */
class PrimaryEntityReferenceViewsHooks {

  use StringTranslationTrait;

  /**
   * Constructs a new PrimaryEntityReferenceViewsHooks instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\views\FieldViewsDataProvider|null $fieldViewsDataProvider
   *   The field views data provider service (available in Drupal 11.2+).
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityFieldManagerInterface $entityFieldManager,
    protected readonly ?FieldViewsDataProvider $fieldViewsDataProvider = NULL,
  ) {}

  /**
   * Implements hook_field_views_data().
   *
   * Views integration for primary entity reference fields. Adds standard
   * relationships, primary-only relationships, and argument handling.
   *
   * @param \Drupal\field\FieldStorageConfigInterface $field_storage
   *   The field storage configuration.
   *
   * @return array
   *   The views data for the field.
   */
  #[Hook('field_views_data')]
  public function fieldViewsData(FieldStorageConfigInterface $field_storage): array {
    $data = DeprecationHelper::backwardsCompatibleCall(
      currentVersion: \Drupal::VERSION,
      deprecatedVersion: '11.2',
      currentCallable: fn() => $this->fieldViewsDataProvider?->defaultFieldImplementation($field_storage),
      deprecatedCallable: fn() => views_field_default_views_data($field_storage),
    );

    // The code below only deals with the primary_entity_reference field type.
    if ($field_storage->getType() !== 'primary_entity_reference') {
      return $data;
    }

    $entity_type_id        = $field_storage->getTargetEntityTypeId();
    $target_entity_type_id = $field_storage->getSetting('target_type');
    $target_entity_type    = $this->entityTypeManager->getDefinition($target_entity_type_id);
    $entity_type           = $this->entityTypeManager->getDefinition($entity_type_id);
    $target_base_table     = $target_entity_type->getDataTable() ?: $target_entity_type->getBaseTable();
    $field_name            = $field_storage->getName();

    /** @var \Drupal\Core\Entity\Sql\DefaultTableMapping $table_mapping */
    $table_mapping = $this->entityTypeManager->getStorage($entity_type_id)->getTableMapping();

    foreach ($data as $table_name => $table_data) {
      if (!($target_entity_type instanceof ContentEntityTypeInterface)) {
        continue;
      }

      // Provide a standard relationship for all entity references.
      $args = [
        '@label'      => $target_entity_type->getLabel(),
        '@field_name' => $field_name,
      ];
      $data[$table_name][$field_name]['relationship'] = [
        'title'              => $this->t('@label referenced from @field_name', $args),
        'label'              => $this->t('@field_name: @label', $args),
        'group'              => $entity_type->getLabel(),
        'help'               => $this->t('Appears in: @bundles.', [
          '@bundles' => implode(', ', $field_storage->getBundles()),
        ]),
        'id'                 => 'standard',
        'base'               => $target_base_table,
        'entity type'        => $target_entity_type_id,
        'base field'         => $target_entity_type->getKey('id'),
        'relationship field' => $field_name . '_target_id',
      ];

      // Provide a primary-only relationship with join condition.
      // Use a different key name to avoid conflict with the primary flag field.
      $primary_relationship_field_name = $field_name . '__primary_target';
      $data[$table_name][$primary_relationship_field_name]['relationship'] = [
        'title'              => $this->t('Primary @label from @field_name', $args),
        'label'              => $this->t('@field_name: Primary @label', $args),
        'group'              => $entity_type->getLabel(),
        'help'               => $this->t('Appears in: @bundles. This relationship will only include the primary reference.', [
          '@bundles' => implode(', ', $field_storage->getBundles()),
        ]),
        'id'                 => 'standard',
        'base'               => $target_base_table,
        'entity type'        => $target_entity_type_id,
        'base field'         => $target_entity_type->getKey('id'),
        'relationship field' => $field_name . '_target_id',
        'relationship table' => $table_name,
        'extra'              => [
          [
            'table'   => $table_name,
            'field'   => $field_name . '_primary',
            'value'   => 1,
            'numeric' => TRUE,
          ],
        ],
      ];

      // Provide a reverse relationship for the entity type that is referenced
      // by the field.
      $args['@entity'] = $entity_type->getLabel();
      $args['@label'] = $target_entity_type->getSingularLabel();
      $pseudo_field_name = 'reverse__' . $entity_type_id . '__' . $field_name;
      $data[$target_base_table][$pseudo_field_name]['relationship'] = [
        'title'      => $this->t('@entity using @field_name', $args),
        'label'      => $this->t('@field_name', ['@field_name' => $field_name]),
        'group'      => $target_entity_type->getLabel(),
        'help'       => $this->t('Relate each @entity with a @field_name set to the @label.', $args),
        'id'         => 'entity_reverse',
        'base'       => $entity_type->getDataTable() ?: $entity_type->getBaseTable(),
        'entity_type' => $entity_type_id,
        'base field' => $entity_type->getKey('id'),
        'field_name' => $field_name,
        'field table' => $table_mapping->getDedicatedDataTableName($field_storage),
        'field field' => $field_name . '_target_id',
        'join_extra' => [
          [
            'field'   => 'deleted',
            'value'   => 0,
            'numeric' => TRUE,
          ],
        ],
      ];

      // Add reverse relationship for primary references only.
      $primary_reverse_pseudo_field_name = 'reverse_primary__' . $entity_type_id . '__' . $field_name;
      $data[$target_base_table][$primary_reverse_pseudo_field_name]['relationship'] = [
        'title'       => $this->t('Primary @entity using @field_name', $args),
        'label'       => $this->t('@field_name (Primary only)', ['@field_name' => $field_name]),
        'group'       => $target_entity_type->getLabel(),
        'help'        => $this->t('Relate each @entity with a @field_name set to the @label where the reference is marked as primary.', $args),
        'id'          => 'entity_reverse',
        'base'        => $entity_type->getDataTable() ?: $entity_type->getBaseTable(),
        'entity_type' => $entity_type_id,
        'base field'  => $entity_type->getKey('id'),
        'field_name'  => $field_name,
        'field table' => $table_mapping->getDedicatedDataTableName($field_storage),
        'field field' => $field_name . '_target_id',
        'join_extra'  => [
          [
            'field'   => 'deleted',
            'value'   => 0,
            'numeric' => TRUE,
          ],
          [
            'field'   => $field_name . '_primary',
            'value'   => 1,
            'numeric' => TRUE,
          ],
        ],
      ];

      // Add the primary flag field to sort and filter.
      $data[$table_name][$field_name . '_primary'] = [
        'title'  => $this->t('@field_name: Primary flag', ['@field_name' => $field_name]),
        'help'   => $this->t('Whether this reference is marked as primary.'),
        'group'  => $entity_type->getLabel(),
        'field'  => [
          'id'           => 'boolean',
          'label'        => $this->t('@field_name: Primary', ['@field_name' => $field_name]),
          'type'         => 'yes-no',
          'field_name'   => $field_name . '_primary',
        ],
        'sort'   => [
          'id'         => 'standard',
          'field_name' => $field_name . '_primary',
        ],
        'filter' => [
          'id'         => 'boolean',
          'field_name' => $field_name . '_primary',
          'label'      => $this->t('Primary'),
        ],
      ];

      // Provide an argument plugin that has a meaningful titleQuery()
      // implementation getting the entity label.
      $data[$table_name][$field_name . '_target_id']['argument']['id']                    = 'entity_target_id';
      $data[$table_name][$field_name . '_target_id']['argument']['target_entity_type_id'] = $target_entity_type_id;
    }

    return $data;
  }

  /**
   * Implements hook_field_views_data_views_data_alter().
   *
   * Views integration to provide reverse relationships on primary entity
   * reference fields.
   *
   * @param array $data
   *   The views data array to alter.
   * @param \Drupal\field\FieldStorageConfigInterface $field_storage
   *   The field storage configuration.
   */
  #[Hook('field_views_data_views_data_alter')]
  public function fieldViewsDataViewsDataAlter(array &$data, FieldStorageConfigInterface $field_storage): void {
    // The reverse relationships are already handled in fieldViewsData() method.
    // This hook is left empty but implemented for completeness and future
    // extensibility.
  }

}
