<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Kernel\Views;

use Drupal\primary_entity_reference\Hook\PrimaryEntityReferenceViewsHooks;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\views\Views;

/**
 * Tests that the Views hooks are properly discovered.
 *
 * @group primary_entity_reference
 */
class HookDiscoveryTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'entity_test',
    'primary_entity_reference',
    'views',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('entity_test_mul');
    $this->installConfig(['field', 'system']);
  }

  /**
   * Tests that the hook service is registered.
   */
  public function testHookServiceRegistered(): void {
    $hook_service = \Drupal::service(PrimaryEntityReferenceViewsHooks::class);
    $this->assertNotNull($hook_service, 'Hook service should be registered.');
  }

  /**
   * Tests that the module handler can find the hook.
   */
  public function testModuleHandlerFindsHook(): void {
    $module_handler = \Drupal::moduleHandler();

    // Test if module is enabled.
    $this->assertTrue($module_handler->moduleExists('primary_entity_reference'), 'Module should be enabled.');

    // Test if the hook implementation is found.
    $has_impl = $module_handler->hasImplementations('field_views_data', 'primary_entity_reference');
    $this->assertTrue($has_impl, 'Module handler should find field_views_data implementation for primary_entity_reference module.');

    // With Drupal 10's hook system using attributes, we can't easily
    // enumerate all implementations. Instead, verify the hook service is
    // callable and registered properly.
    $hook_service = \Drupal::service(PrimaryEntityReferenceViewsHooks::class);
    $this->assertTrue(method_exists($hook_service, 'fieldViewsData'), 'Hook service should have fieldViewsData method.');
  }

  /**
   * Tests that the hook is actually invoked.
   */
  public function testHookInvocation(): void {
    // Create a field storage.
    $field_storage = FieldStorageConfig::create([
      'field_name'  => 'field_test_ref',
      'entity_type' => 'entity_test',
      'type'        => 'primary_entity_reference',
      'cardinality' => FieldStorageConfig::CARDINALITY_UNLIMITED,
      'settings'    => [
        'target_type' => 'entity_test_mul',
      ],
    ]);
    $field_storage->save();

    FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle'        => 'entity_test',
      'label'         => 'Test Reference',
    ])->save();

    // Check the type provider.
    $type_provider = $field_storage->getTypeProvider();
    $this->assertEquals('primary_entity_reference', $type_provider, 'Type provider should be primary_entity_reference.');

    // Try to invoke the hook directly.
    $module_handler = \Drupal::moduleHandler();
    $result = $module_handler->invoke('primary_entity_reference', 'field_views_data', [$field_storage]);

    $this->assertIsArray($result, 'Hook should return an array.');
    $this->assertNotEmpty($result, 'Hook result should not be empty.');

    // Check if the field table key exists.
    $this->assertArrayHasKey('entity_test__field_test_ref', $result, 'Result should contain the field table.');
  }

  /**
   * Tests that Views data is generated for the field.
   */
  public function testViewsDataGeneration(): void {
    // Create a field storage.
    $field_storage = FieldStorageConfig::create([
      'field_name'  => 'field_test_ref',
      'entity_type' => 'entity_test',
      'type'        => 'primary_entity_reference',
      'cardinality' => FieldStorageConfig::CARDINALITY_UNLIMITED,
      'settings'    => [
        'target_type' => 'entity_test_mul',
      ],
    ]);
    $field_storage->save();

    FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle'        => 'entity_test',
      'label'         => 'Test Reference',
    ])->save();

    // Clear and regenerate views data.
    if (\Drupal::hasService('views.views_data')) {
      \Drupal::service('views.views_data')->clear();
    }

    $views_data = Views::viewsData()->get('entity_test__field_test_ref');

    // Debug output.
    if (empty($views_data)) {
      $all_tables = array_keys(Views::viewsData()->getAll());
      $this->fail('Views data is empty. Available tables: ' . implode(', ', array_slice($all_tables, 0, 10)));
    }

    $this->assertNotEmpty($views_data, 'Views data should be generated for the field.');

    // Check for field_test_ref key.
    if (!isset($views_data['field_test_ref'])) {
      $available_keys = array_keys($views_data);
      $this->fail('field_test_ref not found. Available keys: ' . implode(', ', array_slice($available_keys, 0, 20)));
    }

    $this->assertArrayHasKey('field_test_ref', $views_data, 'Views data should include the field.');
  }

}
