<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Unit\Plugin\Field\FieldWidget;

use Drupal\Tests\UnitTestCase;
use Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceInlineFormWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface;
use Drupal\inline_entity_form\InlineFormInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Unit tests for the PrimaryEntityReferenceInlineFormWidget.
 *
 * @group primary_entity_reference
 * @coversDefaultClass \Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceInlineFormWidget
 */
class PrimaryEntityReferenceInlineFormWidgetTest extends UnitTestCase {

  /**
   * The widget under test.
   *
   * @var \Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceInlineFormWidget
   */
  protected $widget;

  /**
   * The field definition.
   *
   * @var \Drupal\Core\Field\FieldDefinitionInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fieldDefinition;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityStorage;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeBundleInfo;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityDisplayRepository;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The selection plugin manager.
   *
   * @var \Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $selectionManager;

  /**
   * The inline form handler.
   *
   * @var \Drupal\inline_entity_form\InlineFormInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $inlineFormHandler;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $container->set('entity_type.manager', $this->entityTypeManager);

    $this->entityStorage = $this->createMock(EntityStorageInterface::class);

    $this->entityTypeBundleInfo = $this->createMock(EntityTypeBundleInfoInterface::class);
    $this->entityDisplayRepository = $this->createMock(EntityDisplayRepositoryInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->selectionManager = $this->createMock(SelectionPluginManagerInterface::class);

    // Mock the inline form handler.
    $this->inlineFormHandler = $this->createMock(InlineFormInterface::class);
    $this->inlineFormHandler->expects($this->any())
      ->method('getEntityTypeLabels')
      ->willReturn([
        'singular' => 'item',
        'plural' => 'items',
      ]);

    $field_storage_definition = $this->createMock(FieldStorageDefinitionInterface::class);
    $field_storage_definition->expects($this->any())
      ->method('getSetting')
      ->willReturn('node');

    $this->fieldDefinition = $this->createMock(FieldDefinitionInterface::class);
    $this->fieldDefinition->expects($this->any())
      ->method('getName')
      ->willReturn('test_field');

    $this->fieldDefinition->expects($this->any())
      ->method('getSetting')
      ->with('target_type')
      ->willReturn('node');

    $this->fieldDefinition->expects($this->any())
      ->method('getFieldStorageDefinition')
      ->willReturn($field_storage_definition);

    $this->fieldDefinition->expects($this->any())
      ->method('getTargetEntityTypeId')
      ->willReturn('node');

    $this->fieldDefinition->expects($this->any())
      ->method('getTargetBundle')
      ->willReturn('article');

    // Mock entity type definition.
    $entityType = $this->createMock(EntityTypeInterface::class);
    $entityType->expects($this->any())
      ->method('getSingularLabel')
      ->willReturn('content item');
    $entityType->expects($this->any())
      ->method('getLabel')
      ->willReturn('Content');

    // Configure entity type manager.
    $this->entityTypeManager->expects($this->any())
      ->method('getHandler')
      ->with('node', 'inline_form')
      ->willReturn($this->inlineFormHandler);

    $this->entityTypeManager->expects($this->any())
      ->method('getDefinition')
      ->with('node')
      ->willReturn($entityType);

    \Drupal::setContainer($container);

    $this->widget = new PrimaryEntityReferenceInlineFormWidget(
      'test_widget',
      [],
      $this->fieldDefinition,
      [],
      [],
      $this->entityTypeBundleInfo,
      $this->entityTypeManager,
      $this->entityDisplayRepository,
      $this->moduleHandler,
      $this->selectionManager
    );
  }

  /**
   * Tests the default settings.
   */
  public function testDefaultSettings(): void {
    $settings = PrimaryEntityReferenceInlineFormWidget::defaultSettings();

    $this->assertArrayHasKey('primary_label', $settings);
    $this->assertEquals('Primary', $settings['primary_label']);
  }

  /**
   * Tests the settings form.
   */
  public function testSettingsForm(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $element = $this->widget->settingsForm($form, $form_state);

    $this->assertArrayHasKey('primary_selection_settings', $element);
    $this->assertArrayHasKey('primary_label', $element['primary_selection_settings']);
  }

  /**
   * Tests the settings summary.
   */
  public function testSettingsSummary(): void {
    $summary = $this->widget->settingsSummary();

    $this->assertIsArray($summary);
    $this->assertNotEmpty($summary);

    // Check that the summary contains primary selection information.
    $hasPrimaryInfo = FALSE;
    foreach ($summary as $item) {
      // Convert TranslatableMarkup to string for comparison.
      $itemString = (string) $item;
      if (strpos($itemString, 'Primary') !== FALSE) {
        $hasPrimaryInfo = TRUE;
        break;
      }
    }
    $this->assertTrue($hasPrimaryInfo, 'Settings summary should contain primary selection information.');
  }

  /**
   * Tests the target entity type getter.
   */
  public function testGetTargetEntityType(): void {
    $targetType = $this->widget->getTargetEntityType();
    $this->assertEquals('node', $targetType);
  }

  /**
   * Tests the target entity type label getter.
   */
  public function testGetTargetEntityTypeLabel(): void {
    $label = $this->widget->getTargetEntityTypeLabel();
    $this->assertEquals('Content', $label);
  }

  /**
   * Tests building primary options.
   */
  public function testBuildPrimaryOptions(): void {
    // Mock field items with some test data.
    $items = $this->createMock('Drupal\Core\Field\FieldItemListInterface');

    // Mock entity storage and entities.
    $entity1 = $this->createMock(EntityInterface::class);
    $entity1->expects($this->once())
      ->method('label')
      ->willReturn('Test Node 1');

    $entity2 = $this->createMock(EntityInterface::class);
    $entity2->expects($this->once())
      ->method('label')
      ->willReturn('Test Node 2');

    // Build entities array like the IEF widget state would have.
    $entities = [
      0 => ['entity' => $entity1],
      1 => ['entity' => $entity2],
    ];

    $options = $this->widget->buildPrimaryOptions($items, $entities);

    $this->assertArrayHasKey(0, $options);
    $this->assertArrayHasKey(1, $options);
    $this->assertEquals('Test Node 1', $options[0]);
    $this->assertEquals('Test Node 2', $options[1]);
  }

  /**
   * Tests building primary options with missing entities.
   */
  public function testBuildPrimaryOptionsWithMissingEntities(): void {
    $items = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $items->expects($this->once())
      ->method('getValue')
      ->willReturn([
    // Non-existent entity.
        ['target_id' => 999],
      ]);

    $this->entityTypeManager->expects($this->once())
      ->method('getStorage')
      ->with('node')
      ->willReturn($this->entityStorage);

    $this->entityStorage->expects($this->once())
      ->method('load')
      ->with(999)
      ->willReturn(NULL);

    // Test without entities array (fallback mode).
    $options = $this->widget->buildPrimaryOptions($items, []);

    $this->assertArrayHasKey(0, $options);
    $this->assertEquals('Item 1', $options[0]);
  }

  /**
   * Tests building primary options with empty values.
   */
  public function testBuildPrimaryOptionsWithEmptyValues(): void {
    $items = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $items->expects($this->once())
      ->method('getValue')
      ->willReturn([]);

    // Test without entities array (fallback mode).
    $options = $this->widget->buildPrimaryOptions($items, []);

    $this->assertArrayHasKey(0, $options);
    $this->assertEquals('First item', $options[0]);
  }

  /**
   * Tests massaging form values.
   */
  public function testMassageFormValues(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with(['test_field', 'primary'])
      ->willReturn(1);

    $form = [
      '#parents' => [],
    ];

    $values = [
      ['entity' => $this->createMock(EntityInterface::class), 'primary' => 0],
      ['entity' => $this->createMock(EntityInterface::class), 'primary' => 0],
      ['entity' => $this->createMock(EntityInterface::class), 'primary' => 0],
    ];

    $result = $this->widget->massageFormValues($values, $form, $form_state);

    $this->assertEquals(0, $result[0]['primary']);
    // This should be primary.
    $this->assertEquals(1, $result[1]['primary']);
    $this->assertEquals(0, $result[2]['primary']);
  }

}
