<?php

declare(strict_types=1);

namespace Drupal\primary_entity_reference\Hook;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Token;
use Drupal\primary_entity_reference\Plugin\Field\PrimaryEntityReferenceFieldItemList;

/**
 * Token hook implementations for primary_entity_reference.
 */
class PrimaryEntityReferenceTokenHooks {

  use StringTranslationTrait;

  /**
   * Constructs a new PrimaryEntityReferenceTokenHooks instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityFieldManagerInterface $entityFieldManager,
    protected readonly Token $token,
  ) {}

  /**
   * Implements hook_token_info_alter().
   *
   * Adds token info for primary entity reference fields:
   * - [entity:field_name:primary] - The primary referenced entity.
   * - [entity:field_name:primary:*] - Chained tokens for the primary entity.
   * - [entity:field_name:N:is_primary] - Whether delta N is the primary.
   *
   * @param array $info
   *   The token info array to alter.
   */
  #[Hook('token_info_alter')]
  public function tokenInfoAlter(array &$info): void {
    // Get all primary entity reference fields across all entity types.
    foreach ($this->getPrimaryEntityReferenceFields() as $entity_type_id => $fields) {
      // Skip if no tokens for this entity type.
      if (!isset($info['tokens'][$entity_type_id])) {
        continue;
      }

      foreach ($fields as $field_name => $field_definition) {
        $target_type = $field_definition->getSetting('target_type');
        $target_entity_type = $this->entityTypeManager->getDefinition($target_type, FALSE);

        if (!$target_entity_type) {
          continue;
        }

        // Add the 'primary' pseudo-delta token.
        $info['tokens'][$entity_type_id][$field_name . ':primary'] = [
          'name' => $this->t('@field_name: Primary', ['@field_name' => $field_definition->getLabel()]),
          'description' => $this->t('The primary referenced @type.', [
            '@type' => $target_entity_type->getSingularLabel(),
          ]),
          'type' => $target_type,
        ];

        // Add the 'is_primary' property token.
        // This is added as a dynamic token that works with any delta.
        $info['tokens'][$entity_type_id][$field_name . ':?:is_primary'] = [
          'name' => $this->t('@field_name: Is primary', ['@field_name' => $field_definition->getLabel()]),
          'description' => $this->t('Whether the item at the specified delta is the primary reference (1 or 0).'),
          'dynamic' => TRUE,
        ];
      }
    }
  }

  /**
   * Implements hook_tokens().
   *
   * Provides token replacements for primary entity reference fields.
   *
   * @param string $type
   *   The machine-readable name of the type of token being replaced.
   * @param array $tokens
   *   An array of tokens to be replaced.
   * @param array $data
   *   An associative array of data objects to be used for replacement.
   * @param array $options
   *   An associative array of options for token replacement.
   * @param \Drupal\Core\Render\BubbleableMetadata $bubbleable_metadata
   *   The bubbleable metadata.
   *
   * @return array
   *   An associative array of replacement values.
   */
  #[Hook('tokens')]
  public function tokens(string $type, array $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata): array {
    $replacements = [];

    // Get the entity from data if available.
    $entity = $data[$type] ?? NULL;

    if (!$entity instanceof ContentEntityInterface) {
      return $replacements;
    }

    // Get primary entity reference fields for this entity type.
    $fields = $this->getPrimaryEntityReferenceFieldsForEntityType($entity->getEntityTypeId(), $entity->bundle());

    if (empty($fields)) {
      return $replacements;
    }

    foreach ($fields as $field_name => $field_definition) {
      // Handle :primary tokens.
      $this->processPrimaryTokens($entity, $field_name, $tokens, $options, $bubbleable_metadata, $replacements);

      // Handle :N:is_primary tokens.
      $this->processIsPrimaryTokens($entity, $field_name, $tokens, $replacements);
    }

    return $replacements;
  }

  /**
   * Process primary-related tokens for a field.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   * @param array $tokens
   *   The tokens to process.
   * @param array $options
   *   Token replacement options.
   * @param \Drupal\Core\Render\BubbleableMetadata $bubbleable_metadata
   *   The bubbleable metadata.
   * @param array $replacements
   *   The replacements array to add to.
   */
  protected function processPrimaryTokens(
    ContentEntityInterface $entity,
    string $field_name,
    array $tokens,
    array $options,
    BubbleableMetadata $bubbleable_metadata,
    array &$replacements,
  ): void {
    if (!$entity->hasField($field_name)) {
      return;
    }

    $field = $entity->get($field_name);
    if (!$field instanceof PrimaryEntityReferenceFieldItemList) {
      return;
    }

    // Get the primary item.
    $primary_item = $field->primary();
    if (!$primary_item) {
      return;
    }

    $target_type = $field->getFieldDefinition()->getSetting('target_type');

    // Handle field_name:primary token (returns entity label).
    if (isset($tokens[$field_name . ':primary'])) {
      $referenced_entity = $primary_item->entity;
      if ($referenced_entity) {
        $replacements[$tokens[$field_name . ':primary']] = $referenced_entity->label();
        $bubbleable_metadata->addCacheableDependency($referenced_entity);
      }
    }

    // Handle field_name:primary:target_id token.
    if (isset($tokens[$field_name . ':primary:target_id'])) {
      $replacements[$tokens[$field_name . ':primary:target_id']] = $primary_item->target_id;
    }

    // Handle field_name:primary:entity and chained tokens.
    if ($primary_entity_tokens = $this->token->findWithPrefix($tokens, $field_name . ':primary')) {
      $referenced_entity = $primary_item->entity;
      if ($referenced_entity) {
        $bubbleable_metadata->addCacheableDependency($referenced_entity);

        // Handle :entity chained tokens.
        if ($entity_tokens = $this->token->findWithPrefix($primary_entity_tokens, 'entity')) {
          $replacements += $this->token->generate(
            $target_type,
            $entity_tokens,
            [$target_type => $referenced_entity],
            $options,
            $bubbleable_metadata
          );
        }

        // Also handle direct chaining without :entity (e.g., :primary:title).
        // Remove known tokens and entity-prefixed which are handled above.
        $chained_tokens = $primary_entity_tokens;
        unset($chained_tokens['target_id']);

        // Remove all entity-prefixed tokens (they're handled above).
        foreach (array_keys($chained_tokens) as $key) {
          if ($key === 'entity' || str_starts_with($key, 'entity:')) {
            unset($chained_tokens[$key]);
          }
        }

        if (!empty($chained_tokens)) {
          $replacements += $this->token->generate(
            $target_type,
            $chained_tokens,
            [$target_type => $referenced_entity],
            $options,
            $bubbleable_metadata
          );
        }
      }
    }
  }

  /**
   * Process is_primary tokens for a field.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   * @param array $tokens
   *   The tokens to process.
   * @param array $replacements
   *   The replacements array to add to.
   */
  protected function processIsPrimaryTokens(
    ContentEntityInterface $entity,
    string $field_name,
    array $tokens,
    array &$replacements,
  ): void {
    if (!$entity->hasField($field_name)) {
      return;
    }

    $field = $entity->get($field_name);
    if (!$field instanceof PrimaryEntityReferenceFieldItemList) {
      return;
    }

    // Look for field_name:N:is_primary patterns.
    foreach ($tokens as $name => $original) {
      // Match patterns like field_name:0:is_primary, field_name:1:is_primary.
      if (preg_match('/^' . preg_quote($field_name, '/') . ':(\d+):is_primary$/', $name, $matches)) {
        $delta = (int) $matches[1];

        if (isset($field[$delta])) {
          $item = $field[$delta];
          $is_primary = (bool) $item->get('primary')->getValue();
          $replacements[$original] = $is_primary ? '1' : '0';
        }
      }
    }
  }

  /**
   * Gets all primary entity reference fields grouped by entity type.
   *
   * @return array
   *   An array of field definitions keyed by entity type ID and field name.
   */
  protected function getPrimaryEntityReferenceFields(): array {
    $fields = [];

    $field_map = $this->entityFieldManager->getFieldMapByFieldType('primary_entity_reference');

    foreach ($field_map as $entity_type_id => $entity_fields) {
      foreach ($entity_fields as $field_name => $field_info) {
        // Get the field definition from the first bundle.
        $bundle = reset($field_info['bundles']);
        $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);

        if (isset($field_definitions[$field_name])) {
          $fields[$entity_type_id][$field_name] = $field_definitions[$field_name];
        }
      }
    }

    return $fields;
  }

  /**
   * Gets primary entity reference fields for a specific entity type and bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle
   *   The bundle.
   *
   * @return array
   *   An array of field definitions keyed by field name.
   */
  protected function getPrimaryEntityReferenceFieldsForEntityType(string $entity_type_id, string $bundle): array {
    $fields = [];

    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);

    foreach ($field_definitions as $field_name => $field_definition) {
      if ($field_definition->getType() === 'primary_entity_reference') {
        $fields[$field_name] = $field_definition;
      }
    }

    return $fields;
  }

}
