<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Kernel\Views;

use Drupal\primary_entity_reference\Hook\PrimaryEntityReferenceViewsHooks;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\entity_test\Entity\EntityTestMul;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;
use Drupal\views\Entity\View;
use Drupal\views\Views;

/**
 * Tests primary entity reference Views integration.
 *
 * @group primary_entity_reference
 */
class PrimaryEntityReferenceViewsTest extends ViewsKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
    'field',
    'entity_test',
    'primary_entity_reference',
    'views',
  ];

  /**
   * The test entities.
   *
   * @var \Drupal\entity_test\Entity\EntityTest[]
   */
  protected array $entities = [];

  /**
   * The referenced test entities.
   *
   * @var \Drupal\entity_test\Entity\EntityTestMul[]
   */
  protected array $referencedEntities = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = FALSE): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('entity_test_mul');
    $this->installConfig(['field']);

    // Create a primary_entity_reference field from entity_test
    // to entity_test_mul.
    $field_storage = FieldStorageConfig::create([
      'field_name'  => 'field_test_ref',
      'entity_type' => 'entity_test',
      'type'        => 'primary_entity_reference',
      'cardinality' => FieldStorageConfig::CARDINALITY_UNLIMITED,
      'settings'    => [
        'target_type' => 'entity_test_mul',
      ],
    ]);
    $field_storage->save();

    FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle'        => 'entity_test',
      'label'         => 'Test Reference',
    ])->save();

    // Clear views data cache to ensure fresh data.
    if (\Drupal::hasService('views.views_data')) {
      \Drupal::service('views.views_data')->clear();
    }

    // Create test entities.
    $this->createTestEntities();
  }

  /**
   * Creates test entities with primary entity references.
   */
  protected function createTestEntities(): void {
    // Create referenced entities.
    for ($i = 1; $i <= 3; $i++) {
      $referenced_entity = EntityTestMul::create([
        'name' => 'Referenced ' . $i,
      ]);
      $referenced_entity->save();
      $this->referencedEntities[$i] = $referenced_entity;
    }

    // Create entity with multiple references, second one is primary.
    $entity1 = EntityTest::create([
      'name'           => 'Entity 1',
      'field_test_ref' => [
        [
          'target_id' => $this->referencedEntities[1]->id(),
          'primary'   => 0,
        ],
        [
          'target_id' => $this->referencedEntities[2]->id(),
          'primary'   => 1,
        ],
        [
          'target_id' => $this->referencedEntities[3]->id(),
          'primary'   => 0,
        ],
      ],
    ]);
    $entity1->save();
    $this->entities[1] = $entity1;

    // Create entity with single primary reference.
    $entity2 = EntityTest::create([
      'name'           => 'Entity 2',
      'field_test_ref' => [
        [
          'target_id' => $this->referencedEntities[1]->id(),
          'primary'   => 1,
        ],
      ],
    ]);
    $entity2->save();
    $this->entities[2] = $entity2;

    // Create entity with multiple references, none marked primary.
    $entity3 = EntityTest::create([
      'name'           => 'Entity 3',
      'field_test_ref' => [
        [
          'target_id' => $this->referencedEntities[3]->id(),
          'primary'   => 0,
        ],
      ],
    ]);
    $entity3->save();
    $this->entities[3] = $entity3;
  }

  /**
   * Tests the Views data structure for primary entity reference fields.
   */
  public function testViewsData(): void {
    // Verify the field storage was created with the correct type.
    $field_storage = FieldStorageConfig::load('entity_test.field_test_ref');
    $this->assertNotNull($field_storage, 'Field storage should exist.');
    $this->assertEquals('primary_entity_reference', $field_storage->getType(), 'Field should be primary_entity_reference type.');

    // Check the type provider.
    $type_provider = $field_storage->getTypeProvider();
    $this->assertEquals('primary_entity_reference', $type_provider, 'Type provider should be primary_entity_reference module.');

    // Test if the hook service is available.
    $hook_service = \Drupal::service(PrimaryEntityReferenceViewsHooks::class);
    $this->assertNotNull($hook_service, 'Hook service should be available.');

    // Test if the module handler knows about our hook.
    $module_handler = \Drupal::moduleHandler();
    $has_impl = $module_handler->hasImplementations('field_views_data', 'primary_entity_reference');
    $this->assertTrue($has_impl, 'Module handler should find field_views_data implementation.');

    Views::viewsData()->clear();

    $views_data = Views::viewsData()->get('entity_test__field_test_ref');

    // Debug: Check what tables exist.
    if (empty($views_data)) {
      $all_tables = array_keys(Views::viewsData()->getAll());
      $this->fail('Views data is empty for entity_test__field_test_ref. Available tables: ' . implode(', ', $all_tables));
    }

    // Verify that the views data exists for this field.
    $this->assertNotEmpty($views_data, 'Views data should exist for the field table.');

    // Debug: Check what keys exist.
    if (!isset($views_data['field_test_ref'])) {
      $this->fail('field_test_ref key not found. Available keys: ' . implode(', ', array_keys($views_data)));
    }

    // Test standard relationship.
    $this->assertArrayHasKey('field_test_ref', $views_data, 'Standard relationship field should exist.');
    $this->assertArrayHasKey('relationship', $views_data['field_test_ref'], 'Standard relationship should be defined.');
    $this->assertEquals('standard', $views_data['field_test_ref']['relationship']['id']);
    $this->assertEquals('entity_test_mul_property_data', $views_data['field_test_ref']['relationship']['base']);
    $this->assertEquals('field_test_ref_target_id', $views_data['field_test_ref']['relationship']['relationship field']);

    // Test primary-only relationship.
    $this->assertArrayHasKey('field_test_ref__primary_target', $views_data);
    $this->assertEquals('standard', $views_data['field_test_ref__primary_target']['relationship']['id']);
    $this->assertEquals('entity_test_mul_property_data', $views_data['field_test_ref__primary_target']['relationship']['base']);
    $this->assertArrayHasKey('extra', $views_data['field_test_ref__primary_target']['relationship']);
    $extra = $views_data['field_test_ref__primary_target']['relationship']['extra'];
    $this->assertCount(1, $extra);
    $this->assertEquals('entity_test__field_test_ref', $extra[0]['table']);
    $this->assertEquals('field_test_ref_primary', $extra[0]['field']);
    $this->assertEquals(1, $extra[0]['value']);
    $this->assertTrue($extra[0]['numeric']);

    // Test primary flag field (for filtering and sorting).
    $this->assertArrayHasKey('field_test_ref_primary', $views_data);
    $this->assertArrayHasKey('field', $views_data['field_test_ref_primary']);
    $this->assertEquals('boolean', $views_data['field_test_ref_primary']['field']['id']);
    $this->assertArrayHasKey('sort', $views_data['field_test_ref_primary']);
    $this->assertEquals('standard', $views_data['field_test_ref_primary']['sort']['id']);
    $this->assertArrayHasKey('filter', $views_data['field_test_ref_primary']);
    $this->assertEquals('boolean', $views_data['field_test_ref_primary']['filter']['id']);

    // Test reverse relationship.
    $reverse_views_data = Views::viewsData()->get('entity_test_mul_property_data');
    $this->assertArrayHasKey('reverse__entity_test__field_test_ref', $reverse_views_data);
    $this->assertEquals('entity_reverse', $reverse_views_data['reverse__entity_test__field_test_ref']['relationship']['id']);
    $this->assertEquals('entity_test', $reverse_views_data['reverse__entity_test__field_test_ref']['relationship']['base']);
    $this->assertEquals('entity_test__field_test_ref', $reverse_views_data['reverse__entity_test__field_test_ref']['relationship']['field table']);

    // Test reverse primary-only relationship.
    $this->assertArrayHasKey('reverse_primary__entity_test__field_test_ref', $reverse_views_data);
    $this->assertEquals('entity_reverse', $reverse_views_data['reverse_primary__entity_test__field_test_ref']['relationship']['id']);
    $reverse_join_extra = $reverse_views_data['reverse_primary__entity_test__field_test_ref']['relationship']['join_extra'];
    $this->assertCount(2, $reverse_join_extra);
    // Check for deleted condition.
    $this->assertEquals('deleted', $reverse_join_extra[0]['field']);
    $this->assertEquals(0, $reverse_join_extra[0]['value']);
    // Check for primary condition.
    $this->assertEquals('field_test_ref_primary', $reverse_join_extra[1]['field']);
    $this->assertEquals(1, $reverse_join_extra[1]['value']);
    $this->assertTrue($reverse_join_extra[1]['numeric']);
  }

  /**
   * Tests filtering by the primary flag.
   */
  public function testPrimaryFlagFilter(): void {
    // Create a view programmatically.
    $view = View::create([
      'base_table' => 'entity_test',
      'base_field' => 'id',
      'id'         => 'test_primary_filter',
      'label'      => 'Test Primary Filter',
      'display'    => [
        'default' => [
          'id'              => 'default',
          'display_plugin'  => 'default',
          'display_title'   => 'Default',
          'display_options' => [
            'fields' => [
              'id' => [
                'id'    => 'id',
                'table' => 'entity_test',
                'field' => 'id',
              ],
            ],
            'filters' => [
              'field_test_ref_primary' => [
                'id'       => 'field_test_ref_primary',
                'table'    => 'entity_test__field_test_ref',
                'field'    => 'field_test_ref_primary',
                'value'    => '1',
                'plugin_id' => 'boolean',
              ],
            ],
          ],
        ],
      ],
    ]);
    $view->save();

    $executable = Views::getView('test_primary_filter');
    $executable->execute();

    // Should return rows where primary = 1.
    // Entity 1 has 1 primary reference, Entity 2 has 1 primary reference,
    // Entity 3 has 0 primary references.
    // We should get 2 rows total (one for entity 1's primary ref, one for
    // entity 2's primary ref).
    $this->assertCount(2, $executable->result);
  }

  /**
   * Tests sorting by the primary flag.
   */
  public function testPrimaryFlagSort(): void {
    // Create a view programmatically.
    $view = View::create([
      'base_table' => 'entity_test',
      'base_field' => 'id',
      'id'         => 'test_primary_sort',
      'label'      => 'Test Primary Sort',
      'display'    => [
        'default' => [
          'id'              => 'default',
          'display_plugin'  => 'default',
          'display_title'   => 'Default',
          'display_options' => [
            'fields' => [
              'id' => [
                'id'    => 'id',
                'table' => 'entity_test',
                'field' => 'id',
              ],
            ],
            'sorts' => [
              'field_test_ref_primary' => [
                'id'        => 'field_test_ref_primary',
                'table'     => 'entity_test__field_test_ref',
                'field'     => 'field_test_ref_primary',
                'order'     => 'DESC',
                'plugin_id' => 'standard',
              ],
            ],
          ],
        ],
      ],
    ]);
    $view->save();

    $executable = Views::getView('test_primary_sort');
    $executable->execute();

    // The results should be sorted with primary references first.
    // We should have rows with primary = 1 before rows with primary = 0.
    $this->assertGreaterThan(0, count($executable->result));

    // Check that primary references appear before non-primary ones.
    $primary_count    = 0;
    $non_primary_seen = FALSE;

    foreach ($executable->result as $row) {
      $primary_value = $row->entity_test__field_test_ref_field_test_ref_primary ?? 0;
      if ($primary_value == 1) {
        // Once we've seen a non-primary, we shouldn't see any more primary.
        $this->assertFalse($non_primary_seen, 'Primary references should appear before non-primary ones.');
        $primary_count++;
      }
      else {
        $non_primary_seen = TRUE;
      }
    }

    // We should have at least some primary references.
    $this->assertGreaterThan(0, $primary_count);
  }

  /**
   * Tests the standard relationship (all references).
   */
  public function testStandardRelationship(): void {
    // Create a view with standard relationship.
    $view = View::create([
      'base_table' => 'entity_test',
      'base_field' => 'id',
      'id'         => 'test_standard_relationship',
      'label'      => 'Test Standard Relationship',
      'display'    => [
        'default' => [
          'id'              => 'default',
          'display_plugin'  => 'default',
          'display_title'   => 'Default',
          'display_options' => [
            'relationships' => [
              'field_test_ref' => [
                'id'        => 'field_test_ref',
                'table'     => 'entity_test__field_test_ref',
                'field'     => 'field_test_ref',
                'required'  => FALSE,
                'plugin_id' => 'standard',
              ],
            ],
            'fields' => [
              'id' => [
                'id'    => 'id',
                'table' => 'entity_test',
                'field' => 'id',
              ],
              'name' => [
                'id'           => 'name',
                'table'        => 'entity_test_mul_property_data',
                'field'        => 'name',
                'relationship' => 'field_test_ref',
              ],
            ],
          ],
        ],
      ],
    ]);
    $view->save();

    $executable = Views::getView('test_standard_relationship');
    $executable->execute();

    // Entity 1 has 3 references, entity 2 has 1, entity 3 has 1.
    // Total: 5 rows (one for each reference).
    $this->assertCount(5, $executable->result);
  }

  /**
   * Tests the primary-only relationship.
   */
  public function testPrimaryOnlyRelationship(): void {
    // Create a view with primary-only relationship.
    $view = View::create([
      'base_table' => 'entity_test',
      'base_field' => 'id',
      'id'         => 'test_primary_relationship',
      'label'      => 'Test Primary Relationship',
      'display'    => [
        'default' => [
          'id'              => 'default',
          'display_plugin'  => 'default',
          'display_title'   => 'Default',
          'display_options' => [
            'relationships' => [
              'field_test_ref__primary_target' => [
                'id'        => 'field_test_ref__primary_target',
                'table'     => 'entity_test__field_test_ref',
                'field'     => 'field_test_ref__primary_target',
                'required'  => TRUE,
                'plugin_id' => 'standard',
              ],
            ],
            'fields' => [
              'id' => [
                'id'    => 'id',
                'table' => 'entity_test',
                'field' => 'id',
              ],
              'name' => [
                'id'           => 'name',
                'table'        => 'entity_test_mul_property_data',
                'field'        => 'name',
                'relationship' => 'field_test_ref__primary_target',
              ],
            ],
          ],
        ],
      ],
    ]);
    $view->save();

    $executable = Views::getView('test_primary_relationship');
    $executable->execute();

    // Only primary references should be included.
    // Entity 1 has 1 primary reference (Referenced 2).
    // Entity 2 has 1 primary reference (Referenced 1).
    // Entity 3 has 0 primary references.
    // Total: 2 rows.
    $this->assertCount(2, $executable->result);

    // Verify the correct entities are returned.
    $entity_ids = [];
    foreach ($executable->result as $row) {
      $entity_ids[] = $row->id;
    }
    sort($entity_ids);

    $this->assertEquals([1, 2], $entity_ids, 'Only entities with primary references should be returned.');
  }

  /**
   * Tests the reverse relationship.
   */
  public function testReverseRelationship(): void {
    // Create a view from the referenced entity's perspective.
    $view = View::create([
      'base_table' => 'entity_test_mul_property_data',
      'base_field' => 'id',
      'id'         => 'test_reverse_relationship',
      'label'      => 'Test Reverse Relationship',
      'display'    => [
        'default' => [
          'id'              => 'default',
          'display_plugin'  => 'default',
          'display_title'   => 'Default',
          'display_options' => [
            'relationships' => [
              'reverse__entity_test__field_test_ref' => [
                'id'        => 'reverse__entity_test__field_test_ref',
                'table'     => 'entity_test_mul_property_data',
                'field'     => 'reverse__entity_test__field_test_ref',
                'required'  => FALSE,
                'plugin_id' => 'entity_reverse',
              ],
            ],
            'fields' => [
              'id' => [
                'id'    => 'id',
                'table' => 'entity_test_mul_property_data',
                'field' => 'id',
              ],
              'name' => [
                'id'           => 'name',
                'table'        => 'entity_test',
                'field'        => 'name',
                'relationship' => 'reverse__entity_test__field_test_ref',
              ],
            ],
          ],
        ],
      ],
    ]);
    $view->save();

    $executable = Views::getView('test_reverse_relationship');
    $executable->execute();

    // Should return rows for each reference.
    // Referenced 1: referenced by Entity 1 and Entity 2 (2 rows).
    // Referenced 2: referenced by Entity 1 (1 row).
    // Referenced 3: referenced by Entity 1 and Entity 3 (2 rows).
    // Total: 5 rows.
    $this->assertCount(5, $executable->result);
  }

  /**
   * Tests the reverse primary-only relationship.
   */
  public function testReversePrimaryOnlyRelationship(): void {
    // Create a view from the referenced entity's perspective.
    $view = View::create([
      'base_table' => 'entity_test_mul_property_data',
      'base_field' => 'id',
      'id'         => 'test_reverse_primary',
      'label'      => 'Test Reverse Primary',
      'display'    => [
        'default' => [
          'id'              => 'default',
          'display_plugin'  => 'default',
          'display_title'   => 'Default',
          'display_options' => [
            'relationships' => [
              'reverse_primary__entity_test__field_test_ref' => [
                'id'        => 'reverse_primary__entity_test__field_test_ref',
                'table'     => 'entity_test_mul_property_data',
                'field'     => 'reverse_primary__entity_test__field_test_ref',
                'required'  => TRUE,
                'plugin_id' => 'entity_reverse',
              ],
            ],
            'fields' => [
              'id' => [
                'id'    => 'id',
                'table' => 'entity_test_mul_property_data',
                'field' => 'id',
              ],
              'name' => [
                'id'           => 'name',
                'table'        => 'entity_test',
                'field'        => 'name',
                'relationship' => 'reverse_primary__entity_test__field_test_ref',
              ],
            ],
          ],
        ],
      ],
    ]);
    $view->save();

    $executable = Views::getView('test_reverse_primary');
    $executable->execute();

    // Should return only primary references.
    // Referenced 1: primary reference from Entity 2 (1 row).
    // Referenced 2: primary reference from Entity 1 (1 row).
    // Referenced 3: no primary references (0 rows).
    // Total: 2 rows.
    $this->assertCount(2, $executable->result);
  }

}
