<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Unit\Hook;

use Drupal\primary_entity_reference\Hook\PrimaryEntityReferenceFormatterHooks;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for PrimaryEntityReferenceFormatterHooks.
 *
 * @group primary_entity_reference
 * @coversDefaultClass \Drupal\primary_entity_reference\Hook\PrimaryEntityReferenceFormatterHooks
 */
class PrimaryEntityReferenceFormatterHooksTest extends UnitTestCase {

  /**
   * The hooks instance under test.
   *
   * @var \Drupal\primary_entity_reference\Hook\PrimaryEntityReferenceFormatterHooks
   */
  protected PrimaryEntityReferenceFormatterHooks $hooks;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->hooks = new PrimaryEntityReferenceFormatterHooks();
  }

  /**
   * Test that primary_entity_reference is added to entity_reference formatters.
   *
   * @covers ::fieldFormatterInfoAlter
   */
  public function testFieldFormatterInfoAlterAddsFieldType(): void {
    $info = [
      'entity_reference_label' => [
        'field_types' => ['entity_reference'],
      ],
      'entity_reference_entity_id' => [
        'field_types' => ['entity_reference'],
      ],
    ];

    $this->hooks->fieldFormatterInfoAlter($info);

    $this->assertContains('primary_entity_reference', $info['entity_reference_label']['field_types']);
    $this->assertContains('primary_entity_reference', $info['entity_reference_entity_id']['field_types']);
  }

  /**
   * Tests that formatters not supporting entity_reference are not altered.
   *
   * @covers ::fieldFormatterInfoAlter
   */
  public function testFieldFormatterInfoAlterDoesNotAlterOtherFormatters(): void {
    $info = [
      'string' => [
        'field_types' => ['string', 'string_long'],
      ],
      'number_decimal' => [
        'field_types' => ['decimal', 'float'],
      ],
    ];

    $original_info = $info;
    $this->hooks->fieldFormatterInfoAlter($info);

    $this->assertEquals($original_info, $info);
  }

  /**
   * Tests that mixed formatters are handled correctly.
   *
   * @covers ::fieldFormatterInfoAlter
   */
  public function testFieldFormatterInfoAlterMixedFormatters(): void {
    $info = [
      'entity_reference_label' => [
        'field_types' => ['entity_reference'],
      ],
      'string' => [
        'field_types' => ['string'],
      ],
      'entity_reference_entity_view' => [
        'field_types' => ['entity_reference'],
      ],
    ];

    $this->hooks->fieldFormatterInfoAlter($info);

    $this->assertContains('primary_entity_reference', $info['entity_reference_label']['field_types']);
    $this->assertContains('primary_entity_reference', $info['entity_reference_entity_view']['field_types']);
    $this->assertNotContains('primary_entity_reference', $info['string']['field_types']);
  }

  /**
   * Tests that an empty info array is handled correctly.
   *
   * @covers ::fieldFormatterInfoAlter
   */
  public function testFieldFormatterInfoAlterEmptyInfo(): void {
    $info = [];

    $this->hooks->fieldFormatterInfoAlter($info);

    $this->assertEmpty($info);
  }

  /**
   * Tests formatters with multiple field types including entity_reference.
   *
   * @covers ::fieldFormatterInfoAlter
   */
  public function testFieldFormatterInfoAlterMultipleFieldTypes(): void {
    $info = [
      'multi_type_formatter' => [
        'field_types' => ['entity_reference', 'some_other_type'],
      ],
    ];

    $this->hooks->fieldFormatterInfoAlter($info);

    $this->assertContains('primary_entity_reference', $info['multi_type_formatter']['field_types']);
    $this->assertContains('entity_reference', $info['multi_type_formatter']['field_types']);
    $this->assertContains('some_other_type', $info['multi_type_formatter']['field_types']);
  }

  /**
   * Tests that primary_entity_reference is not duplicated if already present.
   *
   * @covers ::fieldFormatterInfoAlter
   */
  public function testFieldFormatterInfoAlterDoesNotDuplicate(): void {
    $info = [
      'entity_reference_label' => [
        'field_types' => ['entity_reference', 'primary_entity_reference'],
      ],
    ];

    $this->hooks->fieldFormatterInfoAlter($info);

    // Count occurrences of primary_entity_reference.
    $count = array_count_values($info['entity_reference_label']['field_types']);
    $this->assertEquals(2, $count['primary_entity_reference']);
  }

}
