<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Unit\Hook;

use PHPUnit\Framework\MockObject\MockObject;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\Core\Utility\Token;
use Drupal\primary_entity_reference\Hook\PrimaryEntityReferenceTokenHooks;
use Drupal\primary_entity_reference\Plugin\Field\PrimaryEntityReferenceFieldItemList;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Unit tests for PrimaryEntityReferenceTokenHooks.
 *
 * @group primary_entity_reference
 * @coversDefaultClass \Drupal\primary_entity_reference\Hook\PrimaryEntityReferenceTokenHooks
 */
class PrimaryEntityReferenceTokenHooksTest extends UnitTestCase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityFieldManager;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $tokenService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $container->set('entity_type.manager', $this->entityTypeManager);

    $this->entityFieldManager = $this->createMock(EntityFieldManagerInterface::class);
    $container->set('entity_field.manager', $this->entityFieldManager);

    $this->tokenService = $this->createMock(Token::class);
    $container->set('token', $this->tokenService);

    \Drupal::setContainer($container);
  }

  /**
   * Tests constructor.
   *
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $this->assertInstanceOf(PrimaryEntityReferenceTokenHooks::class, $hooks);
  }

  /**
   * Tests tokenInfoAlter with no primary entity reference fields.
   *
   * @covers ::tokenInfoAlter
   */
  public function testTokenInfoAlterWithNoFields(): void {
    $this->entityFieldManager->expects($this->once())
      ->method('getFieldMapByFieldType')
      ->with('primary_entity_reference')
      ->willReturn([]);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $info = ['tokens' => ['node' => []]];
    $hooks->tokenInfoAlter($info);

    // Info should remain unchanged.
    $this->assertEquals(['tokens' => ['node' => []]], $info);
  }

  /**
   * Tests tokenInfoAlter adds primary tokens for fields.
   *
   * @covers ::tokenInfoAlter
   */
  public function testTokenInfoAlterAddsTokens(): void {
    // Create mock field definition.
    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->expects($this->any())
      ->method('getSetting')
      ->with('target_type')
      ->willReturn('user');
    $field_definition->expects($this->any())
      ->method('getLabel')
      ->willReturn('Primary Contact');
    $field_definition->expects($this->any())
      ->method('getType')
      ->willReturn('primary_entity_reference');

    // Create mock target entity type.
    $target_entity_type = $this->createMock(EntityTypeInterface::class);
    $target_entity_type->expects($this->any())
      ->method('getSingularLabel')
      ->willReturn('user');

    $this->entityTypeManager->expects($this->any())
      ->method('getDefinition')
      ->with('user', FALSE)
      ->willReturn($target_entity_type);

    $this->entityFieldManager->expects($this->once())
      ->method('getFieldMapByFieldType')
      ->with('primary_entity_reference')
      ->willReturn([
        'node' => [
          'field_contact' => ['bundles' => ['article']],
        ],
      ]);

    $this->entityFieldManager->expects($this->any())
      ->method('getFieldDefinitions')
      ->with('node', 'article')
      ->willReturn(['field_contact' => $field_definition]);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $info = ['tokens' => ['node' => []]];
    $hooks->tokenInfoAlter($info);

    // Check that primary token was added.
    $this->assertArrayHasKey('field_contact:primary', $info['tokens']['node']);
    $this->assertEquals('user', $info['tokens']['node']['field_contact:primary']['type']);

    // Check that is_primary token was added.
    $this->assertArrayHasKey('field_contact:?:is_primary', $info['tokens']['node']);
    $this->assertTrue($info['tokens']['node']['field_contact:?:is_primary']['dynamic']);
  }

  /**
   * Tests tokenInfoAlter skips entity types not in token info.
   *
   * @covers ::tokenInfoAlter
   */
  public function testTokenInfoAlterSkipsUnknownEntityTypes(): void {
    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->expects($this->never())
      ->method('getSetting');

    $this->entityFieldManager->expects($this->once())
      ->method('getFieldMapByFieldType')
      ->with('primary_entity_reference')
      ->willReturn([
        'custom_entity' => [
          'field_ref' => ['bundles' => ['default']],
        ],
      ]);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    // Info does not have 'custom_entity' token type.
    $info = ['tokens' => ['node' => []]];
    $hooks->tokenInfoAlter($info);

    // Should not have added any tokens for custom_entity.
    $this->assertArrayNotHasKey('custom_entity', $info['tokens']);
  }

  /**
   * Tests tokens() with non-entity data.
   *
   * @covers ::tokens
   */
  public function testTokensWithNonEntityData(): void {
    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $bubbleable_metadata = new BubbleableMetadata();
    $replacements = $hooks->tokens('node', [], ['node' => 'not_an_entity'], [], $bubbleable_metadata);

    $this->assertEquals([], $replacements);
  }

  /**
   * Tests tokens() with entity having no primary entity reference fields.
   *
   * @covers ::tokens
   */
  public function testTokensWithNoFields(): void {
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->expects($this->any())
      ->method('getEntityTypeId')
      ->willReturn('node');
    $entity->expects($this->any())
      ->method('bundle')
      ->willReturn('article');

    $this->entityFieldManager->expects($this->any())
      ->method('getFieldDefinitions')
      ->with('node', 'article')
      ->willReturn([]);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $bubbleable_metadata = new BubbleableMetadata();
    $replacements = $hooks->tokens('node', [], ['node' => $entity], [], $bubbleable_metadata);

    $this->assertEquals([], $replacements);
  }

  /**
   * Creates a mock content entity with cache methods.
   *
   * @return \PHPUnit\Framework\MockObject\MockObject
   *   The mock entity.
   */
  protected function createCacheableEntityMock(): MockObject {
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->expects($this->any())
      ->method('getCacheContexts')
      ->willReturn([]);
    $entity->expects($this->any())
      ->method('getCacheTags')
      ->willReturn([]);
    $entity->expects($this->any())
      ->method('getCacheMaxAge')
      ->willReturn(-1);
    return $entity;
  }

  /**
   * Creates a mock field item with entity and target_id properties.
   *
   * @param mixed $entity
   *   The referenced entity.
   * @param mixed $target_id
   *   The target ID.
   * @param \Drupal\Core\TypedData\TypedDataInterface $primary_typed_data
   *   The primary typed data mock.
   *
   * @return \PHPUnit\Framework\MockObject\MockObject
   *   The mock field item.
   */
  protected function createFieldItemMock($entity, $target_id, TypedDataInterface $primary_typed_data) {
    $item = $this->getMockBuilder(EntityReferenceItem::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['get', '__get'])
      ->getMock();

    $item->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($primary_typed_data);

    $item->expects($this->any())
      ->method('__get')
      ->willReturnCallback(function ($name) use ($entity, $target_id) {
        if ($name === 'entity') {
          return $entity;
        }
        if ($name === 'target_id') {
          return $target_id;
        }
        return NULL;
      });

    return $item;
  }

  /**
   * Tests tokens() for :primary token replacement.
   *
   * @covers ::tokens
   * @covers ::processPrimaryTokens
   */
  public function testTokensForPrimaryToken(): void {
    // Create mock referenced entity with cache methods.
    $referenced_entity = $this->createCacheableEntityMock();
    $referenced_entity->expects($this->any())
      ->method('label')
      ->willReturn('John Doe');

    // Create mock primary typed data.
    $primary_typed_data = $this->createMock(TypedDataInterface::class);
    $primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(1);

    // Create mock field item with entity and target_id properties.
    $item = $this->createFieldItemMock($referenced_entity, 42, $primary_typed_data);

    // Create mock field item list.
    $field = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['primary', 'getFieldDefinition'])
      ->getMock();

    $field->expects($this->any())
      ->method('primary')
      ->willReturn($item);

    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->expects($this->any())
      ->method('getSetting')
      ->with('target_type')
      ->willReturn('user');
    $field_definition->expects($this->any())
      ->method('getType')
      ->willReturn('primary_entity_reference');

    $field->expects($this->any())
      ->method('getFieldDefinition')
      ->willReturn($field_definition);

    // Create mock entity.
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->expects($this->any())
      ->method('getEntityTypeId')
      ->willReturn('node');
    $entity->expects($this->any())
      ->method('bundle')
      ->willReturn('article');
    $entity->expects($this->any())
      ->method('hasField')
      ->with('field_contact')
      ->willReturn(TRUE);
    $entity->expects($this->any())
      ->method('get')
      ->with('field_contact')
      ->willReturn($field);

    $this->entityFieldManager->expects($this->any())
      ->method('getFieldDefinitions')
      ->with('node', 'article')
      ->willReturn(['field_contact' => $field_definition]);

    $this->tokenService->expects($this->any())
      ->method('findWithPrefix')
      ->willReturn([]);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $tokens = [
      'field_contact:primary' => '[node:field_contact:primary]',
    ];

    $bubbleable_metadata = new BubbleableMetadata();
    $replacements = $hooks->tokens('node', $tokens, ['node' => $entity], [], $bubbleable_metadata);

    $this->assertArrayHasKey('[node:field_contact:primary]', $replacements);
    $this->assertEquals('John Doe', $replacements['[node:field_contact:primary]']);
  }

  /**
   * Tests tokens() for :primary:target_id token replacement.
   *
   * @covers ::tokens
   * @covers ::processPrimaryTokens
   */
  public function testTokensForPrimaryTargetIdToken(): void {
    // Create mock referenced entity with cache methods.
    $referenced_entity = $this->createCacheableEntityMock();

    // Create mock primary typed data.
    $primary_typed_data = $this->createMock(TypedDataInterface::class);
    $primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(1);

    // Create mock field item with entity and target_id properties.
    $item = $this->createFieldItemMock($referenced_entity, 42, $primary_typed_data);

    // Create mock field item list.
    $field = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['primary', 'getFieldDefinition'])
      ->getMock();

    $field->expects($this->any())
      ->method('primary')
      ->willReturn($item);

    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->expects($this->any())
      ->method('getSetting')
      ->with('target_type')
      ->willReturn('user');
    $field_definition->expects($this->any())
      ->method('getType')
      ->willReturn('primary_entity_reference');

    $field->expects($this->any())
      ->method('getFieldDefinition')
      ->willReturn($field_definition);

    // Create mock entity.
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->expects($this->any())
      ->method('getEntityTypeId')
      ->willReturn('node');
    $entity->expects($this->any())
      ->method('bundle')
      ->willReturn('article');
    $entity->expects($this->any())
      ->method('hasField')
      ->with('field_contact')
      ->willReturn(TRUE);
    $entity->expects($this->any())
      ->method('get')
      ->with('field_contact')
      ->willReturn($field);

    $this->entityFieldManager->expects($this->any())
      ->method('getFieldDefinitions')
      ->with('node', 'article')
      ->willReturn(['field_contact' => $field_definition]);

    $this->tokenService->expects($this->any())
      ->method('findWithPrefix')
      ->willReturn([]);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $tokens = [
      'field_contact:primary:target_id' => '[node:field_contact:primary:target_id]',
    ];

    $bubbleable_metadata = new BubbleableMetadata();
    $replacements = $hooks->tokens('node', $tokens, ['node' => $entity], [], $bubbleable_metadata);

    $this->assertArrayHasKey('[node:field_contact:primary:target_id]', $replacements);
    $this->assertEquals(42, $replacements['[node:field_contact:primary:target_id]']);
  }

  /**
   * Tests tokens() for :N:is_primary token replacement.
   *
   * @covers ::tokens
   * @covers ::processIsPrimaryTokens
   */
  public function testTokensForIsPrimaryToken(): void {
    // Create mock typed data for primary and non-primary.
    $primary_typed_data = $this->createMock(TypedDataInterface::class);
    $primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(1);

    $non_primary_typed_data = $this->createMock(TypedDataInterface::class);
    $non_primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(0);

    // Create mock field items.
    $item0 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item0->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);

    $item1 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item1->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($primary_typed_data);

    // Create mock field item list.
    $field = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['primary', 'getFieldDefinition', 'offsetExists', 'offsetGet'])
      ->getMock();

    $field->expects($this->any())
      ->method('primary')
      ->willReturn($item1);

    $field->expects($this->any())
      ->method('offsetExists')
      ->willReturnCallback(function ($offset) {
        return in_array($offset, [0, 1]);
      });

    $field->expects($this->any())
      ->method('offsetGet')
      ->willReturnCallback(function ($offset) use ($item0, $item1) {
        return $offset === 0 ? $item0 : $item1;
      });

    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->expects($this->any())
      ->method('getSetting')
      ->with('target_type')
      ->willReturn('user');
    $field_definition->expects($this->any())
      ->method('getType')
      ->willReturn('primary_entity_reference');

    $field->expects($this->any())
      ->method('getFieldDefinition')
      ->willReturn($field_definition);

    // Create mock entity.
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->expects($this->any())
      ->method('getEntityTypeId')
      ->willReturn('node');
    $entity->expects($this->any())
      ->method('bundle')
      ->willReturn('article');
    $entity->expects($this->any())
      ->method('hasField')
      ->with('field_contact')
      ->willReturn(TRUE);
    $entity->expects($this->any())
      ->method('get')
      ->with('field_contact')
      ->willReturn($field);

    $this->entityFieldManager->expects($this->any())
      ->method('getFieldDefinitions')
      ->with('node', 'article')
      ->willReturn(['field_contact' => $field_definition]);

    $this->tokenService->expects($this->any())
      ->method('findWithPrefix')
      ->willReturn([]);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $tokens = [
      'field_contact:0:is_primary' => '[node:field_contact:0:is_primary]',
      'field_contact:1:is_primary' => '[node:field_contact:1:is_primary]',
    ];

    $bubbleable_metadata = new BubbleableMetadata();
    $replacements = $hooks->tokens('node', $tokens, ['node' => $entity], [], $bubbleable_metadata);

    $this->assertArrayHasKey('[node:field_contact:0:is_primary]', $replacements);
    $this->assertEquals('0', $replacements['[node:field_contact:0:is_primary]']);

    $this->assertArrayHasKey('[node:field_contact:1:is_primary]', $replacements);
    $this->assertEquals('1', $replacements['[node:field_contact:1:is_primary]']);
  }

  /**
   * Tests tokens() when primary item is NULL.
   *
   * @covers ::tokens
   * @covers ::processPrimaryTokens
   */
  public function testTokensWithNoPrimaryItem(): void {
    // Create mock field item list with no primary.
    $field = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['primary', 'getFieldDefinition'])
      ->getMock();

    $field->expects($this->any())
      ->method('primary')
      ->willReturn(NULL);

    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->expects($this->any())
      ->method('getSetting')
      ->with('target_type')
      ->willReturn('user');
    $field_definition->expects($this->any())
      ->method('getType')
      ->willReturn('primary_entity_reference');

    $field->expects($this->any())
      ->method('getFieldDefinition')
      ->willReturn($field_definition);

    // Create mock entity.
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->expects($this->any())
      ->method('getEntityTypeId')
      ->willReturn('node');
    $entity->expects($this->any())
      ->method('bundle')
      ->willReturn('article');
    $entity->expects($this->any())
      ->method('hasField')
      ->with('field_contact')
      ->willReturn(TRUE);
    $entity->expects($this->any())
      ->method('get')
      ->with('field_contact')
      ->willReturn($field);

    $this->entityFieldManager->expects($this->any())
      ->method('getFieldDefinitions')
      ->with('node', 'article')
      ->willReturn(['field_contact' => $field_definition]);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $tokens = [
      'field_contact:primary' => '[node:field_contact:primary]',
    ];

    $bubbleable_metadata = new BubbleableMetadata();
    $replacements = $hooks->tokens('node', $tokens, ['node' => $entity], [], $bubbleable_metadata);

    // No replacement should be made when primary is NULL.
    $this->assertArrayNotHasKey('[node:field_contact:primary]', $replacements);
  }

  /**
   * Tests tokens() for chained entity tokens.
   *
   * @covers ::tokens
   * @covers ::processPrimaryTokens
   */
  public function testTokensForChainedEntityTokens(): void {
    // Create mock referenced entity with cache methods.
    $referenced_entity = $this->createCacheableEntityMock();

    // Create mock primary typed data.
    $primary_typed_data = $this->createMock(TypedDataInterface::class);
    $primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(1);

    // Create mock field item with entity and target_id properties.
    $item = $this->createFieldItemMock($referenced_entity, 42, $primary_typed_data);

    // Create mock field item list.
    $field = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['primary', 'getFieldDefinition'])
      ->getMock();

    $field->expects($this->any())
      ->method('primary')
      ->willReturn($item);

    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->expects($this->any())
      ->method('getSetting')
      ->with('target_type')
      ->willReturn('user');
    $field_definition->expects($this->any())
      ->method('getType')
      ->willReturn('primary_entity_reference');

    $field->expects($this->any())
      ->method('getFieldDefinition')
      ->willReturn($field_definition);

    // Create mock entity.
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->expects($this->any())
      ->method('getEntityTypeId')
      ->willReturn('node');
    $entity->expects($this->any())
      ->method('bundle')
      ->willReturn('article');
    $entity->expects($this->any())
      ->method('hasField')
      ->with('field_contact')
      ->willReturn(TRUE);
    $entity->expects($this->any())
      ->method('get')
      ->with('field_contact')
      ->willReturn($field);

    $this->entityFieldManager->expects($this->any())
      ->method('getFieldDefinitions')
      ->with('node', 'article')
      ->willReturn(['field_contact' => $field_definition]);

    // Set up token service to handle chained tokens.
    $this->tokenService->expects($this->any())
      ->method('findWithPrefix')
      ->willReturnCallback(function ($tokens, $prefix) {
        if ($prefix === 'field_contact:primary') {
          return ['entity:name' => '[user:name]'];
        }
        if ($prefix === 'entity') {
          return ['name' => '[user:name]'];
        }
        return [];
      });

    $this->tokenService->expects($this->any())
      ->method('generate')
      ->with('user', ['name' => '[user:name]'], $this->anything(), $this->anything(), $this->anything())
      ->willReturn(['[user:name]' => 'John Doe']);

    $hooks = new PrimaryEntityReferenceTokenHooks(
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->tokenService
    );

    $tokens = [
      'field_contact:primary:entity:name' => '[node:field_contact:primary:entity:name]',
    ];

    $bubbleable_metadata = new BubbleableMetadata();
    $replacements = $hooks->tokens('node', $tokens, ['node' => $entity], [], $bubbleable_metadata);

    $this->assertArrayHasKey('[user:name]', $replacements);
    $this->assertEquals('John Doe', $replacements['[user:name]']);
  }

}
