<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Unit\Plugin\Field\FieldWidget;

use Drupal\Tests\UnitTestCase;
use Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceInlineFormWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface;
use Drupal\inline_entity_form\InlineFormInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Unit tests for the PrimaryEntityReferenceInlineFormWidget.
 *
 * @group primary_entity_reference
 * @coversDefaultClass \Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceInlineFormWidget
 */
class PrimaryEntityReferenceInlineFormWidgetTest extends UnitTestCase {

  /**
   * The widget under test.
   *
   * @var \Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceInlineFormWidget
   */
  protected $widget;

  /**
   * The field definition.
   *
   * @var \Drupal\Core\Field\FieldDefinitionInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fieldDefinition;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityStorage;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeBundleInfo;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityDisplayRepository;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The selection plugin manager.
   *
   * @var \Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $selectionManager;

  /**
   * The inline form handler.
   *
   * @var \Drupal\inline_entity_form\InlineFormInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $inlineFormHandler;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $container->set('entity_type.manager', $this->entityTypeManager);

    $this->entityStorage = $this->createMock(EntityStorageInterface::class);

    $this->entityTypeBundleInfo = $this->createMock(EntityTypeBundleInfoInterface::class);
    $this->entityDisplayRepository = $this->createMock(EntityDisplayRepositoryInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->selectionManager = $this->createMock(SelectionPluginManagerInterface::class);

    // Mock the inline form handler.
    $this->inlineFormHandler = $this->createMock(InlineFormInterface::class);
    $this->inlineFormHandler->expects($this->any())
      ->method('getEntityTypeLabels')
      ->willReturn([
        'singular' => 'item',
        'plural' => 'items',
      ]);

    $field_storage_definition = $this->createMock(FieldStorageDefinitionInterface::class);
    $field_storage_definition->expects($this->any())
      ->method('getSetting')
      ->willReturn('node');

    $this->fieldDefinition = $this->createMock(FieldDefinitionInterface::class);
    $this->fieldDefinition->expects($this->any())
      ->method('getName')
      ->willReturn('test_field');

    $this->fieldDefinition->expects($this->any())
      ->method('getSetting')
      ->with('target_type')
      ->willReturn('node');

    $this->fieldDefinition->expects($this->any())
      ->method('getFieldStorageDefinition')
      ->willReturn($field_storage_definition);

    $this->fieldDefinition->expects($this->any())
      ->method('getTargetEntityTypeId')
      ->willReturn('node');

    $this->fieldDefinition->expects($this->any())
      ->method('getTargetBundle')
      ->willReturn('article');

    // Mock entity type definition.
    $entityType = $this->createMock(EntityTypeInterface::class);
    $entityType->expects($this->any())
      ->method('getSingularLabel')
      ->willReturn('content item');
    $entityType->expects($this->any())
      ->method('getLabel')
      ->willReturn('Content');

    // Configure entity type manager.
    $this->entityTypeManager->expects($this->any())
      ->method('getHandler')
      ->with('node', 'inline_form')
      ->willReturn($this->inlineFormHandler);

    $this->entityTypeManager->expects($this->any())
      ->method('getDefinition')
      ->with('node')
      ->willReturn($entityType);

    \Drupal::setContainer($container);

    $this->widget = new PrimaryEntityReferenceInlineFormWidget(
      'test_widget',
      [],
      $this->fieldDefinition,
      [],
      [],
      $this->entityTypeBundleInfo,
      $this->entityTypeManager,
      $this->entityDisplayRepository,
      $this->moduleHandler,
      $this->selectionManager
    );
  }

  /**
   * Tests the default settings.
   */
  public function testDefaultSettings(): void {
    $settings = PrimaryEntityReferenceInlineFormWidget::defaultSettings();

    $this->assertArrayHasKey('primary_label', $settings);
    $this->assertEquals('Primary', $settings['primary_label']);
  }

  /**
   * Tests the settings form.
   */
  public function testSettingsForm(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $element = $this->widget->settingsForm($form, $form_state);

    $this->assertArrayHasKey('primary_selection_settings', $element);
    $this->assertArrayHasKey('primary_label', $element['primary_selection_settings']);
  }

  /**
   * Tests the settings summary.
   */
  public function testSettingsSummary(): void {
    $summary = $this->widget->settingsSummary();

    $this->assertIsArray($summary);
    $this->assertNotEmpty($summary);

    // Check that the summary contains primary selection information.
    $hasPrimaryInfo = FALSE;
    foreach ($summary as $item) {
      // Convert TranslatableMarkup to string for comparison.
      $itemString = (string) $item;
      if (strpos($itemString, 'Primary') !== FALSE) {
        $hasPrimaryInfo = TRUE;
        break;
      }
    }
    $this->assertTrue($hasPrimaryInfo, 'Settings summary should contain primary selection information.');
  }

  /**
   * Tests the target entity type getter.
   */
  public function testGetTargetEntityType(): void {
    $targetType = $this->widget->getTargetEntityType();
    $this->assertEquals('node', $targetType);
  }

  /**
   * Tests the target entity type label getter.
   */
  public function testGetTargetEntityTypeLabel(): void {
    $label = $this->widget->getTargetEntityTypeLabel();
    $this->assertEquals('Content', $label);
  }

  /**
   * Tests building primary options.
   */
  public function testBuildPrimaryOptions(): void {
    // Mock field items with some test data.
    $items = $this->createMock('Drupal\Core\Field\FieldItemListInterface');

    // Mock entity storage and entities.
    $entity1 = $this->createMock(EntityInterface::class);
    $entity1->expects($this->once())
      ->method('label')
      ->willReturn('Test Node 1');

    $entity2 = $this->createMock(EntityInterface::class);
    $entity2->expects($this->once())
      ->method('label')
      ->willReturn('Test Node 2');

    // Build entities array like the IEF widget state would have.
    $entities = [
      0 => ['entity' => $entity1],
      1 => ['entity' => $entity2],
    ];

    $options = $this->widget->buildPrimaryOptions($items, $entities);

    $this->assertArrayHasKey(0, $options);
    $this->assertArrayHasKey(1, $options);
    $this->assertEquals('Test Node 1', $options[0]);
    $this->assertEquals('Test Node 2', $options[1]);
  }

  /**
   * Tests building primary options with missing entities.
   */
  public function testBuildPrimaryOptionsWithMissingEntities(): void {
    $items = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $items->expects($this->once())
      ->method('getValue')
      ->willReturn([
    // Non-existent entity.
        ['target_id' => 999],
      ]);

    $this->entityTypeManager->expects($this->once())
      ->method('getStorage')
      ->with('node')
      ->willReturn($this->entityStorage);

    $this->entityStorage->expects($this->once())
      ->method('load')
      ->with(999)
      ->willReturn(NULL);

    // Test without entities array (fallback mode).
    $options = $this->widget->buildPrimaryOptions($items, []);

    $this->assertArrayHasKey(0, $options);
    $this->assertEquals('Item 1', $options[0]);
  }

  /**
   * Tests building primary options with empty values.
   */
  public function testBuildPrimaryOptionsWithEmptyValues(): void {
    $items = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $items->expects($this->once())
      ->method('getValue')
      ->willReturn([]);

    // Test without entities array (fallback mode).
    $options = $this->widget->buildPrimaryOptions($items, []);

    $this->assertArrayHasKey(0, $options);
    $this->assertEquals('First item', $options[0]);
  }

  /**
   * Tests massaging form values.
   */
  public function testMassageFormValues(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with(['test_field', 'primary'])
      ->willReturn(1);

    $form = [
      '#parents' => [],
    ];

    $values = [
      ['entity' => $this->createMock(EntityInterface::class), 'primary' => 0],
      ['entity' => $this->createMock(EntityInterface::class), 'primary' => 0],
      ['entity' => $this->createMock(EntityInterface::class), 'primary' => 0],
    ];

    $result = $this->widget->massageFormValues($values, $form, $form_state);

    $this->assertEquals(0, $result[0]['primary']);
    // This should be primary.
    $this->assertEquals(1, $result[1]['primary']);
    $this->assertEquals(0, $result[2]['primary']);
  }

  /**
   * Tests building primary options with values missing target_id.
   *
   * @covers ::buildPrimaryOptions
   */
  public function testBuildPrimaryOptionsWithMissingTargetId(): void {
    $items = $this->createMock('Drupal\Core\Field\FieldItemListInterface');
    $items->expects($this->once())
      ->method('getValue')
      ->willReturn([
        // Value without target_id key.
        ['some_other_key' => 'value'],
      ]);

    $this->entityTypeManager->expects($this->once())
      ->method('getStorage')
      ->with('node')
      ->willReturn($this->entityStorage);

    // Test without entities array (fallback mode).
    $options = $this->widget->buildPrimaryOptions($items, []);

    $this->assertArrayHasKey(0, $options);
    $this->assertEquals('Item 1', $options[0]);
  }

  /**
   * Tests building primary options with entity having null label.
   *
   * @covers ::buildPrimaryOptions
   */
  public function testBuildPrimaryOptionsWithNullLabel(): void {
    // Mock field items with some test data.
    $items = $this->createMock('Drupal\Core\Field\FieldItemListInterface');

    // Mock entity with null label.
    $entity = $this->createMock(EntityInterface::class);
    $entity->expects($this->once())
      ->method('label')
      ->willReturn(NULL);

    // Build entities array like the IEF widget state would have.
    $entities = [
      0 => ['entity' => $entity],
    ];

    $options = $this->widget->buildPrimaryOptions($items, $entities);

    $this->assertArrayHasKey(0, $options);
    $this->assertEquals('Item 1', (string) $options[0]);
  }

  /**
   * Tests the renderPrimaryField static callback.
   *
   * @covers ::renderPrimaryField
   */
  public function testRenderPrimaryField(): void {
    // Create a mock entity.
    $entity = $this->createMock(EntityInterface::class);

    // Create form array structure similar to IEF.
    $form = [
      0 => [
        '#entity' => $entity,
        'primary' => [
          '#type' => 'radio',
          '#return_value' => 0,
        ],
      ],
    ];

    $variables = ['form' => $form];

    $result = PrimaryEntityReferenceInlineFormWidget::renderPrimaryField($entity, $variables);

    $this->assertEquals($form[0]['primary'], $result);
  }

  /**
   * Tests renderPrimaryField returns empty markup when entity not found.
   *
   * @covers ::renderPrimaryField
   */
  public function testRenderPrimaryFieldNotFound(): void {
    // Create a mock entity that's not in the form.
    $entity = $this->createMock(EntityInterface::class);
    $different_entity = $this->createMock(EntityInterface::class);

    // Create form array with a different entity.
    $form = [
      0 => [
        '#entity' => $different_entity,
        'primary' => [
          '#type' => 'radio',
          '#return_value' => 0,
        ],
      ],
    ];

    $variables = ['form' => $form];

    $result = PrimaryEntityReferenceInlineFormWidget::renderPrimaryField($entity, $variables);

    $this->assertArrayHasKey('#markup', $result);
    $this->assertEquals('', $result['#markup']);
  }

  /**
   * Tests renderPrimaryField when primary key is missing.
   *
   * @covers ::renderPrimaryField
   */
  public function testRenderPrimaryFieldMissingPrimaryKey(): void {
    // Create a mock entity.
    $entity = $this->createMock(EntityInterface::class);

    // Create form array without primary key.
    $form = [
      0 => [
        '#entity' => $entity,
        // No 'primary' key.
      ],
    ];

    $variables = ['form' => $form];

    $result = PrimaryEntityReferenceInlineFormWidget::renderPrimaryField($entity, $variables);

    $this->assertArrayHasKey('#markup', $result);
    $this->assertEquals('', $result['#markup']);
  }

  /**
   * Tests validatePrimarySelection with valid selection.
   *
   * @covers ::validatePrimarySelection
   */
  public function testValidatePrimarySelectionValid(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with(['test_field'])
      ->willReturn(['primary' => 1]);

    // Should not set error when primary is set.
    $form_state->expects($this->never())
      ->method('setError');

    $element = ['#parents' => ['test_field']];
    $form = [];

    PrimaryEntityReferenceInlineFormWidget::validatePrimarySelection($element, $form_state, $form);
  }

  /**
   * Tests validatePrimarySelection with null selection.
   *
   * @covers ::validatePrimarySelection
   */
  public function testValidatePrimarySelectionNull(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with(['test_field'])
      ->willReturn(['primary' => NULL]);

    // Should set error when primary is null.
    $form_state->expects($this->once())
      ->method('setError');

    $element = ['#parents' => ['test_field']];
    $form = [];

    PrimaryEntityReferenceInlineFormWidget::validatePrimarySelection($element, $form_state, $form);
  }

  /**
   * Tests validatePrimarySelection with empty string selection.
   *
   * @covers ::validatePrimarySelection
   */
  public function testValidatePrimarySelectionEmptyString(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with(['test_field'])
      ->willReturn(['primary' => '']);

    // Should set error when primary is empty string.
    $form_state->expects($this->once())
      ->method('setError');

    $element = ['#parents' => ['test_field']];
    $form = [];

    PrimaryEntityReferenceInlineFormWidget::validatePrimarySelection($element, $form_state, $form);
  }

  /**
   * Tests validatePrimarySelection with missing primary key.
   *
   * @covers ::validatePrimarySelection
   */
  public function testValidatePrimarySelectionMissingKey(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with(['test_field'])
      ->willReturn([]);

    // Should set error when primary key is missing.
    $form_state->expects($this->once())
      ->method('setError');

    $element = ['#parents' => ['test_field']];
    $form = [];

    PrimaryEntityReferenceInlineFormWidget::validatePrimarySelection($element, $form_state, $form);
  }

  /**
   * Tests validatePrimarySelection with zero as valid selection.
   *
   * @covers ::validatePrimarySelection
   */
  public function testValidatePrimarySelectionZero(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with(['test_field'])
      ->willReturn(['primary' => 0]);

    // Zero is a valid selection (first item), should not set error.
    $form_state->expects($this->never())
      ->method('setError');

    $element = ['#parents' => ['test_field']];
    $form = [];

    PrimaryEntityReferenceInlineFormWidget::validatePrimarySelection($element, $form_state, $form);
  }

  /**
   * Tests extractFormValues method exists and is callable.
   *
   * The full functionality of extractFormValues requires a Drupal kernel
   * environment to properly test iteration over field items.
   *
   * @covers ::extractFormValues
   */
  public function testExtractFormValuesMethodExists(): void {
    $this->assertTrue(
      method_exists($this->widget, 'extractFormValues'),
      'extractFormValues method should exist on the widget.'
    );
  }

}
