<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Unit\Plugin\Field\FieldWidget;

use Drupal\Core\Form\FormStateInterface;
use Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceOptionsButtonsWidget;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for the PrimaryEntityReferenceOptionsButtonsWidget.
 *
 * @group primary_entity_reference
 * @coversDefaultClass \Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceOptionsButtonsWidget
 */
class PrimaryEntityReferenceOptionsButtonsWidgetTest extends UnitTestCase {

  /**
   * Tests the class extends the expected parent.
   */
  public function testClassInheritance(): void {
    $this->assertTrue(
      is_subclass_of(
        PrimaryEntityReferenceOptionsButtonsWidget::class,
        'Drupal\Core\Field\Plugin\Field\FieldWidget\OptionsButtonsWidget'
      ),
      'PrimaryEntityReferenceOptionsButtonsWidget should extend OptionsButtonsWidget.'
    );
  }

  /**
   * Creates a widget mock with constructor disabled.
   *
   * @return \Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryEntityReferenceOptionsButtonsWidget|\PHPUnit\Framework\MockObject\MockObject
   *   The widget mock.
   */
  protected function createWidgetMock() {
    // Create mock without constructor to avoid OptionsWidgetBase issues.
    $widget = $this->getMockBuilder(PrimaryEntityReferenceOptionsButtonsWidget::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getOptions'])
      ->getMock();

    // Set required properties via reflection that parent methods need.
    $reflection = new \ReflectionClass($widget);

    // Set the column property used by parent massageFormValues.
    $columnProperty = $reflection->getProperty('column');
    $columnProperty->setAccessible(TRUE);
    $columnProperty->setValue($widget, 'target_id');

    return $widget;
  }

  /**
   * Tests the massageFormValues method.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValues(): void {
    $widget = $this->createWidgetMock();

    // Input values as they come from the form - parent massageFormValues
    // first processes these, then our override adds primary handling.
    // Values with 'primary' key set indicate the primary selection.
    $values = [
      0 => ['primary' => 2],
      1 => ['target_id' => 1],
      2 => ['target_id' => 2],
    ];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    // The primary entry (0 => ['primary' => 2]) should be skipped.
    // Items with target_id should remain, with primary flag set appropriately.
    $this->assertGreaterThanOrEqual(1, count($result));

    // Find items and verify primary flag setting.
    foreach ($result as $item) {
      if (isset($item['target_id'])) {
        if ($item['target_id'] == 1) {
          $this->assertEquals(0, $item['primary'], 'Target ID 1 should not be primary.');
        }
        if ($item['target_id'] == 2) {
          $this->assertEquals(1, $item['primary'], 'Target ID 2 should be primary.');
        }
      }
    }
  }

  /**
   * Tests massageFormValues with no primary selected.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValuesNoPrimary(): void {
    $widget = $this->createWidgetMock();

    $values = [
      0 => ['target_id' => 1],
      1 => ['target_id' => 2],
    ];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    // All items should have primary = 0 when no primary is set.
    foreach ($result as $item) {
      if (isset($item['primary'])) {
        $this->assertEquals(0, $item['primary']);
      }
    }
  }

  /**
   * Tests massageFormValues with empty values.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValuesEmpty(): void {
    $widget = $this->createWidgetMock();

    $values = [];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    $this->assertCount(0, $result);
  }

  /**
   * Tests massageFormValues skips the primary entry when processing values.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValuesSkipsPrimaryEntry(): void {
    $widget = $this->createWidgetMock();

    // The primary selection comes in as its own entry.
    $values = [
      0 => ['primary' => 5],
      5 => ['target_id' => 5],
      10 => ['target_id' => 10],
    ];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    // Target ID 5 should be primary.
    $found_primary = FALSE;
    foreach ($result as $item) {
      if (isset($item['target_id']) && $item['target_id'] == 5) {
        $this->assertEquals(1, $item['primary']);
        $found_primary = TRUE;
      }
      elseif (isset($item['target_id'])) {
        $this->assertEquals(0, $item['primary']);
      }
    }
    $this->assertTrue($found_primary, 'Primary item should be found.');
  }

}
