<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Unit\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryReferenceAutocompleteWidget;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Unit tests for the PrimaryReferenceAutocompleteWidget.
 *
 * @group primary_entity_reference
 * @coversDefaultClass \Drupal\primary_entity_reference\Plugin\Field\FieldWidget\PrimaryReferenceAutocompleteWidget
 */
class PrimaryReferenceAutocompleteWidgetTest extends UnitTestCase {

  /**
   * The field definition.
   *
   * @var \Drupal\Core\Field\FieldDefinitionInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fieldDefinition;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The selection plugin manager.
   *
   * @var \Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $selectionManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $container->set('entity_type.manager', $this->entityTypeManager);

    $this->selectionManager = $this->createMock(SelectionPluginManagerInterface::class);
    $container->set('plugin.manager.entity_reference_selection', $this->selectionManager);

    $field_storage_definition = $this->createMock(FieldStorageDefinitionInterface::class);
    $field_storage_definition->expects($this->any())
      ->method('getSetting')
      ->willReturn('node');
    $field_storage_definition->expects($this->any())
      ->method('getCardinality')
      ->willReturn(-1);

    $this->fieldDefinition = $this->createMock(FieldDefinitionInterface::class);
    $this->fieldDefinition->expects($this->any())
      ->method('getName')
      ->willReturn('test_field');

    $this->fieldDefinition->expects($this->any())
      ->method('getSetting')
      ->willReturnMap([
        ['target_type', 'node'],
        ['handler', 'default:node'],
        ['handler_settings', ['target_bundles' => ['page' => 'page']]],
      ]);

    $this->fieldDefinition->expects($this->any())
      ->method('getFieldStorageDefinition')
      ->willReturn($field_storage_definition);

    // Mock entity type definition.
    $entityType = $this->createMock(EntityTypeInterface::class);
    $entityType->expects($this->any())
      ->method('getLabel')
      ->willReturn('Content');

    $this->entityTypeManager->expects($this->any())
      ->method('getDefinition')
      ->with('node')
      ->willReturn($entityType);

    \Drupal::setContainer($container);
  }

  /**
   * Tests the class extends the expected parent.
   */
  public function testClassInheritance(): void {
    $this->assertTrue(
      is_subclass_of(
        PrimaryReferenceAutocompleteWidget::class,
        'Drupal\Core\Field\Plugin\Field\FieldWidget\EntityReferenceAutocompleteWidget'
      ),
      'PrimaryReferenceAutocompleteWidget should extend EntityReferenceAutocompleteWidget.'
    );
  }

  /**
   * Tests the massageFormValues method.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValues(): void {
    $widget = new PrimaryReferenceAutocompleteWidget(
      'test_widget',
      [],
      $this->fieldDefinition,
      [],
      []
    );

    // Input values as they come from the form after parent massaging.
    // The primary value indicates which _original_delta should be marked.
    // primary=1 means the item with _original_delta=1 should be primary.
    $values = [
      0 => ['target_id' => 1, 'primary' => 1, '_original_delta' => 0],
      1 => ['target_id' => 2, 'primary' => 0, '_original_delta' => 1],
      2 => ['target_id' => 3, 'primary' => 0, '_original_delta' => 2],
    ];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    // The first loop finds primary=1 and sets $primary=1.
    // The second loop sets primary=1 where _original_delta==1.
    $this->assertEquals(0, $result[0]['primary'], 'Item 0 should not be primary.');
    $this->assertEquals(1, $result[1]['primary'], 'Item 1 should be primary because _original_delta matches primary value.');
    $this->assertEquals(0, $result[2]['primary'], 'Item 2 should not be primary.');
  }

  /**
   * Tests massageFormValues when primary matches original delta.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValuesPrimaryMatchesOriginalDelta(): void {
    $widget = new PrimaryReferenceAutocompleteWidget(
      'test_widget',
      [],
      $this->fieldDefinition,
      [],
      []
    );

    // primary=0 should mark the item with _original_delta=0 as primary.
    $values = [
      0 => ['target_id' => 1, 'primary' => 0, '_original_delta' => 0],
      1 => ['target_id' => 2, 'primary' => 0, '_original_delta' => 1],
    ];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    // Item 0 has _original_delta=0, which matches primary=0 (from first item).
    // But wait - primary=0 is falsy, so the first loop won't set $primary=0.
    // The default $primary stays 0, and _original_delta=0 matches it.
    $this->assertEquals(1, $result[0]['primary'], 'Item 0 should be primary because _original_delta matches default primary.');
    $this->assertEquals(0, $result[1]['primary'], 'Item 1 should not be primary.');
  }

  /**
   * Tests massageFormValues with no primary set.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValuesNoPrimary(): void {
    $widget = new PrimaryReferenceAutocompleteWidget(
      'test_widget',
      [],
      $this->fieldDefinition,
      [],
      []
    );

    $values = [
      0 => ['target_id' => 1, '_original_delta' => 0],
      1 => ['target_id' => 2, '_original_delta' => 1],
    ];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    // When no primary is explicitly set, $primary defaults to 0.
    // Item with _original_delta=0 becomes primary.
    $this->assertEquals(1, $result[0]['primary'], 'First item should be primary by default.');
    $this->assertEquals(0, $result[1]['primary']);
  }

  /**
   * Tests massageFormValues with first item as primary.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValuesFirstItemPrimary(): void {
    $widget = new PrimaryReferenceAutocompleteWidget(
      'test_widget',
      [],
      $this->fieldDefinition,
      [],
      []
    );

    // Explicitly set primary=0 to mark first item.
    $values = [
      0 => ['target_id' => 1, 'primary' => 0, '_original_delta' => 0],
      1 => ['target_id' => 2, 'primary' => 0, '_original_delta' => 1],
    ];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    // First item should be primary since _original_delta=0 matches primary=0.
    $this->assertEquals(1, $result[0]['primary']);
    $this->assertEquals(0, $result[1]['primary']);
  }

  /**
   * Tests massageFormValues with empty values.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValuesEmpty(): void {
    $widget = new PrimaryReferenceAutocompleteWidget(
      'test_widget',
      [],
      $this->fieldDefinition,
      [],
      []
    );

    $values = [];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    $this->assertCount(0, $result);
  }

  /**
   * Tests massageFormValues with second item as primary.
   *
   * @covers ::massageFormValues
   */
  public function testMassageFormValuesSecondItemPrimary(): void {
    $widget = new PrimaryReferenceAutocompleteWidget(
      'test_widget',
      [],
      $this->fieldDefinition,
      [],
      []
    );

    // Set primary=2 to mark item with _original_delta=2 as primary.
    $values = [
      0 => ['target_id' => 1, 'primary' => 2, '_original_delta' => 0],
      1 => ['target_id' => 2, 'primary' => 0, '_original_delta' => 1],
      2 => ['target_id' => 3, 'primary' => 0, '_original_delta' => 2],
    ];

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $widget->massageFormValues($values, $form, $form_state);

    $this->assertEquals(0, $result[0]['primary']);
    $this->assertEquals(0, $result[1]['primary']);
    $this->assertEquals(1, $result[2]['primary'], 'Item 2 should be primary.');
  }

}
