<?php

declare(strict_types=1);

namespace Drupal\Tests\primary_entity_reference\Unit\Plugin\Field;

use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\primary_entity_reference\Plugin\Field\PrimaryEntityReferenceFieldItemList;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Unit tests for PrimaryEntityReferenceFieldItemList.
 *
 * @group primary_entity_reference
 * @coversDefaultClass \Drupal\primary_entity_reference\Plugin\Field\PrimaryEntityReferenceFieldItemList
 */
class PrimaryEntityReferenceFieldItemListTest extends UnitTestCase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();
    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);
    \Drupal::setContainer($container);
  }

  /**
   * Tests the class extends the expected parent.
   */
  public function testClassInheritance(): void {
    $this->assertTrue(
      is_subclass_of(
        PrimaryEntityReferenceFieldItemList::class,
        'Drupal\Core\Field\EntityReferenceFieldItemList'
      ),
      'PrimaryEntityReferenceFieldItemList should extend EntityReferenceFieldItemList.'
    );
  }

  /**
   * Tests __get method returns primary item when accessing 'primary' property.
   *
   * @covers ::__get
   */
  public function testGetPrimaryProperty(): void {
    // Create mock for primary typed data that returns true.
    $primary_typed_data = $this->createMock(TypedDataInterface::class);
    $primary_typed_data->expects($this->once())
      ->method('getValue')
      ->willReturn(1);

    // Create mock item that is marked as primary.
    $item = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item->expects($this->once())
      ->method('get')
      ->with('primary')
      ->willReturn($primary_typed_data);

    // Create a partial mock of the item list without mocking any methods.
    $itemList = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->getMock();

    // Use reflection to set the protected list property.
    $reflection = new \ReflectionClass(PrimaryEntityReferenceFieldItemList::class);
    $listProperty = $reflection->getProperty('list');
    $listProperty->setAccessible(TRUE);
    $listProperty->setValue($itemList, [$item]);

    // Call the real __get method by using a real instance with the list set.
    $realItemList = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods([])
      ->getMock();
    $listProperty->setValue($realItemList, [$item]);

    // Access the 'primary' property via __get.
    $result = $realItemList->primary;

    $this->assertSame($item, $result);
  }

  /**
   * Tests primary() returns the primary item when one exists.
   *
   * @covers ::primary
   */
  public function testPrimaryReturnsMarkedItem(): void {
    // Create mock for non-primary typed data.
    $non_primary_typed_data = $this->createMock(TypedDataInterface::class);
    $non_primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(0);

    // Create mock for primary typed data.
    $primary_typed_data = $this->createMock(TypedDataInterface::class);
    $primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(1);

    // Create non-primary item.
    $item1 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item1->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);

    // Create primary item.
    $item2 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item2->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($primary_typed_data);

    // Create another non-primary item.
    $item3 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item3->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);

    // Create a partial mock of the item list.
    $itemList = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods([])
      ->getMock();

    // Use reflection to set the protected list property.
    $reflection = new \ReflectionClass(PrimaryEntityReferenceFieldItemList::class);
    $listProperty = $reflection->getProperty('list');
    $listProperty->setAccessible(TRUE);
    $listProperty->setValue($itemList, [$item1, $item2, $item3]);

    $result = $itemList->primary();

    $this->assertSame($item2, $result);
  }

  /**
   * Tests primary() returns first item when none marked as primary.
   *
   * @covers ::primary
   */
  public function testPrimaryReturnsFallbackToFirst(): void {
    // Create mock for non-primary typed data.
    $non_primary_typed_data = $this->createMock(TypedDataInterface::class);
    $non_primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(0);

    // Create non-primary items.
    $item1 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item1->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);

    $item2 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item2->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);

    // Create a partial mock of the item list.
    $itemList = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods([])
      ->getMock();

    // Use reflection to set the protected list property.
    $reflection = new \ReflectionClass(PrimaryEntityReferenceFieldItemList::class);
    $listProperty = $reflection->getProperty('list');
    $listProperty->setAccessible(TRUE);
    $listProperty->setValue($itemList, [$item1, $item2]);

    $result = $itemList->primary();

    // Should return first item as fallback.
    $this->assertSame($item1, $result);
  }

  /**
   * Tests primary() returns NULL when list is empty.
   *
   * @covers ::primary
   */
  public function testPrimaryReturnsNullWhenEmpty(): void {
    // Create a partial mock of the item list.
    $itemList = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods([])
      ->getMock();

    // Use reflection to set the protected list property.
    $reflection = new \ReflectionClass(PrimaryEntityReferenceFieldItemList::class);
    $listProperty = $reflection->getProperty('list');
    $listProperty->setAccessible(TRUE);
    $listProperty->setValue($itemList, []);

    $result = $itemList->primary();

    $this->assertNull($result);
  }

  /**
   * Tests preSave() sorts items with primary first.
   *
   * @covers ::preSave
   */
  public function testPreSaveSortsItems(): void {
    // Create mock for non-primary typed data.
    $non_primary_typed_data = $this->createMock(TypedDataInterface::class);
    $non_primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(0);

    // Create mock for primary typed data.
    $primary_typed_data = $this->createMock(TypedDataInterface::class);
    $primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(1);

    // Create items: first non-primary, second primary, third non-primary.
    $item1 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item1->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);
    $item1->expects($this->any())
      ->method('preSave')
      ->willReturn(NULL);

    $item2 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item2->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($primary_typed_data);
    $item2->expects($this->any())
      ->method('preSave')
      ->willReturn(NULL);

    $item3 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item3->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);
    $item3->expects($this->any())
      ->method('preSave')
      ->willReturn(NULL);

    // Create a partial mock of the item list.
    $itemList = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['filterEmptyItems'])
      ->getMock();

    $itemList->expects($this->any())
      ->method('filterEmptyItems')
      ->willReturnSelf();

    // Use reflection to set the protected list property.
    $reflection = new \ReflectionClass(PrimaryEntityReferenceFieldItemList::class);
    $listProperty = $reflection->getProperty('list');
    $listProperty->setAccessible(TRUE);
    $listProperty->setValue($itemList, [$item1, $item2, $item3]);

    // Call preSave.
    $itemList->preSave();

    // After sorting, item2 (primary) should be first.
    $list = $listProperty->getValue($itemList);
    $this->assertSame($item2, $list[0], 'Primary item should be sorted to first position.');
  }

  /**
   * Tests preSave() with all non-primary items.
   *
   * @covers ::preSave
   */
  public function testPreSaveWithNoPrimary(): void {
    // Create mock for non-primary typed data.
    $non_primary_typed_data = $this->createMock(TypedDataInterface::class);
    $non_primary_typed_data->expects($this->any())
      ->method('getValue')
      ->willReturn(0);

    $item1 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item1->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);
    $item1->expects($this->any())
      ->method('preSave')
      ->willReturn(NULL);

    $item2 = $this->createMock('Drupal\Core\Field\FieldItemInterface');
    $item2->expects($this->any())
      ->method('get')
      ->with('primary')
      ->willReturn($non_primary_typed_data);
    $item2->expects($this->any())
      ->method('preSave')
      ->willReturn(NULL);

    // Create a partial mock of the item list.
    $itemList = $this->getMockBuilder(PrimaryEntityReferenceFieldItemList::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['filterEmptyItems'])
      ->getMock();

    $itemList->expects($this->any())
      ->method('filterEmptyItems')
      ->willReturnSelf();

    // Use reflection to set the protected list property.
    $reflection = new \ReflectionClass(PrimaryEntityReferenceFieldItemList::class);
    $listProperty = $reflection->getProperty('list');
    $listProperty->setAccessible(TRUE);
    $listProperty->setValue($itemList, [$item1, $item2]);

    // Call preSave.
    $itemList->preSave();

    // Order should remain the same (stable sort).
    $list = $listProperty->getValue($itemList);
    $this->assertCount(2, $list);
  }

}
