<?php

declare(strict_types=1);

namespace Drupal\privacy_log\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;

/**
 * Alters forms.
 */
class Form {

  use StringTranslationTrait;

  /**
   * Constructs a new Form hook provider.
   *
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    TranslationInterface $string_translation,
  ) {
    $this->setStringTranslation($string_translation);
  }

  /**
   * Alters the logging settings form provided by Drupal Core's system module.
   *
   * Adds "Database log messages expiry" setting to the form, if the database
   * logging module is enabled.
   *
   * @param array<string|int,mixed> $form
   *   Nested array of form elements that comprise the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form. The arguments that
   *   \Drupal::formBuilder()->getForm() was originally called with are
   *   available in the array $form_state->getBuildInfo()['args'].
   *
   * @see hook_form_FORM_ID_alter()
   * @see privacy_log_form_system_logging_settings_alter()
   */
  #[Hook('form_system_logging_settings_alter')]
  public function alterSystemLoggingSettingsForm(array &$form, FormStateInterface $form_state): void {
    if (!isset($form['dblog_row_limit'])) {
      return;
    }

    $form['privacy_log_dblog_expiry'] = [
      '#type' => 'select',
      '#title' => $this->t('Database log messages expiry'),
      '#config_target' => 'privacy_log.settings:dblog_expiry',
      '#options' => [
        0 => $this->t('Never'),
        1 => $this->formatPlural(1, '1 hour', '@count hours'),
        3 => $this->formatPlural(3, '1 hour', '@count hours'),
        6 => $this->formatPlural(6, '1 hour', '@count hours'),
        12 => $this->formatPlural(12, '1 hour', '@count hours'),
        24 => $this->formatPlural(1, '1 day', '@count days'),
        48 => $this->formatPlural(48 / 24, '1 day', '@count days'),
        72 => $this->formatPlural(72 / 24, '1 day', '@count days'),
        96 => $this->formatPlural(96 / 24, '1 day', '@count days'),
        120 => $this->formatPlural(120 / 24, '1 day', '@count days'),
        144 => $this->formatPlural(144 / 24, '1 day', '@count days'),
        168 => $this->formatPlural(1, '1 week', '@count weeks'),
      ],
      '#description' => $this->t('Maximum age of a message in the database log before it will be deleted. Requires a <a href=":cron">cron maintenance task</a>.', [
        ':cron' => Url::fromRoute('system.status')->toString(),
      ]),
    ];
  }

}
