<?php

declare(strict_types=1);

namespace Drupal\Tests\privacy_log\Kernel;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\KernelTests\KernelTestBase;
use Drupal\privacy_log\PrivacyProtectingLoggerChannelFactory;
use Symfony\Component\ErrorHandler\BufferingLogger;

/**
 * Tests that IP addresses on log messages are removed correctly.
 *
 * @group privacy_log
 */
class PrivacyLogTest extends KernelTestBase {

  /**
   * The service name for a logger implementation that collects anything logged.
   *
   * @var string
   */
  protected $testLogServiceName = 'privacy_log_test.logger';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['system', 'privacy_log'];

  /**
   * Tests logging via Privacy Log.
   */
  public function testLogging(): void {
    // Verify nothing is logged.
    self::assertEmpty($this->container->get($this->testLogServiceName)->cleanLogs());

    // Verify privacy protecting logger channel factory is in use.
    $factory = \Drupal::getContainer()->get('logger.factory');
    self::assertEquals(PrivacyProtectingLoggerChannelFactory::class, $factory::class);

    // Log something.
    $factory->get($this->testLogServiceName)->warning('Hello World');
    \Drupal::logger($this->testLogServiceName)->critical('Hello World');

    // Verify IP address is empty.
    $logs = $this->container->get($this->testLogServiceName)->cleanLogs();
    foreach ($logs as $key => $log) {
      self::assertArrayHasKey(1, $logs[$key]);
      self::assertEquals('Hello World', $logs[$key][1]);
      self::assertArrayHasKey(2, $logs[$key]);
      self::assertArrayHasKey('ip', $logs[$key][2]);
      $ip = $logs[$key][2]['ip'];
      self::assertEquals('', $ip);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function register(ContainerBuilder $container): void {
    parent::register($container);
    $container
      ->register($this->testLogServiceName, BufferingLogger::class)
      ->addTag('logger');
  }

}
