<?php

declare(strict_types=1);

namespace Drupal\Tests\privacy_log\Kernel;

use Drupal\Core\Database\Database;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests integration with database log module.
 *
 * @group privacy_log
 */
class DbLogTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['dblog', 'system', 'privacy_log'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('dblog', ['watchdog']);
    $this->installConfig(['system']);
  }

  /**
   * Tests that cron correctly applies the database log expiry setting.
   */
  public function testDbLogCron(): void {

    $user = new AnonymousUserSession();
    $time = \Drupal::time()->getRequestTime();

    // Prepare the fields to be logged.
    $log = [
      'channel'     => 'privacy_log_test',
      'message'     => '',
      'variables'   => [],
      'severity'    => RfcLogLevel::NOTICE,
      'link'        => NULL,
      'uid'         => $user->id(),
      'request_uri' => \Drupal::request()->getRequestUri(),
      'referer'     => \Drupal::request()->server->get('HTTP_REFERER'),
      'ip'          => '127.0.0.1',
      'timestamp'   => '',
    ];

    // Log 4 test messages with different timestamps.
    $logger = $this->container->get('logger.dblog');

    $log['message'] = 'Privacy Log test log message #0';
    $log['timestamp'] = $time - (8 * 24 * 60 * 60);
    $logger->log($log['severity'], $log['message'], $log);

    $log['message'] = 'Privacy Log test log message #1';
    $log['timestamp'] = $time - (7 * 24 * 60 * 60);
    $logger->log($log['severity'], $log['message'], $log);

    $log['message'] = 'Privacy Log test log message #2';
    $log['timestamp'] = $time - (6 * 24 * 60 * 60);
    $logger->log($log['severity'], $log['message'], $log);

    $log['message'] = 'Privacy Log test log message #3';
    $log['timestamp'] = $time;
    $logger->log($log['severity'], $log['message'], $log);

    $connection = Database::getConnection();

    // Verify 4 log messages present.
    $rows = $connection->select('watchdog')
      ->condition('message', [
        'Privacy Log test log message #0',
        'Privacy Log test log message #1',
        'Privacy Log test log message #2',
        'Privacy Log test log message #3',
      ], 'IN')
      ->countQuery()
      ->execute()
      ->fetchField();
    self::assertEquals(4, $rows);

    // Run cron.
    \Drupal::service('cron')->run();

    // Verify 3 log messages present.
    $rows = $connection->select('watchdog')
      ->condition('message', [
        'Privacy Log test log message #0',
        'Privacy Log test log message #1',
        'Privacy Log test log message #2',
        'Privacy Log test log message #3',
      ], 'IN')
      ->countQuery()
      ->execute()
      ->fetchField();
    self::assertEquals(3, $rows);

    // Verify message #0 deleted.
    $rows = $connection->select('watchdog')
      ->condition('message', 'Privacy Log test log message #0')
      ->countQuery()
      ->execute()
      ->fetchField();
    self::assertEquals(0, $rows);

    // Verify message #1 not deleted.
    $rows = $connection->select('watchdog')
      ->condition('message', 'Privacy Log test log message #1')
      ->countQuery()
      ->execute()
      ->fetchField();
    self::assertEquals(1, $rows);
  }

}
