<?php

namespace Drupal\privatemsg\Drush\Commands;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\privatemsg\Entity\PrivatemsgMessage;
use Drupal\privatemsg\PrivateMsgService;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;

/**
 * A Drush command file.
 */
class PrivatemsgCommands extends DrushCommands {

  use AutowireTrait;

  /**
   * Constructs a PrivatemsgCommands object.
   *
   * @param \Drupal\privatemsg\PrivateMsgService $privateMsgService
   *   Common functions.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    protected readonly PrivateMsgService $privateMsgService,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly Connection $database,
  ) {
    parent::__construct();
  }

  /**
   * Migrate data from 1.x to 2.x.
   */
  #[CLI\Command(name: 'privatemsg:1to2', aliases: ['pmsg1to2'])]
  #[CLI\Usage(name: 'privatemsg:1to2', description: 'Usage description')]
  public function privatemsg1to2(): void {
    if ($this->database->schema()->tableExists('pm_index_old') && $this->database->schema()->tableExists('pm_message_old')) {
      $this->output()->writeln('1.x database tables found.');

      $query = $this->database->select('pm_message_old', 'm');
      $query->innerJoin('pm_index_old', 'i', 'i.mid = m.mid');
      $query->fields('m', ['mid', 'author', 'subject', 'body', 'format', 'timestamp']);
      $query->fields('i', ['thread_id', 'recipient']);
      $messages_all = $query->execute()->fetchAll();
      $messages = [];
      foreach ($messages_all as $message) {
        if ($message->author !== $message->recipient) {
          $messages[$message->thread_id][$message->mid] = $message;
        }
      }

      foreach ($messages as $thread) {
        $messages_for_thread = [];
        $thread_members = [];
        $subject = '';
        $updated_custom = NULL;
        foreach ($thread as $thread_message) {
          $message_entity = PrivatemsgMessage::create([
            'id' => $thread_message->mid,
            'owner' => $thread_message->author,
            'message' => [
              'value' => $thread_message->body,
              'format' => 'basic_html',
            ],
            'created' => $thread_message->timestamp,
          ]);
          $message_entity->save();
          $messages_for_thread[] = $message_entity->id();
          $thread_members[$thread_message->author] = $thread_message->author;
          $thread_members[$thread_message->recipient] = $thread_message->recipient;
          $subject = $thread_message->subject;
          $updated_custom = $thread_message->timestamp;
        }

        $group_id = $this->privateMsgService->getLastThreadGroupNumber();
        ++$group_id;
        foreach ($thread_members as $thread_member) {
          $data = [
            'subject' => $subject,
            'members' => $thread_members,
            'owner' => $thread_member,
            'group' => $group_id,
            'private_messages' => $messages_for_thread,
            'updated_custom' => $updated_custom,
          ];
          $thread_entity = $this->entityTypeManager
            ->getStorage('privatemsg_thread')
            ->create($data);
          $thread_entity->save();
        }
      }
      $this->output()->writeln('Migration completed.');
    }
    else {
      $this->output()->writeln('1.x database tables not found.');
    }
  }

}
