<?php

namespace Drupal\privatemsg\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\privatemsg\PrivateMsgService;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Returns responses for Private messages routes.
 */
class PrivatemsgController extends ControllerBase {

  /**
   * Constructs a PrivatemsgController.
   *
   * @param \Drupal\privatemsg\PrivateMsgService $privateMsgService
   *   Common functions.
   */
  public function __construct(
    protected readonly PrivateMsgService $privateMsgService,
  ) {
  }

  /**
   * Remove message.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse|\Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function removeMessage(Request $request) {
    $thread_id = $request->attributes->get('thread_id');
    $mid = $request->attributes->get('mid');

    $incorrect_response = new Response();
    $incorrect_response->setContent('Incorrect request.');
    $incorrect_response->setStatusCode(400);

    if (!$thread_id || !$mid) {
      return $incorrect_response;
    }

    /** @var \Drupal\privatemsg\Entity\PrivatemsgMessage $message */
    $message = $this->entityTypeManager()->getStorage('privatemsg_message')->load($mid);
    if ($message && $this->currentUser()->hasPermission('privatemsg delete own messages') && $message->getOwnerId() === $this->currentUser()->id()) {
      $message->markMessageAsDeleted();
      $message->save();

      $response = new AjaxResponse();
      $url = Url::fromRoute('entity.privatemsg_thread.canonical', [
        'privatemsg_thread' => $thread_id,
      ]);
      $response->addCommand(new RedirectCommand($url->toString()));
      return $response;
    }

    return $incorrect_response;
  }

  /**
   * Block user.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse|\Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function blockUser(Request $request) {
    $user_id = $request->attributes->get('user');

    $incorrect_response = new Response();
    $incorrect_response->setContent('Incorrect request.');
    $incorrect_response->setStatusCode(400);

    if (!$user_id) {
      return $incorrect_response;
    }

    $isBlocked = $this->privateMsgService->isUserBlocked($this->currentUser()->id(), $user_id);
    $response = new AjaxResponse();
    $selector = '#privatemsg-block-user-' . $user_id;

    $url_object = Url::fromRoute('privatemsg.block_user', [
      'user' => $user_id,
    ]);
    $url = $url_object->toString();

    if ($isBlocked) {
      $this->privateMsgService->unblockUser($user_id);
      $text = $this->t('block');
      $content = "<a id=\"privatemsg-block-user-$user_id\" class=\"privatemsg-block-user use-ajax\" href=\"$url\">$text</a>";
      $response->addCommand(new ReplaceCommand($selector, $content));
      return $response;
    }

    $this->privateMsgService->blockUser($user_id);
    $text = $this->t('unblock');
    $content = "<a id=\"privatemsg-block-user-$user_id\" class=\"privatemsg-block-user use-ajax\" href=\"$url\">$text</a>";
    $response->addCommand(new ReplaceCommand($selector, $content));
    return $response;
  }

  /**
   * Leave multiple chat.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse|\Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function leaveChat(Request $request) {
    $thread_id = $request->attributes->get('thread_id');

    $incorrect_response = new Response();
    $incorrect_response->setContent('Incorrect request.');
    $incorrect_response->setStatusCode(400);

    if (!$thread_id) {
      return $incorrect_response;
    }

    /** @var \Drupal\privatemsg\Entity\PrivatemsgThreadInterface $thread */
    $thread = $this->entityTypeManager()->getStorage('privatemsg_thread')->load($thread_id);

    if (!$thread) {
      return $incorrect_response;
    }

    $members = $thread->getMembers();
    if ($this->currentUser()->id() !== $thread->getOwnerId() || count($members) < 3) {
      return $incorrect_response;
    }

    $group = $thread->getGroup();
    $threads = $this->privateMsgService->getThreadsFromGroup($group);
    foreach ($threads as $thread) {
      $thread->removeMember($this->currentUser()->id());

      if ($this->currentUser()->id() === $thread->getOwnerId()) {
        $thread->markAsDeleted();
      }

      $thread->save();
    }

    $redirect_url = Url::fromRoute('view.all_privatemsg_threads.page_1');
    return new RedirectResponse($redirect_url->toString());
  }

}
