<?php

declare(strict_types=1);

namespace Drupal\Tests\privatemsg\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests for the privatemsg module.
 */
class PrivatemsgMassChangeTagsTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'privatemsg',
  ];

  /**
   * The User used for the test.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user1;

  /**
   * The User used for the test.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user2;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->user1 = $this->DrupalCreateUser([
      'privatemsg write messages',
    ]);
    \Drupal::service('user.data')->set('privatemsg', $this->user1->id(), 'enable', 1);
    $this->user2 = $this->DrupalCreateUser([
      'privatemsg write messages',
      'privatemsg use messages actions',
    ]);
    \Drupal::service('user.data')->set('privatemsg', $this->user2->id(), 'enable', 1);
  }

  /**
   * Helper function that executes en operation.
   *
   * @param string $path
   *   The path of the View page that includes VBO.
   * @param string $button_text
   *   The form submits button text.
   * @param int[] $selection
   *   The selected items' indexes.
   * @param array $data
   *   Additional parameters for the submitted form.
   */
  protected function executeAction(string $path, string $button_text, array $selection = [], array $data = []): void {
    foreach ($selection as $index) {
      $data["views_bulk_operations_bulk_form[$index]"] = TRUE;
    }
    $this->drupalGet($path);
    $this->submitForm($data, $button_text);
  }

  /**
   * Tests VBO mass change tags.
   */
  public function testMassChangeTags(): void {
    $this->drupalLogin($this->user1);
    $this->drupalGet('/messages/new');

    $this->submitForm([
      'thread_members' => $this->user2->getDisplayName() . ' (' . $this->user2->id() . ')',
      'message[0][value]' => $this->getRandomGenerator()->sentences(5),
    ], 'Send message', 'privatemsg-message-add-form');

    // User 1 doesn't have permissions to use actions.
    $selected = [0];
    $data = ['action' => 0];
    $this->executeAction('/messages', 'Apply to selected items', $selected, $data);
    $this->submitForm([
      'thread_tags' => 'Favorites (1)',
    ], 'Apply');
    $this->assertSession()->elementNotExists('xpath', '//td[@class="views-field views-field-tags-target-id"]/a/text()');

    // User 2 has permissions to use actions.
    $this->drupalLogin($this->user2);
    $selected = [0];
    $data = ['action' => 0];

    /* Create new tag */
    $this->executeAction('/messages', 'Apply to selected items', $selected, $data);
    $this->submitForm([
      'thread_tags' => 'NewTag',
    ], 'Apply');
    $this->assertSession()->elementTextContains('css', 'td.views-field-tags-target-id a', 'NewTag');
    $this->assertSession()->elementsCount('xpath', "//td[@class='views-field views-field-tags-target-id'][1]/a", 1);

    /* Clear tags */
    $this->executeAction('/messages', 'Apply to selected items', $selected, $data);
    $this->submitForm([
      'thread_tags' => NULL,
    ], 'Apply');
    $this->assertSession()->elementTextContains('css', 'td.views-field-tags-target-id', '');
  }

}
