<?php

namespace Drupal\privatemsg\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\privatemsg\Entity\PrivatemsgThreadInterface;
use Drupal\privatemsg\PrivateMsgService;
use Drupal\views_bulk_operations\Action\ViewsBulkOperationsActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Remove thread.
 */
#[Action(
  id: 'privatemsg_remove_thread_action',
  label: new TranslatableMarkup('Remove'),
  type: 'privatemsg_thread'
)]
class PrivatemsgRemoveThreadAction extends ViewsBulkOperationsActionBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * Constructs a PrivatemsgRemoveThreadAction object.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly PrivateMsgService $privateMsgService,
    protected readonly AccountInterface $currentUser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('privatemsg.common'),
      $container->get('current_user'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    return $object->access('remove_thread', $account, $return_as_object);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(?ContentEntityInterface $entity = NULL) {
    if ($entity instanceof PrivatemsgThreadInterface) {
      $entity->markAsDeleted();
      $entity->save();

      $members = $entity->getMembers();
      if (count($members) > 2) {
        $group = $entity->getGroup();
        $threads = $this->privateMsgService->getThreadsFromGroup($group);
        foreach ($threads as $thread) {
          $thread->removeMember($this->currentUser->id());
          $thread->save();
        }
      }

      return 'Selected threads were removed';
    }

    return 'You must first select one (or more) messages before you can take that action.';
  }

}
