<?php

namespace Drupal\privatemsg\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\privatemsg\PrivateMsgService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a private messages block.
 *
 * @Block(
 *   id = "privatemsg_block",
 *   admin_label = @Translation("Private Messages"),
 *   category = @Translation("Private Messages")
 * )
 */
class PrivateMessagesBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new PrivateMessagesBlock object.
   *
   * @param array $configuration
   *   The configuration array for the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\privatemsg\PrivateMsgService $privateMsgService
   *   The privatemsg.common service used to handle private messages.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user account.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly PrivateMsgService $privateMsgService,
    protected readonly AccountInterface $currentUser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('privatemsg.common'),
      $container->get('current_user'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $count = $this->privateMsgService->getUnreadThreadCount();
    $markup = '';

    if ($count < 1) {
      $markup .= '<p><a href="/messages" class="privatemsg-block-new-messages-count">' . $this->t('All messages') . '</a></p>';
    }
    else {
      $markup .= '<p><strong><a href="/messages" class="privatemsg-block-new-messages-count">' . $this->t('All messages') . ' (' . $count . ')</a></strong></p>';
    }

    $markup .= '<p><a href="/messages/new">' . $this->t('New message') . '</a></p>';

    $build['content'] = [
      '#markup' => $markup,
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

}
