<?php

declare(strict_types=1);

namespace Drupal\Tests\privatemsg\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests for the privatemsg module.
 */
class PrivatemsgRemoveThreadActionTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'privatemsg',
  ];

  /**
   * The User used for the test.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user1;

  /**
   * The User used for the test.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user2;

  /**
   * The User used for the test.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user3;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->user1 = $this->DrupalCreateUser([
      'privatemsg write messages',
      'privatemsg use messages actions',
    ]);
    \Drupal::service('user.data')->set('privatemsg', $this->user1->id(), 'enable', 1);

    $this->user2 = $this->DrupalCreateUser([
      'privatemsg write messages',
      'privatemsg use messages actions',
    ]);
    \Drupal::service('user.data')->set('privatemsg', $this->user2->id(), 'enable', 1);

    $this->user3 = $this->DrupalCreateUser([
      'privatemsg write messages',
      'privatemsg use messages actions',
    ]);
    \Drupal::service('user.data')->set('privatemsg', $this->user3->id(), 'enable', 1);
  }

  /**
   * Helper function that executes en operation.
   *
   * @param string $path
   *   The path of the View page that includes VBO.
   * @param string $button_text
   *   The form submits button text.
   * @param int[] $selection
   *   The selected items' indexes.
   * @param array $data
   *   Additional parameters for the submitted form.
   */
  protected function executeAction(string $path, string $button_text, array $selection = [], array $data = []): void {
    foreach ($selection as $index) {
      $data["views_bulk_operations_bulk_form[$index]"] = TRUE;
    }
    $this->drupalGet($path);
    $this->submitForm($data, $button_text);
  }

  /**
   * Tests removing thread functionality.
   */
  public function testRemoveThreadAction(): void {
    $this->drupalLogin($this->user1);
    $this->drupalGet('/messages/new');
    $this->submitForm([
      'thread_members' => $this->user2->getDisplayName() . ' (' . $this->user2->id() . ')',
      'message[0][value]' => $this->getRandomGenerator()->sentences(5),
    ], 'Send message', 'privatemsg-message-add-form');

    $this->drupalLogin($this->user2);

    // Check thread availability.
    $this->drupalGet('messages/view/1');
    $this->assertSession()->statusCodeEquals(200);
    // Check thread visibility.
    $this->drupalGet('messages');
    $this->assertSession()->elementsCount('xpath', '//table/tbody/tr', 1);

    // Remove thread.
    $selected = [0];
    $data = ['action' => 3];
    $this->executeAction('messages', 'Apply to selected items', $selected, $data);

    // Check thread availability.
    $this->drupalGet('messages/view/1');
    $this->assertSession()->statusCodeEquals(403);
    // Check thread visibility.
    $this->drupalGet('messages');
    $this->assertSession()->elementTextContains('css', '.vbo-view-form', 'No messages available.');

    // User 1 can't write in thread because User 2 removed own thread.
    $this->drupalLogin($this->user1);
    $this->drupalGet('messages/view/2');
    $this->assertSession()->pageTextContains("You can't write messages because your interlocutor deleted his thread.");
  }

  /**
   * Tests removing thread functionality.
   */
  public function testRemoveFromMultipleThread(): void {
    $this->drupalLogin($this->user1);
    $this->drupalGet('/messages/new');
    $this->submitForm([
      'thread_members' => $this->user2->getDisplayName() . ' (' . $this->user2->id() . '),' . $this->user3->getDisplayName() . ' (' . $this->user3->id() . ')',
      'message[0][value]' => $this->getRandomGenerator()->sentences(5),
    ], 'Send message', 'privatemsg-message-add-form');

    $this->drupalLogin($this->user2);

    // Remove thread.
    $selected = [0];
    $data = ['action' => 3];
    $this->executeAction('messages', 'Apply to selected items', $selected, $data);

    // Check members count.
    $threads_storage = $this->container->get('entity_type.manager')->getStorage('privatemsg_thread');
    $threads_storage->resetCache([1, 2, 3]);
    for ($i = 1; $i <= 3; $i++) {
      $thread = $threads_storage->load($i);
      $members = $thread->getMembers();
      $this->assertCount(2, $members);
    }

    $this->drupalLogin($this->user1);
    $this->drupalGet('messages/view/3');
    $this->assertSession()->pageTextNotContains("You can't write messages because your interlocutor deleted his thread.");
  }

}
