<?php

namespace Drupal\proc\Drush\Commands;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityStorageException;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\proc\ProcKeyManager;

/**
 * Commands for managing protected content.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
final class ProcCommands extends DrushCommands implements ProcCommandsInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The key manager.
   *
   * @var \Drupal\proc\ProcKeyManager
   */
  protected ProcKeyManager $keyManager;

  /**
   * Constructs a ProcCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\proc\ProcKeyManager $keyManager
   *   The proc key manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ProcKeyManager $keyManager
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->keyManager = $keyManager;
    parent::__construct();
  }

  /**
   * Create a new instance of this command.
   */
  public static function create(ContainerInterface $container): ProcCommands {
    return new ProcCommands(
      $container->get('entity_type.manager'),
      $container->get('proc.key_manager')
    );
  }

  /**
   * Remove contents of wished set of recipients field in a given proc.
   *
   * @command proc:remove-wished
   * @aliases prw
   * @usage proc:remove-wished
   *     Remove contents of wished set of recipients field in a given proc.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  #[CLI\Command(name: 'proc:remove-wished', aliases: ['prw'])]
  #[CLI\Usage(name: 'proc:remove-wished', description: 'Remove contents of wished set of recipients field in a given proc cipher text entity, if any.')]
  public function removeWishedRecipients(array $options = ['proc_id' => NULL]): void {
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($proc_id)) {
      $this->logger()->error('No proc_id provided.');
      return;
    }

    $entity = NULL;

    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if ($storage) {
        $entity = $storage->load($proc_id);
      }
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      // Try next candidate type.
      $this->logger()->error("Proc entity with ID {$proc_id} not found.");
      return;
    }
    if (!$entity->hasField('field_wished_recipients_set')) {
      $this->logger()->warning("Entity {$proc_id} does not have field `field_wished_recipients_set`.");
      return;
    }
    if (empty($entity->get('field_wished_recipients_set')->getValue())) {
      $this->logger()->error("No wished recipients to remove for proc {$proc_id}.");
      return;
    }
    // Reset the field to an empty state.
    $entity->set('field_wished_recipients_set', []);
    try {
      $entity->save();
      $this->logger()->success("Reset field_wished_recipients_set for proc {$proc_id}.");
    }
    catch (EntityStorageException $e) {
      $this->logger()->error($e->getMessage());
    }
  }

  /**
   * Get update jobs from given key ID.
   *
   * @command proc:get-update-jobs
   * @aliases guj
   * @usage proc:get-update-jobs
   *     Get update jobs from given key ID.
   */
  #[CLI\Command(name: 'proc:get-update-jobs', aliases: ['guj'])]
  #[CLI\Usage(name: 'guj', description: 'Get update jobs, if any, from given user ID.')]
  public function getUpdateJobsFromUserId(array $options = ['user_id' => NULL]): void {
    $user_id = $options['user_id'] ?? NULL;

    if (empty($user_id)) {
      $this->logger()->error('No user ID provided.');
      return;
    }

    try {
      $keyring = $this->keyManager->getKeys($user_id, 'user_id');
    }
    catch (InvalidPluginDefinitionException|PluginNotFoundException $e) {
      $this->logger()->error("Keyring for user ID {$user_id} not found.");
      return;
    }
    $update_jobs = $keyring['keyring_entity']->get('meta')->getValue()[0]['update_jobs'] ?? '';
    if (empty($update_jobs)) {
      $this->logger()->info('No update jobs found.');
      return;
    }
    $update_jobs_string = implode("\n", $update_jobs);
    $this->logger()->success("Update job(s) for user ID {$user_id}: \n {$update_jobs_string}");

  }

  /**
   * Get update workers from given proc ID.
   *
   * @command proc:get-update-workers
   * @aliases guw
   * @usage proc:get-update-jobs
   *     Get update workers from given proc ID.
   */
  #[CLI\Command(name: 'proc:get-update-workers', aliases: ['guw'])]
  #[CLI\Usage(name: 'guw', description: 'Get update workers, if any, from given proc ID.')]
  public function getUpdateWorkers(array $options = ['proc_id' => NULL]): void {
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($proc_id)) {
      $this->logger()->error('No proc ID provided.');
      return;
    }

    $entity = NULL;

    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if ($storage) {
        $entity = $storage->load($proc_id);
      }
    }
    catch (InvalidPluginDefinitionException|PluginNotFoundException $e) {
      // Try next candidate type.
      $this->logger()->error("Proc entity with ID {$proc_id} not found.");
      return;
    }
    // Check among the recipients if any is marked as update worker, by having
    // in the update_jobs list in the metadata of the keyring the proc ID.
    $recipients = $entity->get('field_recipients_set')->getValue();
    $update_workers = [];
    foreach ($recipients as $recipient) {
      $recipient_id = $recipient['target_id'] ?? NULL;
      if (empty($recipient_id)) {
        continue;
      }
      try {
        $keyring = $this->keyManager->getKeys($recipient_id, 'user_id');
      }
      catch (InvalidPluginDefinitionException|PluginNotFoundException $e) {
        $this->logger()
          ->error("Keyring for recipient ID {$recipient_id} not found.");
        continue;
      }
      $update_jobs = $keyring['keyring_entity']->get('meta')
        ->getValue()[0]['update_jobs'] ?? [];
      if (in_array($proc_id, $update_jobs, TRUE)) {
        $update_workers[] = $recipient_id;
      }
    }
    if (empty($update_workers)) {
      $this->logger()->info('No update workers found.');
      return;
    }
    // Sort the update workers list:
    sort($update_workers);

    $update_workers_string = implode("\n",$update_workers);
    $this->logger()
      ->success("Update workers for proc ID {$proc_id}: {$update_workers_string}");
  }

  /**
   * Remove given recipient from given proc by their IDs.
   *
   * @command proc:remove-recipient
   * @aliases prr
   * @usage proc:remove-recipient
   *     Remove given recipient from given proc by their IDs.
   */
  #[CLI\Command(name: 'proc:remove-recipient', aliases: ['prr'])]
  #[CLI\Usage(name: 'prr', description: 'Remove given recipient from given proc by their IDs.')]
  public function removeRecipient(array $options = ['proc_id' => NULL, 'user_id' => NULL]): void {
    $proc_id = $options['proc_id'] ?? NULL;
    $user_id = $options['user_id'] ?? NULL;

    if (empty($proc_id) || empty($user_id)) {
      $this->logger()->error('Both proc_id and user_id must be provided.');
      return;
    }

    // Load proc entity.
    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if (!$storage) {
        $this->logger()->error("Proc storage not available.");
        return;
      }
      $proc = $storage->load($proc_id);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error("Proc entity with ID {$proc_id} not found.");
      return;
    }

    if (!$proc) {
      $this->logger()->error("Proc entity with ID {$proc_id} not found.");
      return;
    }

    if (!$proc->hasField('field_recipients_set')) {
      $this->logger()->warning("Entity {$proc_id} does not have field `field_recipients_set`.");
      return;
    }

    $recipients = $proc->get('field_recipients_set')->getValue() ?? [];
    if (empty($recipients)) {
      $this->logger()->info("No recipients present on proc {$proc_id}.");
    }

    // Filter out the recipient to remove.
    $new_recipients = array_values(array_filter($recipients, function ($item) use ($user_id) {
      $target = $item['target_id'] ?? NULL;
      return $target === NULL || (string) $target !== (string) $user_id;
    }));

    if (count($new_recipients) === count($recipients)) {
      $this->logger()->info("Recipient {$user_id} not found on proc {$proc_id}.");
    }
    else {
      try {
        $proc->set('field_recipients_set', $new_recipients);
        $proc->save();
        $this->logger()->success("Removed recipient {$user_id} from proc {$proc_id}.");
      }
      catch (EntityStorageException $e) {
        $this->logger()->error($e->getMessage());
      }
    }
  }

  /**
   * Remove update jobs, if any, from given user ID.
   *
   * @command proc:remove-update-jobs
   * @aliases ruj
   * @usage proc:remove-update-jobs
   *     Remove update jobs, if any, from given user ID.
   */
  #[CLI\Command(name: 'proc:remove-update-jobs', aliases: ['ruj'])]
  #[CLI\Usage(name: 'guj', description: 'Remove update jobs, if any, from given user ID.')]
  public function removeUpdateJobsFromUserId(array $options = ['user_id' => NULL]): void {
    $user_id = $options['user_id'] ?? NULL;

    if (empty($user_id)) {
      $this->logger()->error('No user ID provided.');
      return;
    }

    try {
      $keyring = $this->keyManager->getKeys($user_id, 'user_id');
    }
    catch (InvalidPluginDefinitionException|PluginNotFoundException $e) {
      $this->logger()->error("Keyring for user ID {$user_id} not found.");
      return;
    }

    $keyring_entity = $keyring['keyring_entity'] ?? NULL;
    if (empty($keyring_entity) || !$keyring_entity->hasField('meta')) {
      $this->logger()->warning("Keyring entity for user ID {$user_id} has no meta field.");
      return;
    }

    $meta_values = $keyring_entity->get('meta')->getValue();
    $current_update_jobs = $meta_values[0]['update_jobs'] ?? [];

    if (empty($current_update_jobs)) {
      $this->logger()->info("No update jobs to remove for user ID {$user_id}.");
      return;
    }

    // Clear the update_jobs entry while preserving other meta values.
    $meta_item = $meta_values[0];
    $meta_item['update_jobs'] = [];
    $meta_values[0] = $meta_item;

    try {
      $keyring_entity->set('meta', $meta_values);
      $keyring_entity->save();
      $this->logger()->success("Removed update jobs for user ID {$user_id}.");
    }
    catch (EntityStorageException|\Exception $e) {
      $this->logger()->error($e->getMessage());
    }
  }

  /**
   * Add update job for user by given user and proc IDs.
   *
   * @command proc:add-update-job
   * @aliases auj
   * @usage proc:add-update-job
   *     Add update job for user by given user and proc IDs
   */
  #[CLI\Command(name: 'proc:add-update-job', aliases: ['auj'])]
  #[CLI\Usage(name: 'guj', description: 'Add update job for user by given user and proc IDs')]
  public function addUpdateJob(array $options = ['user_id' => NULL, 'proc_id' => NULL]): void {
    $user_id = $options['user_id'] ?? NULL;
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($user_id) || empty($proc_id)) {
      $this->logger()->error('Both user_id and proc_id must be provided.');
      return;
    }

    try {
      $keyring = $this->keyManager->getKeys($user_id, 'user_id');
    }
    catch (InvalidPluginDefinitionException|PluginNotFoundException $e) {
      $this->logger()->error("Keyring for user ID {$user_id} not found.");
      return;
    }

    $keyring_entity = $keyring['keyring_entity'] ?? NULL;
    if (empty($keyring_entity) || !$keyring_entity->hasField('meta')) {
      $this->logger()->warning("Keyring entity for user ID {$user_id} has no meta field.");
      return;
    }

    $meta_values = $keyring_entity->get('meta')->getValue();
    $meta_item = $meta_values[0] ?? [];
    $current_update_jobs = $meta_item['update_jobs'] ?? [];

    if (!is_array($current_update_jobs)) {
      $current_update_jobs = (array) $current_update_jobs;
    }

    // Avoid adding duplicates (string comparison to be robust).
    foreach ($current_update_jobs as $existing) {
      if ((string) $existing === (string) $proc_id) {
        $this->logger()->info("Proc ID {$proc_id} is already registered as an update job for user {$user_id}.");
        return;
      }
    }

    $current_update_jobs[] = $proc_id;
    $meta_item['update_jobs'] = $current_update_jobs;
    $meta_values[0] = $meta_item;

    try {
      $keyring_entity->set('meta', $meta_values);
      $keyring_entity->save();
      $this->logger()->success("Added proc ID {$proc_id} to update jobs for user ID {$user_id}.");
    }
    catch (EntityStorageException|\Exception $e) {
      $this->logger()->error($e->getMessage());
    }
  }

  /**
   * Remove all occurrences of given update job.
   *
   * @command proc:remove-all-jobs
   * @aliases raj
   * @usage proc:remove-all-jobs
   *     Remove all occurrences of given update job
   */
  #[CLI\Command(name: 'proc:remove-all-jobs', aliases: ['raj'])]
  #[CLI\Usage(name: 'raj', description: 'Remove all occurrences of given update job.')]
  public function removeUpdateJobForAllKeys(array $options = ['proc_id' => NULL]): void {
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($proc_id)) {
      $this->logger()->error('No proc_id provided.');
      return;
    }

    $modified = 0;
    $checked = 0;
    $keyring_entities = [];

    try {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $users = $user_storage ? $user_storage->loadMultiple() : [];
    }
    catch (InvalidPluginDefinitionException|PluginNotFoundException $e) {
      $this->logger()->error('Unable to load user storage to iterate keys.');
      return;
    }

    foreach ($users as $user) {
      $uid = $user->id();
      try {
        $result = $this->keyManager->getKeys($uid, 'user_id');
      }
      catch (InvalidPluginDefinitionException|PluginNotFoundException $e) {
        // User has no keyring or lookup failed; skip.
        continue;
      }
      if (!empty($result['keyring_entity'])) {
        $keyring_entities[] = $result['keyring_entity'];
      }
    }

    if (empty($keyring_entities)) {
      $this->logger()->info('No keyring entities found to process.');
      return;
    }

    foreach ($keyring_entities as $keyring_entity) {
      $checked++;
      if (empty($keyring_entity) || !$keyring_entity->hasField('meta')) {
        continue;
      }

      $meta_values = $keyring_entity->get('meta')->getValue();
      $meta_item = $meta_values[0] ?? [];
      $update_jobs = $meta_item['update_jobs'] ?? [];

      if (empty($update_jobs) || !is_array($update_jobs)) {
        continue;
      }

      // Remove all occurrences of the proc_id (string-aware).
      $filtered = array_values(array_filter($update_jobs, function ($v) use ($proc_id) {
        return (string) $v !== (string) $proc_id;
      }));

      // If unchanged, continue.
      if (count($filtered) === count($update_jobs)) {
        continue;
      }

      // Persist the modified update_jobs while preserving other meta fields.
      $meta_item['update_jobs'] = $filtered;
      $meta_values[0] = $meta_item;

      try {
        $keyring_entity->set('meta', $meta_values);
        $keyring_entity->save();
        $modified++;
      }
      catch (EntityStorageException|\Exception $e) {
        $this->logger()->error("Failed to update a keyring: " . $e->getMessage());
      }
    }

    $this->logger()->success("Processed {$checked} keyring(s). Removed proc ID {$proc_id} from {$modified} keyring(s).");
  }

}
