/**
 * Get CSRF token.
 * @returns {Promise<string>}
 */

/**
 * CSRF token cache with TTL, single inflight fetch, explicit invalidate,
 * and helper to attach token to fetch requests and retry once on 403/419.
 */
const CSRF_TTL_MS = 10 * 60 * 1000; // 10 minutes

let _cachedToken = null;
let _cachedAt = 0;
let _inflightPromise = null;

function _tokenUrl() {
  return `${window.location.origin + drupalSettings.path.baseUrl}session/token`;
}

export async function getCsrfToken() {
  const now = Date.now();
  if (_cachedToken && (now - _cachedAt) < CSRF_TTL_MS) {
    return _cachedToken;
  }

  if (_inflightPromise) {
    return _inflightPromise;
  }

  _inflightPromise = fetch(_tokenUrl(), {
    method: 'GET',
    credentials: 'same-origin'
  })
    .then((res) => {
      if (!res.ok) {
        throw new Error('Failed to fetch CSRF token');
      }
      return res.text();
    })
    .then((token) => {
      _cachedToken = token;
      _cachedAt = Date.now();
      return token;
    })
    .finally(() => {
      _inflightPromise = null;
    });

  return _inflightPromise;
}

export function invalidateCsrfToken() {
  _cachedToken = null;
  _cachedAt = 0;
  _inflightPromise = null;
}

/**
 * Helper to perform fetch with CSRF token in header.
 * If the response looks like a CSRF failure (403 or 419), it will invalidate,
 * refresh the token once, and retry the request.
 */
export async function fetchWithCsrf(input, init = {}) {
  const tryRequest = async (token) => {
    const headers = new Headers(init.headers || {});
    headers.set('X-CSRF-Token', token);
    headers.set('Accept', headers.get('Accept') || 'application/json');
    const opts = Object.assign({}, init, { headers, credentials: 'same-origin' });
    return fetch(input, opts);
  };

  const token = await getCsrfToken();
  let res = await tryRequest(token);

  if (res.status === 403 || res.status === 419) {
    // Token likely invalid/expired — invalidate and retry once
    invalidateCsrfToken();
    const freshToken = await getCsrfToken();
    res = await tryRequest(freshToken);
  }

  return res;
}
