<?php

namespace Drupal\proc\Drush\Commands;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\proc\Traits\ProcCsvTrait;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\proc\ProcKeyManager;

/**
 * Commands for managing protected content.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 */
final class ProcCommands extends DrushCommands implements ProcCommandsInterface {

  use ProcCsvTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The key manager.
   *
   * @var \Drupal\proc\ProcKeyManager
   */
  protected ProcKeyManager $keyManager;

  /**
   * Constructs a ProcCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\proc\ProcKeyManager $keyManager
   *   The proc key manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ProcKeyManager $keyManager,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->keyManager = $keyManager;
    parent::__construct();
  }

  /**
   * Create a new instance of this command.
   */
  public static function create(ContainerInterface $container): ProcCommands {
    return new ProcCommands(
      $container->get('entity_type.manager'),
      $container->get('proc.key_manager')
    );
  }

  /**
   * Remove contents of wished set of recipients field in a given proc.
   *
   * @command proc:remove-wished
   * @aliases prw
   * @usage proc:remove-wished --proc_id 123
   *     Remove contents of wished set of recipients field in a given proc.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  #[CLI\Command(name: 'proc:remove-wished', aliases: ['prw'])]
  #[CLI\Usage(name: 'proc:remove-wished', description: 'Remove contents of wished set of recipients field in a given proc cipher text entity, if any.')]
  public function removeWishedRecipients(array $options = ['proc_id' => NULL]): void {
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($proc_id)) {
      $this->logger()->error('No proc_id provided.');
      return;
    }

    $entity = NULL;

    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if ($storage) {
        $entity = $storage->load($proc_id);
      }
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      // Try next candidate type.
      $this->logger()->error("Proc entity with ID {$proc_id} not found.");
      return;
    }
    if (!$entity->hasField('field_wished_recipients_set')) {
      $this->logger()->warning("Entity {$proc_id} does not have field `field_wished_recipients_set`.");
      return;
    }
    if (empty($entity->get('field_wished_recipients_set')->getValue())) {
      $this->logger()->error("No wished recipients to remove for proc {$proc_id}.");
      return;
    }
    // Reset the field to an empty state.
    $entity->set('field_wished_recipients_set', []);
    try {
      $entity->save();
      $this->logger()->success("Reset field_wished_recipients_set for proc {$proc_id}.");
    }
    catch (EntityStorageException $e) {
      $this->logger()->error($e->getMessage());
    }
  }

  /**
   * Get update jobs from given key ID.
   *
   * @command proc:get-update-jobs
   * @aliases guj
   * @usage proc:get-update-jobs --user_id 123
   *     Get update jobs from given key ID.
   */
  #[CLI\Command(name: 'proc:get-update-jobs', aliases: ['guj'])]
  #[CLI\Usage(name: 'guj', description: 'Get update jobs, if any, from given user ID.')]
  public function getUpdateJobsFromUserId(array $options = ['user_id' => NULL]): void {
    $user_id = $options['user_id'] ?? NULL;

    if (empty($user_id)) {
      $this->logger()->error('No user ID provided.');
      return;
    }

    try {
      $keyring = $this->keyManager->getKeys($user_id, 'user_id');
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error("Keyring for user ID {$user_id} not found.");
      return;
    }
    $update_jobs = $keyring['keyring_entity']->get('meta')->getValue()[0]['update_jobs'] ?? '';
    if (empty($update_jobs)) {
      $this->logger()->info('No update jobs found.');
      return;
    }
    sort($update_jobs);
    $update_jobs_string = implode("\n", $update_jobs);
    $this->logger()->success("Update job(s) for user ID {$user_id}: \n {$update_jobs_string}");

  }

  /**
   * Get update workers from given proc ID.
   *
   * @command proc:get-update-workers
   * @aliases guw
   * @usage proc:get-update-jobs --proc_id 123
   *     Get update workers from given proc ID.
   */
  #[CLI\Command(name: 'proc:get-update-workers', aliases: ['guw'])]
  #[CLI\Usage(name: 'guw', description: 'Get update workers, if any, from given proc ID.')]
  public function getUpdateWorkers(array $options = ['proc_id' => NULL]): void {
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($proc_id)) {
      $this->logger()->error('No proc ID provided.');
      return;
    }

    $entity = NULL;

    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if ($storage) {
        $entity = $storage->load($proc_id);
      }
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      // Try next candidate type.
      $this->logger()->error("Proc entity with ID {$proc_id} not found.");
      return;
    }
    // Check among the recipients if any is marked as update worker, by having
    // in the update_jobs list in the metadata of the keyring the proc ID.
    $recipients = $entity->get('field_recipients_set')->getValue();
    $update_workers = [];
    foreach ($recipients as $recipient) {
      $recipient_id = $recipient['target_id'] ?? NULL;
      if (empty($recipient_id)) {
        continue;
      }
      try {
        $keyring = $this->keyManager->getKeys($recipient_id, 'user_id');
      }
      catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
        $this->logger()
          ->error("Keyring for recipient ID {$recipient_id} not found.");
        continue;
      }
      $update_jobs = $keyring['keyring_entity']->get('meta')
        ->getValue()[0]['update_jobs'] ?? [];
      if (in_array($proc_id, $update_jobs, TRUE)) {
        $update_workers[] = $recipient_id;
      }
    }
    if (empty($update_workers)) {
      $this->logger()->info('No update workers found.');
      return;
    }
    // Sort the update workers list:
    sort($update_workers);

    $update_workers_str = implode("\n", $update_workers);
    $this->logger()
      ->success("Update workers for proc ID {$proc_id}: {$update_workers_str}");
  }

  /**
   * Remove given recipient from given proc by their IDs.
   *
   * @command proc:remove-recipient
   * @aliases prr
   * @usage proc:remove-recipient --proc_id 123 --user_id 456
   *     Remove given recipient from given proc by their IDs.
   * @SuppressWarnings(PHPMD.CyclomaticComplexity)
   * @SuppressWarnings(PHPMD.NPathComplexity)
   */
  #[CLI\Command(name: 'proc:remove-recipient', aliases: ['prr'])]
  #[CLI\Usage(name: 'prr', description: 'Remove given recipient from given proc by their IDs.')]
  public function removeRecipient(array $options = ['proc_id' => NULL, 'user_id' => NULL]): void {
    $proc_id = $options['proc_id'] ?? NULL;
    $user_id = $options['user_id'] ?? NULL;

    if (empty($proc_id) || empty($user_id)) {
      $this->logger()->error('Both proc_id and user_id must be provided.');
      return;
    }

    // Load proc entity.
    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if (!$storage) {
        $this->logger()->error("Proc storage not available.");
        return;
      }
      $proc = $storage->load($proc_id);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error("Proc entity with ID {$proc_id} not found.");
      return;
    }

    if (!$proc) {
      $this->logger()->error("Proc entity with ID {$proc_id} not found.");
      return;
    }

    if (!$proc->hasField('field_recipients_set')) {
      $this->logger()->warning("Entity {$proc_id} does not have field `field_recipients_set`.");
      return;
    }

    $recipients = $proc->get('field_recipients_set')->getValue() ?? [];
    if (empty($recipients)) {
      $this->logger()->info("No recipients present on proc {$proc_id}.");
    }

    // Filter out the recipient to remove.
    $new_recipients = array_values(array_filter($recipients, function ($item) use ($user_id) {
      $target = $item['target_id'] ?? NULL;
      return $target === NULL || (string) $target !== (string) $user_id;
    }));

    if (count($new_recipients) === count($recipients)) {
      $this->logger()->info("Recipient {$user_id} not found on proc {$proc_id}.");
      return;
    }
    try {
      $proc->set('field_recipients_set', $new_recipients);
      $proc->save();
      $this->logger()->success("Removed recipient {$user_id} from proc {$proc_id}.");
    }
    catch (EntityStorageException $e) {
      $this->logger()->error($e->getMessage());
    }
  }

  /**
   * Remove update jobs, if any, from given user ID. If proc_id is given,
   * remove only that occurrence.
   *
   * @command proc:remove-update-jobs
   * @aliases ruj
   * @usage proc:remove-update-jobs --user_id 123 --proc_id 456
   *     Remove update jobs, if any, from given user ID.
   */
  #[CLI\Command(name: 'proc:remove-update-jobs', aliases: ['ruj'])]
  #[CLI\Usage(name: 'guj', description: 'Remove update jobs, if any, from given user ID. If proc_id is given, remove only that occurrence.')]
  public function removeUpdateJobsFromUserId(array $options = ['user_id' => NULL, 'proc_id' => NULL]): void {
    $user_id = $options['user_id'] ?? NULL;
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($user_id)) {
      $this->logger()->error('No user ID provided.');
      return;
    }

    try {
      $keyring = $this->keyManager->getKeys($user_id, 'user_id');
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error("Keyring for user ID {$user_id} not found.");
      return;
    }

    $keyring_entity = $keyring['keyring_entity'] ?? NULL;
    if (empty($keyring_entity) || !$keyring_entity->hasField('meta')) {
      $this->logger()->warning("Keyring entity for user ID {$user_id} has no meta field.");
      return;
    }

    $meta_values = $keyring_entity->get('meta')->getValue();
    $meta_item = $meta_values[0] ?? [];
    $current_update_jobs = $meta_item['update_jobs'] ?? [];

    if (empty($current_update_jobs)) {
      $this->logger()->info("No update jobs to remove for user ID {$user_id}.");
      return;
    }

    // If a specific proc_id was provided, remove only that occurrence.
    if (!empty($proc_id)) {
      $filtered = array_values(array_filter($current_update_jobs, function ($value) use ($proc_id) {
        return (string) $value !== (string) $proc_id;
      }));

      if (count($filtered) === count($current_update_jobs)) {
        $this->logger()->info("Proc ID {$proc_id} not found among update jobs for user ID {$user_id}.");
        return;
      }

      $meta_item['update_jobs'] = $filtered;
      $meta_values[0] = $meta_item;

      try {
        $keyring_entity->set('meta', $meta_values);
        $keyring_entity->save();
        $this->logger()->success("Removed proc ID {$proc_id} from update jobs for user ID {$user_id}.");
      }
      catch (EntityStorageException | \Exception $e) {
        $this->logger()->error($e->getMessage());
      }

      return;
    }

    $meta_item['update_jobs'] = [];
    $meta_values[0] = $meta_item;

    try {
      $keyring_entity->set('meta', $meta_values);
      $keyring_entity->save();
      $this->logger()->success("Removed update jobs for user ID {$user_id}.");
    }
    catch (EntityStorageException | \Exception $e) {
      $this->logger()->error($e->getMessage());
    }
  }

  /**
   * Add update job for user by given user and proc IDs.
   *
   * @command proc:add-update-job
   * @aliases auj
   * @usage proc:add-update-job --user_id 123 --proc_id 456
   *     Add update job for user by given user and proc IDs
   * @SuppressWarnings(PHPMD.CyclomaticComplexity)
   * @SuppressWarnings(PHPMD.NPathComplexity)
   */
  #[CLI\Command(name: 'proc:add-update-job', aliases: ['auj'])]
  #[CLI\Usage(name: 'auj', description: 'Add update job for user by given user and proc IDs')]
  public function addUpdateJob(array $options = ['user_id' => NULL, 'proc_id' => NULL]): void {
    $user_id = $options['user_id'] ?? NULL;
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($user_id) || empty($proc_id)) {
      $this->logger()->error('Both user_id and proc_id must be provided.');
      return;
    }

    try {
      $keyring = $this->keyManager->getKeys($user_id, 'user_id');
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error("Keyring for user ID {$user_id} not found.");
      return;
    }

    $keyring_entity = $keyring['keyring_entity'] ?? NULL;
    if (empty($keyring_entity) || !$keyring_entity->hasField('meta')) {
      $this->logger()->warning("Keyring entity for user ID {$user_id} has no meta field.");
      return;
    }

    $meta_values = $keyring_entity->get('meta')->getValue();
    $meta_item = $meta_values[0] ?? [];
    $current_update_jobs = $meta_item['update_jobs'] ?? [];

    if (!is_array($current_update_jobs)) {
      $current_update_jobs = (array) $current_update_jobs;
    }

    // Avoid adding duplicates (string comparison to be robust).
    foreach ($current_update_jobs as $existing) {
      if ((string) $existing === (string) $proc_id) {
        $this->logger()->info("Proc ID {$proc_id} is already registered as an update job for user {$user_id}.");
        return;
      }
    }

    $current_update_jobs[] = $proc_id;
    $meta_item['update_jobs'] = $current_update_jobs;
    $meta_values[0] = $meta_item;

    try {
      $keyring_entity->set('meta', $meta_values);
      $keyring_entity->save();
      $this->logger()->success("Added proc ID {$proc_id} to update jobs for user ID {$user_id}.");
    }
    catch (EntityStorageException | \Exception $e) {
      $this->logger()->error($e->getMessage());
    }
  }

  /**
   * Remove all occurrences of given update job.
   *
   * @command proc:remove-all-jobs
   * @aliases raj
   * @usage proc:remove-all-jobs --proc_id 123
   *     Remove all occurrences of given update job
   * @SuppressWarnings(PHPMD.CyclomaticComplexity)
   * @SuppressWarnings(PHPMD.NPathComplexity)
   */
  #[CLI\Command(name: 'proc:remove-all-jobs', aliases: ['raj'])]
  #[CLI\Usage(name: 'raj', description: 'Remove all occurrences of given update job.')]
  public function removeUpdateJobForAllKeys(array $options = ['proc_id' => NULL]): void {
    $proc_id = $options['proc_id'] ?? NULL;

    if (empty($proc_id)) {
      $this->logger()->error('No proc_id provided.');
      return;
    }

    $modified = 0;
    $checked = 0;
    $keyring_entities = [];

    try {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $users = $user_storage ? $user_storage->loadMultiple() : [];
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error('Unable to load user storage to iterate keys.');
      return;
    }

    foreach ($users as $user) {
      $uid = $user->id();
      try {
        $result = $this->keyManager->getKeys($uid, 'user_id');
      }
      catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
        // User has no keyring or lookup failed; skip.
        continue;
      }
      if (!empty($result['keyring_entity'])) {
        $keyring_entities[] = $result['keyring_entity'];
      }
    }

    if (empty($keyring_entities)) {
      $this->logger()->info('No keyring entities found to process.');
      return;
    }

    foreach ($keyring_entities as $keyring_entity) {
      $checked++;
      if (empty($keyring_entity) || !$keyring_entity->hasField('meta')) {
        continue;
      }

      $meta_values = $keyring_entity->get('meta')->getValue();
      $meta_item = $meta_values[0] ?? [];
      $update_jobs = $meta_item['update_jobs'] ?? [];

      if (empty($update_jobs) || !is_array($update_jobs)) {
        continue;
      }

      // Remove all occurrences of the proc_id (string-aware).
      $filtered = array_values(array_filter($update_jobs, function ($value) use ($proc_id) {
        return (string) $value !== (string) $proc_id;
      }));

      // If unchanged, continue.
      if (count($filtered) === count($update_jobs)) {
        continue;
      }

      // Persist the modified update_jobs while preserving other meta fields.
      $meta_item['update_jobs'] = $filtered;
      $meta_values[0] = $meta_item;

      try {
        $keyring_entity->set('meta', $meta_values);
        $keyring_entity->save();
        $modified++;
      }
      catch (EntityStorageException | \Exception $e) {
        $this->logger()->error("Failed to update a keyring: " . $e->getMessage());
      }
    }

    $this->logger()->success("Processed {$checked} keyring(s). Removed proc ID {$proc_id} from {$modified} keyring(s).");
  }

  /**
   * Get IDs of procs where given user is a recipient.
   *
   * @command proc:get-procs-user-is-recipient
   * @aliases gpur
   * @usage proc:get-procs-user-is-recipient --user_id 123
   *     Get IDs of procs where given user is a recipient.
   */
  #[CLI\Command(name: 'proc:get-procs-user-is-recipient', aliases: ['gpur'])]
  #[CLI\Usage(name: 'gpur', description: 'Get IDs of procs where given user is a recipient.')]
  public function getProcsUserIsRecipient(array $options = ['user_id' => NULL]): void {
    $user_id = $options['user_id'] ?? NULL;

    if (empty($user_id)) {
      $this->logger()->error('No user ID provided.');
      return;
    }

    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if (!$storage) {
        $this->logger()->error("Proc storage not available.");
        return;
      }
      $query = $storage->getQuery();
      $query->condition('field_recipients_set.target_id', $user_id)->accessCheck(TRUE);
      $proc_ids = $query->execute();
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error("Unable to query proc entities.");
      return;
    }

    if (empty($proc_ids)) {
      $this->logger()->info("User ID {$user_id} is not a recipient of any proc.");
      return;
    }

    sort($proc_ids);
    $proc_ids_str = implode("\n", $proc_ids);
    $this->logger()->success("User ID {$user_id} is a recipient of the following proc IDs:\n{$proc_ids_str}");

  }

  /**
   * Convert a CSV list of user IDs into a CSV list of the labels of their keys.
   *
   * @command proc:user-keys-labels
   * @aliases pql
   * @usage proc:user-keys-labels --user_ids 1,2,3
   *   Return a CSV list of key labels for the given CSV user IDs.
   */
  #[CLI\Command(name: 'proc:user-keys-labels', aliases: ['pql'])]
  #[CLI\Usage(name: 'proc:user-keys-labels', description: 'Given a CSV list of user IDs, return a CSV list of the labels of their keyrings.')]
  public function userIdsToKeysLabels(array $options = ['user_ids' => NULL]): void {
    $user_ids_csv = $options['user_ids'] ?? NULL;
    if (empty($user_ids_csv)) {
      $this->logger()->error('No user IDs CSV provided.');
      return;
    }
    $ids = $this->getCsvArgument($user_ids_csv);
    if (empty($ids)) {
      $this->logger()->error('No valid user IDs parsed from input.');
      return;
    }

    $labels = [];
    foreach ($ids as $id) {
      try {
        $keyring = $this->keyManager->getKeys($id);
        if (!$keyring) {
          $this->logger()->warning("Keyring for user ID {$id} not found.");
          continue;
        }
        $label = (string) $this->keyManager->getKeys($id)['label'] ?? '';
        if (empty($label)) {
          $this->logger()->warning("Keyring for user ID {$id} has no label.");
          continue;
        }
        $labels[] = $label;
      }
      catch (InvalidPluginDefinitionException|PluginNotFoundException $e) {
        $this->logger()->warning("Keyring for user ID {$id} not found.");
      }
    }

    $labels_csv = implode(',', $labels);
    $this->logger()->success($labels_csv);
  }

  /**
   * Given a CSV list of proc IDs, a host entity type and field machine name,
   * return a CSV list of the referenced host entity IDs.
   *
   * @command proc:procs-host-ids
   * @aliases phis
   * @usage proc:procs-host-ids --proc_ids 10,11,12 --host_field field_host --host_entity_type node
   *   Return a CSV list of host entity IDs referencing the given proc IDs.
   */
  #[CLI\Command(name: 'proc:procs-host-ids', aliases: ['phis'])]
  #[CLI\Usage(name: 'proc:procs-host-ids', description: 'Given a CSV list of proc IDs, a host entity type and field machine name, return a list of host entity IDs.')]
  public function procIdsToHostIds(array $options = ['proc_ids' => NULL, 'host_fields' => NULL, 'host_entity_type' => NULL]): void {
    $proc_ids_csv = $options['proc_ids'] ?? NULL;
    $host_fields = str_getcsv($options['host_fields']) ?? NULL;
    $host_entity_type = $options['host_entity_type'] ?? NULL;

    if (empty($proc_ids_csv) || empty($host_fields) || empty($host_entity_type)) {
      $this->logger()->error('proc_ids, host_fields and host_entity_type must be provided.');
      return;
    }

    $ids = $this->getCsvArgument($proc_ids_csv);
    if (empty($ids)) {
      $this->logger()->error('No valid proc IDs parsed from input.');
      return;
    }

    // Query host entities referencing the given proc IDs.
    try {
      $storage = $this->entityTypeManager->getStorage($host_entity_type);
      if (!$storage) {
        $this->logger()->error("Storage for entity type {$host_entity_type} not available.");
        return;
      }
      $host_entity_ids = [];
      foreach ($host_fields as $host_field) {
        $query = $storage->getQuery();
        $query->condition($host_field, $ids, 'IN')->accessCheck(TRUE);
        $result_ids = $query->execute();
        if (!empty($result_ids)) {
          $host_entity_ids = array_merge($host_entity_ids, $result_ids);
        }
      }
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error("Unable to query entities of type {$host_entity_type}.");
      return;
    }

    if (empty($host_entity_ids)) {
      $this->logger()->info('No host entity IDs found for the provided procs.');
      return;
    }
    // Preserve order but remove duplicates.
    $host_ids = array_values(array_unique($host_entity_ids));
    sort($host_ids);
    $this->logger()->success(implode(',', $host_ids));
  }

  /**
   * Given a CSV list of proc IDs, return a CSV list of their labels.
   *
   * @command proc:procs-labels
   * @aliases plb
   * @usage proc:procs-labels --proc_ids 10,11,12
   *   Return a CSV list of labels for the given proc IDs.
   */
  #[CLI\Command(name: 'proc:procs-labels', aliases: ['plb'])]
  #[CLI\Usage(name: 'proc:procs-labels', description: 'Given a CSV list of proc IDs, return a CSV list of their labels.')]
  public function procIdsToLabels(array $options = ['proc_ids' => NULL]): void {
    $proc_ids_csv = $options['proc_ids'] ?? NULL;
    if (empty($proc_ids_csv)) {
      $this->logger()->error('No proc IDs CSV provided.');
      return;
    }

    $ids = $this->getCsvArgument($proc_ids_csv);
    if (empty($ids)) {
      $this->logger()->error('No valid proc IDs parsed from input.');
      return;
    }

    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if (!$storage) {
        $this->logger()->error('Proc storage not available.');
        return;
      }
    }
    catch (\Exception $e) {
      $this->logger()->error('Unable to access proc storage: ' . $e->getMessage());
      return;
    }

    $labels = [];
    foreach ($ids as $id) {
      try {
        $entity = $storage->load($id);
      }
      catch (\Exception $e) {
        $this->logger()->warning("Failed loading proc {$id}: " . $e->getMessage());
        $labels[] = '';
        continue;
      }

      if (empty($entity)) {
        $this->logger()->warning("Proc entity with ID {$id} not found.");
        $labels[] = '';
        continue;
      }

      // Use the entity label; fall back to an empty string if not available.
      $label = method_exists($entity, 'label') ? (string) $entity->label() : (string) $entity->id();
      $labels[] = $label;
    }

    $labels_csv = implode(',', $labels);
    $this->logger()->success($labels_csv);
  }

  /**
   * Find cipher proc entities missing required meta fields.
   *
   * Checks the 'meta' field of all proc entities of type 'cipher' and returns
   * the proc IDs where any of the following keys are missing or empty:
   * - generation_timestamp
   * - source_file_name
   * - source_file_size
   *
   * New options:
   *   --start_limit <id>
   *   --end_limit <id>
   *
   * @command proc:find-cipher-missing-meta
   * @aliases pcm
   * @usage proc:find-cipher-missing-meta
   *   Return a CSV list of proc IDs of cipher entities missing key meta entries.
   */
  #[CLI\Command(name: 'proc:find-cipher-missing-meta', aliases: ['pcm'])]
  #[CLI\Usage(name: 'proc:find-cipher-missing-meta', description: 'Return proc IDs of cipher entities missing key meta entries.')]
  public function findCipherMissingMeta(array $options = ['start_limit' => NULL, 'end_limit' => NULL]): void {
    $required_keys = [
      'generation_timestamp',
      'source_file_name',
      'source_file_size',
    ];

    // Accept optional range limits.
    $start_limit = $options['start_limit'] ?? NULL;
    $end_limit = $options['end_limit'] ?? NULL;

    if ($start_limit !== NULL && $start_limit !== '' && !is_numeric($start_limit)) {
      $this->logger()->error('start_limit must be an integer.');
      return;
    }
    if ($end_limit !== NULL && $end_limit !== '' && !is_numeric($end_limit)) {
      $this->logger()->error('end_limit must be an integer.');
      return;
    }

    try {
      $storage = $this->entityTypeManager->getStorage('proc');
      if (!$storage) {
        $this->logger()->error('Proc storage not available.');
        return;
      }
      $query = $storage->getQuery();
      $query->condition('type', 'cipher');

      // Apply optional ID range conditions.
      if ($start_limit !== NULL && $start_limit !== '') {
        $query->condition('id', (int) $start_limit, '>=');
      }
      if ($end_limit !== NULL && $end_limit !== '') {
        $query->condition('id', (int) $end_limit, '<=');
      }

      $query->accessCheck(TRUE);
      $proc_ids = $query->execute();
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger()->error('Unable to query proc entities of type cipher.');
      return;
    }
    catch (\Exception $e) {
      $this->logger()->error('Unexpected error while querying proc entities: ' . $e->getMessage());
      return;
    }

    if (empty($proc_ids)) {
      $this->logger()->info('No cipher proc entities found.');
      return;
    }

    $offending = [];
    $batch_ids = array_values($proc_ids);

    // Load in one call; adjust if memory becomes an issue.
    $entities = $storage->loadMultiple($batch_ids);

    foreach ($entities as $entity) {
      if (empty($entity)) {
        continue;
      }
      // If no meta field or no values, treat as missing.
      if (!$entity->hasField('meta')) {
        $offending[] = (string) $entity->id();
        continue;
      }

      $meta_values = $entity->get('meta')->getValue();

      $meta_item = $meta_values[0] ?? [];
      $missing_any = FALSE;

      foreach ($required_keys as $key) {
        // Consider missing or empty as problematic. Use empty() to be permissive.
        if (!array_key_exists($key, $meta_item) || empty($meta_item[$key])) {
          $missing_any = TRUE;
          break;
        }
      }

      if ($missing_any) {
        $offending[] = (string) $entity->id();
      }
    }

    if (empty($offending)) {
      $this->logger()->info('All cipher entities have the required meta entries.');
      return;
    }

    // Return CSV of offending proc IDs.
    $this->logger()->success(implode(',', $offending));
  }

}
