/**
 * @file
 * Process cache.
 */

/**
 * Process cache.
 *
 * @param {string[]} procURLs
 *   Process URLs.
 * @param {number} cipherIndex
 *   Cipher index.
 *
 * @returns {Promise<Response>}
 */
export async function processCache(procURLs, cipherIndex) {
  const cacheDetails = {
    cache: 'caches' in self ? caches.open('proc') : null,
    response: null,
    cipher: null,
    cachedCiphers: null,
  };

  /**
   * Extract cipher ID from URL.
   *
   * @param {string} url
   *   The URL to extract the cipher ID from.
   *
   * @returns {string|null}
   *   The extracted cipher ID or null if not found.
   */
  function extractCipherId(url) {
    const match = url && url.match(/proc\/getcipher\/(\d+)\//);
    return match ? match[1] : null;
  }

  if (cacheDetails.cache) {
    cacheDetails.cachedCiphers = await cacheDetails.cache.then(async function (cache) {
      cacheDetails.response = await cache.match(procURLs[cipherIndex]);
      const id = extractCipherId(procURLs[cipherIndex]);
      if (!cacheDetails.response) {
        console.info(id ? `Adding cipher to cache (id: ${id})` : 'Adding cipher to cache');
        const cipherResponse = await fetch(procURLs[cipherIndex]);
        cacheDetails.cipher = cipherResponse.clone();
        await cache.put(procURLs[cipherIndex], cipherResponse);
        return cacheDetails.cipher;
      }
      console.info(id ? `Reusing cipher from cache (id: ${id})` : 'Reusing cipher from cache');
      return cacheDetails.response;
    });
  }
  return cacheDetails.cachedCiphers;
}
