<?php

namespace Drupal\product_manager_tool\Form;

use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductType;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Block\BlockManagerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\TempStore\SharedTempStoreFactory;
use Drupal\Core\Url;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay;
use Drupal\layout_builder\LayoutTempstoreRepositoryInterface;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionComponent;
use Drupal\product_manager_tool\Service\FieldTableBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Layout Template Form for managing product layouts with translation support.
 */
class LayoutTemplateForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The block manager.
   *
   * @var \Drupal\Core\Block\BlockManagerInterface
   */
  protected $blockManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The layout tempstore repository.
   *
   * @var \Drupal\layout_builder\LayoutTempstoreRepositoryInterface
   */
  protected $layoutTempstoreRepository;

  /**
   * The shared tempstore factory.
   *
   * @var \Drupal\Core\TempStore\SharedTempStoreFactory
   */
  protected $sharedTempStoreFactory;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuid;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The field table builder service.
   *
   * @var \Drupal\product_manager_tool\Service\FieldTableBuilder
   */
  protected $fieldTableBuilder;

  /**
   * Constructs a LayoutTemplateForm object.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    BlockManagerInterface $block_manager,
    ModuleHandlerInterface $module_handler,
    LayoutTempstoreRepositoryInterface $layout_tempstore_repository,
    SharedTempStoreFactory $shared_tempstore_factory,
    LanguageManagerInterface $language_manager,
    DateFormatterInterface $date_formatter,
    UuidInterface $uuid,
    LoggerChannelFactoryInterface $logger_factory,
    FieldTableBuilder $field_table_builder,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->blockManager = $block_manager;
    $this->moduleHandler = $module_handler;
    $this->layoutTempstoreRepository = $layout_tempstore_repository;
    $this->sharedTempStoreFactory = $shared_tempstore_factory;
    $this->languageManager = $language_manager;
    $this->dateFormatter = $date_formatter;
    $this->uuid = $uuid;
    $this->loggerFactory = $logger_factory;
    $this->fieldTableBuilder = $field_table_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.block'),
      $container->get('module_handler'),
      $container->get('layout_builder.tempstore_repository'),
      $container->get('tempstore.shared'),
      $container->get('language_manager'),
      $container->get('date.formatter'),
      $container->get('uuid'),
      $container->get('logger.factory'),
      $container->get('product_manager_tool.field_table_builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'product_layout_template_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    if (!$this->moduleHandler->moduleExists('layout_builder')) {
      $form['error'] = [
        '#markup' => '<div class="messages messages--error">' . $this->t('Layout Builder module is not enabled.') . '</div>',
      ];
      return $form;
    }

    $layout_enabled_types = $this->getLayoutEnabledProductTypes();

    if (empty($layout_enabled_types)) {
      $form['no_types'] = [
        '#markup' => '<div class="messages messages--warning">' . $this->t('No product types have Layout Builder enabled.') . '</div>',
      ];
      return $form;
    }

    $step = $form_state->get('step') ?: 'select';

    if ($step === 'confirm') {
      return $this->buildConfirmationForm($form, $form_state);
    }

    $form['description'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['messages', 'messages--info'],
      ],
      'text' => [
        '#markup' => $this->t('
      <div>
        <strong>Layout Template Manager</strong><br><br>
        <strong>Step 1:</strong> Select or create a template layout.<br>
        <strong>Step 2:</strong> Choose the language and products to apply the template layout.<br><br>
        <em>You can use the Layout Builder visual interface to design your template.</em>
      </div>
    '),
      ],
    ];

    $default_product_type = $form_state->getValue('product_type');

    if (!$default_product_type && $form_state->get('form_values') !== NULL) {
      $stored_values = $form_state->get('form_values');
      if (!empty($stored_values['product_type'])) {
        $default_product_type = $stored_values['product_type'];
      }
    }

    if (!$default_product_type) {
      $default_product_type = array_key_first($layout_enabled_types);
    }

    $form['product_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Product Type'),
      '#options' => $layout_enabled_types,
      '#default_value' => $default_product_type,
      '#ajax' => [
        'callback' => '::updateProductTypeAndTemplate',
        'wrapper' => 'full-form-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
      '#required' => TRUE,
    ];

    $form['template_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Step 1: Choose Template Layout'),
      '#open' => TRUE,
      '#prefix' => '<div id="template-section-wrapper">',
      '#suffix' => '</div>',
    ];

    $user_input = $form_state->getUserInput();

    $form['template_section']['template_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Template Source'),
      '#options' => [
        'existing' => $this->t('Use existing product as template'),
        'create_new' => $this->t('Create or edit template using Layout Builder'),
      ],
      '#default_value' => ($user_input['template_mode'] ?? NULL)
        ?: ($form_state->getValue('template_mode')
          ?? ($form_state->get('form_values')['template_mode'] ?? 'create_new')),
      '#ajax' => [
        'callback' => '::updateTemplateOptions',
        'wrapper' => 'template-options-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
    ];

    $form['template_section']['template_options'] = [
      '#type' => 'container',
      '#prefix' => '<div id="template-options-wrapper">',
      '#suffix' => '</div>',
    ];

    $template_mode = $user_input['template_mode'] ?? $form_state->getValue('template_mode');

    if (!$template_mode) {
      if ($form_state->get('form_values') !== NULL) {
        $stored_values = $form_state->get('form_values');
        $template_mode = $stored_values['template_mode'] ?? NULL;
      }

      if (!$template_mode) {
        $template_mode = $form['template_section']['template_mode']['#default_value'];
      }
    }

    $selected_type = $default_product_type;

    $stored_template_id = NULL;
    $stored_template_type = NULL;

    if ($template_mode === 'create_new') {
      // Use type-specific key based on selected product type.
      $stored_template_id = $selected_type ? $form_state->get('template_product_id_' . $selected_type) : NULL;
      $stored_template_type = $form_state->get('template_product_type');
    }

    $show_template_for_current_type = ($stored_template_id && $stored_template_type == $selected_type);

    if ($template_mode === 'existing') {
      $this->buildExistingTemplateOptions($form['template_section']['template_options'], $form_state, $selected_type);
    }
    else {
      $this->buildNewTemplateOptions(
        $form['template_section']['template_options'],
        $form_state,
        $selected_type,
        $show_template_for_current_type ? $stored_template_id : NULL
          );
    }

    $form['product_container'] = [
      '#type' => 'container',
      '#prefix' => '<div id="product-list-wrapper">',
      '#suffix' => '</div>',
    ];

    if ($selected_type) {
      $this->buildProductSection($form['product_container'], $form_state, $selected_type);
    }

    return $form;
  }

  /**
   * Build existing template options.
   */
  private function buildExistingTemplateOptions(&$form, FormStateInterface $form_state, $product_type) {
    $template_products = $this->getExistingTemplateProducts($product_type);

    if (empty($template_products)) {
      $form['no_templates_container'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['template-empty-state']],
      ];

      $form['no_templates_container']['icon'] = [
        '#type' => 'inline_template',
        '#template' => '<div class="empty-state-icon">
      <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24"
      fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"></path>
        <polyline points="3.27 6.96 12 12.01 20.73 6.96"></polyline>
        <line x1="12" y1="22.08" x2="12" y2="12"></line>
      </svg>
    </div>',
      ];

      $form['no_templates_container']['message'] = [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#value' => $this->t('No templates found'),
        '#attributes' => ['class' => ['empty-state-message']],
      ];

      $form['no_templates_container']['description'] = [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $this->t('No template products found for this product type. Please use "Create new template" option to create one first.'),
        '#attributes' => ['class' => ['empty-state-description']],
      ];

      return;
    }

    $form['existing_product'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Template Product'),
      '#options' => $template_products,
      '#empty_option' => $this->t('- Select a template -'),
      '#description' => $this->t('Choose a template product whose layout will be copied to selected products.'),
      '#required' => FALSE,
      '#ajax' => [
        'callback' => '::previewTemplateLayout',
        'wrapper' => 'template-preview-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
    ];

    $form['preview'] = [
      '#type' => 'container',
      '#prefix' => '<div id="template-preview-wrapper">',
      '#suffix' => '</div>',
    ];

    $selected_product_id = $form_state->getValue('existing_product');
    if ($selected_product_id) {
      $this->buildTemplatePreview($form['preview'], $selected_product_id);
    }
  }

  /**
   * Get existing template products.
   */
  private function getExistingTemplateProducts($product_type) {
    $options = [];

    $product_ids = $this->entityTypeManager->getStorage('commerce_product')
      ->getQuery()
      ->condition('type', $product_type)
      ->accessCheck(TRUE)
      ->sort('changed', 'DESC')
      ->execute();

    if (empty($product_ids)) {
      return $options;
    }

    $products = Product::loadMultiple($product_ids);
    foreach ($products as $product) {
      if (strpos($product->getTitle(), '[TEMPLATE]') === FALSE) {
        continue;
      }

      if (!$product->hasField('layout_builder__layout')) {
        continue;
      }

      $layout_data = $product->get('layout_builder__layout')->getValue();
      $has_layout = FALSE;
      $component_count = 0;

      if (!empty($layout_data)) {
        foreach ($layout_data as $item) {
          if (isset($item['section'])) {
            $section = $item['section'];
            $components = $section->getComponents();
            $component_count += count($components);
            if (!empty($components)) {
              $has_layout = TRUE;
            }
          }
        }
      }

      $status = $product->isPublished() ? 'Published' : 'Unpublished';
      $layout_status = $has_layout ? $component_count . ' components' : 'No layout';
      $options[$product->id()] = $product->getTitle() . ' (' . $status . ', ' . $layout_status . ')';
    }

    return $options;
  }

  /**
   * Build new template options.
   */
  private function buildNewTemplateOptions(&$form, FormStateInterface $form_state, $product_type, $template_product_id = NULL) {
    if ($template_product_id === NULL) {
      // Use type-specific key based on product type.
      $stored_template_id = $product_type ? $form_state->get('template_product_id_' . $product_type) : NULL;
      $stored_template_type = $form_state->get('template_product_type');
      if ($stored_template_id && $stored_template_type === $product_type) {
        $template_product_id = $stored_template_id;
      }
      else {
        // Check DB for existing templates.
        $existing_templates = $this->getExistingTemplateProducts($product_type);
        if (!empty($existing_templates)) {
          $template_product_id = array_key_first($existing_templates);

          // Store this ID in form state, so it persists across rebuilds.
          if ($product_type) {
            $form_state->set('template_product_id_' . $product_type, $template_product_id);
            $form_state->set('template_product_type', $product_type);
          }
        }
      }
    }

    if (!$template_product_id) {
      $form['no_template_container'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['template-empty-state']],
      ];

      $form['no_template_container']['message'] = [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#value' => $this->t('No template exists for this product type yet.'),
        '#attributes' => [
          'class' => ['empty-state-message'],
        ],
      ];

      $form['no_template_container']['description'] = [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $this->t('Create a template product to start building your product layouts.'),
        '#attributes' => [
          'class' => ['empty-state-description'],
        ],
      ];

      $form['no_template_container']['create_template_button'] = [
        '#type' => 'submit',
        '#value' => $this->t('Create Template Product'),
        '#submit' => ['::createTemplateProduct'],
        '#ajax' => [
          'callback' => '::ajaxCreateTemplate',
          'wrapper' => 'template-options-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
        '#limit_validation_errors' => [['product_type']],
        '#attributes' => [
          'class' => ['button', 'button--primary', 'button--large'],
        ],
      ];

      return;
    }

    $template_product = Product::load($template_product_id);

    if (!$template_product) {
      // Template was deleted - clear it and set template_product_id to NULL
      // This will make the method flow into the "no template exists" section above.
      if ($product_type) {
        $form_state->set('template_product_id_' . $product_type, NULL);
      }

      // Set template_product_id to NULL so we show "no template" UI.
      $template_product_id = NULL;

      // Call the method again with NULL to show "no template exists" UI.
      $this->buildNewTemplateOptions($form, $form_state, $product_type, NULL);
      return;
    }

    $form['template_card'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['template-card']],
    ];

    $form['template_card']['header'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['template-card__header']],
    ];

    $form['template_card']['header']['status'] = [
      '#markup' => '<div class="template-card__status">
      <span>' . $this->t('Template Active') . '</span>
    </div>',
    ];

    $form['template_card']['header']['info'] = [
      '#markup' => '<div class="template-card__info">
      <h3 class="template-card__title">' . $template_product->getTitle() . '</h3>
      <p class="template-card__meta">' . $this->t('Product ID: @id', ['@id' => $template_product_id]) . '</p>
    </div>',
    ];

    $form['template_card']['header']['refresh_button'] = [
      '#type' => 'inline_template',
      '#template' => '<div class="template-card__refresh" title="Refresh"
       onclick="jQuery(document).find(\'input[value=create_new]\').trigger(\'change\'); return false;">
    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" width="32px" height="32px">
      <path d="M 16 4 C 10.886719 4 6.617188 7.160156 4.875 11.625 L 6.71875 12.375 C 8.175781 8.640625 11.710938 6 16 6 C 19.242188 6 22.132813 7.589844 23.9375 10 L 20 10 L 20 12 L 27 12 L 27 5 L 25 5 L 25 8.09375 C 22.808594 5.582031 19.570313 4 16 4 Z M 25.28125 19.625 C 23.824219 23.359375 20.289063 26 16 26 C 12.722656 26 9.84375 24.386719 8.03125 22 L 12 22 L 12 20 L 5 20 L 5 27 L 7 27 L 7 23.90625 C 9.1875 26.386719 12.394531 28 16 28 C 21.113281 28 25.382813 24.839844 27.125 20.375 Z"/>
    </svg>
  </div>',
    ];

    $layout_url = Url::fromRoute('layout_builder.overrides.commerce_product.view', [
      'commerce_product' => $template_product_id,
    ]);

    $form['template_card']['body'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['template-card__body']],
    ];

    $form['template_card']['body']['edit_template'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit Template Layout'),
      '#url' => $layout_url,
      '#attributes' => [
        'class' => ['button', 'button--secondary', 'button--icon'],
        'target' => '_blank',
      ],
    ];

    $form['template_card']['body']['template_ready'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('I have finished editing the template layout'),
      '#description' => $this->t('Check this box after you have designed and saved your template layout.'),
      '#required' => TRUE,
      '#default_value' => $form_state->getValue('template_ready') ?? $form_state->get('form_values')['template_ready'] ?? FALSE,
      '#attributes' => ['class' => ['template-ready-checkbox']],
    ];

    $form['template_card']['footer'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['template-card__footer']],
    ];

    $form['template_card']['footer']['delete_template'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete Template & Start Over'),
      '#submit' => ['::deleteTemplateProduct'],
      '#ajax' => [
        'callback' => '::ajaxDeleteTemplate',
        'wrapper' => 'template-options-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
      '#limit_validation_errors' => [],
      '#attributes' => [
        'class' => ['button', 'button--danger', 'button--small', 'button--icon'],
      ],
    ];

    $this->buildTemplatePreview($form, $template_product_id);
  }

  /**
   * AJAX callback for template creation.
   */
  public function ajaxCreateTemplate(array &$form, FormStateInterface $form_state) {
    // Save selected checkboxes before AJAX refresh.
    $this->saveSelectedProducts($form_state);

    $product_type = $form_state->getValue('product_type');
    $user_input = $form_state->getUserInput();
    $template_mode = $user_input['template_mode'] ?? 'create_new';

    $form['template_section']['template_options'] = [
      '#type' => 'container',
      '#prefix' => '<div id="template-options-wrapper">',
      '#suffix' => '</div>',
    ];

    if ($template_mode === 'existing') {
      $this->buildExistingTemplateOptions($form['template_section']['template_options'], $form_state, $product_type);
    }
    else {
      // Use type-specific key based on product type.
      $template_product_id = $product_type ? $form_state->get('template_product_id_' . $product_type) : NULL;
      $this->buildNewTemplateOptions($form['template_section']['template_options'], $form_state, $product_type, $template_product_id);
    }

    return $form['template_section']['template_options'];
  }

  /**
   * AJAX callback for template deletion.
   */
  public function ajaxDeleteTemplate(array &$form, FormStateInterface $form_state) {
    return $form['template_section'];
  }

  /**
   * Create template product.
   */
  public function createTemplateProduct(array &$form, FormStateInterface $form_state) {
    // Save selected checkboxes before AJAX refresh.
    $this->saveSelectedProducts($form_state);

    $product_type = $form_state->getValue('product_type');

    if (!$product_type) {
      $this->messenger()->addError($this->t('Please select a product type first.'));
      return;
    }

    $product_type_entity = ProductType::load($product_type);
    $product_type_label = $product_type_entity ? $product_type_entity->label() : $product_type;

    $template_product = Product::create([
      'type' => $product_type,
      'title' => '[TEMPLATE] ' . $product_type_label . ' - ' . date('Y-m-d H:i:s'),
      'status' => FALSE,
    ]);
    $template_product->save();

    // CRITICAL: Use type-specific key to avoid conflicts between different product types.
    $form_state->set('template_product_id_' . $product_type, $template_product->id());
    $form_state->set('template_product_type', $product_type);

    $form_state->setRebuild(TRUE);
  }

  /**
   * Delete template product.
   */
  public function deleteTemplateProduct(array &$form, FormStateInterface $form_state) {
    // Save selected checkboxes before AJAX refresh.
    $this->saveSelectedProducts($form_state);

    // Get current product type to use type-specific key.
    $product_type = $form_state->getValue('product_type');
    $template_id = $product_type ? $form_state->get('template_product_id_' . $product_type) : NULL;

    if ($template_id) {
      $template = Product::load($template_id);
      if ($template) {
        $template->delete();
      }
      // Clear type-specific template ID from form state.
      if ($product_type) {
        $form_state->set('template_product_id_' . $product_type, NULL);
      }
    }
    else {
      $product_type = $form_state->getValue('product_type') ?: $form_state->getUserInput()['product_type'];
      if ($product_type) {
        $product_ids = $this->entityTypeManager->getStorage('commerce_product')
          ->getQuery()
          ->condition('type', $product_type)
          ->condition('title', '[TEMPLATE]', 'CONTAINS')
          ->accessCheck(TRUE)
          ->sort('created', 'DESC')
          ->range(0, 1)
          ->execute();

        if (!empty($product_ids)) {
          $template_id = reset($product_ids);
          $template = Product::load($template_id);
          if ($template) {
            $template->delete();
          }
        }
      }
    }

    // Clear template_ready value so new template won't be auto-selected.
    $form_state->setValue('template_ready', NULL);
    $form_state->setValue('template_mode', 'create_new');
    $user_input = $form_state->getUserInput();
    $form_state->setValue('product_type', $user_input['product_type']);
    $user_input['template_mode'] = 'create_new';
    $user_input['template_ready'] = NULL;
    $form_state->setUserInput($user_input);

    $form_state->setRebuild(TRUE);
  }

  /**
   * Build template preview.
   */
  private function buildTemplatePreview(&$form, $product_id) {
    $product = Product::load($product_id);

    if (!$product || !$product->hasField('layout_builder__layout')) {
      return;
    }

    $layout_data = $product->get('layout_builder__layout')->getValue();

    if (empty($layout_data)) {
      $form['preview_container'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['layout-preview-empty']],
      ];

      $form['preview_container']['content'] = [
        '#type' => 'inline_template',
        '#template' => '<div class="layout-preview-empty__content">
      <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24"
      fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round"
      stroke-linejoin="round" style="opacity: 0.5;">
        <rect x="3" y="3" width="18" height="18" rx="2" ry="2"></rect>
        <line x1="9" y1="9" x2="15" y2="15"></line>
        <line x1="15" y1="9" x2="9" y2="15"></line>
      </svg>
      <h4>' . $this->t('No Layout Configured') . '</h4>
      <p>' . $this->t('This template has no layout configured yet. Please edit the layout first.') . '</p>
    </div>',
      ];

      return;
    }

    $form['preview_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Template Layout Preview'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['layout-preview-section']],
    ];

    $sections_info = [];
    foreach ($layout_data as $item) {
      if (isset($item['section'])) {
        $section = $item['section'];
        $layout_id = $section->getLayoutId();
        $components = $section->getComponents();

        $component_list = [];
        foreach ($components as $component) {
          $plugin_id = $component->getPluginId();
          $config = $component->get('configuration');
          $component_list[] = [
            'label' => $config['label'] ?? 'Unnamed',
            'plugin' => $plugin_id,
            'region' => $component->getRegion(),
          ];
        }

        $sections_info[] = [
          'layout' => $layout_id,
          'components' => $component_list,
        ];
      }
    }

    $form['preview_section']['sections'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['layout-preview-grid']],
    ];

    foreach ($sections_info as $index => $section_info) {
      $section_key = 'section_' . $index;

      $form['preview_section']['sections'][$section_key] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['layout-section-card']],
      ];

      $form['preview_section']['sections'][$section_key]['header'] = [
        '#markup' => '<div class="layout-section-card__header">
        <span class="layout-section-card__badge">' . $this->t('Section @num', ['@num' => $index + 1]) . '</span>
        <span class="layout-section-card__layout">' . $section_info['layout'] . '</span>
      </div>',
      ];

      $components_html = '';
      foreach ($section_info['components'] as $comp) {
        $components_html .= '<div class="layout-component-item">
        <div class="layout-component-item__icon">
        </div>
        <div class="layout-component-item__content">
          <strong>' . $comp['label'] . '</strong>
          <span class="layout-component-item__meta">' . $comp['plugin'] . ' <br>Region: ' . $comp['region'] . '</span>
        </div>
      </div>';
      }

      $form['preview_section']['sections'][$section_key]['components'] = [
        '#markup' => '<div class="layout-section-card__body">
        <div class="layout-components-list">' . $components_html . '</div>
      </div>',
      ];

      $form['preview_section']['sections'][$section_key]['footer'] = [
        '#markup' => '<div class="layout-section-card__footer">
        <span class="component-count">' .
        $this->formatPlural(
            count($section_info['components']),
            '1 component',
            '@count components'
        ) . '</span>
      </div>',
      ];
    }

  }

  /**
   * Build confirmation form with translation support.
   */
  private function buildConfirmationForm(&$form, FormStateInterface $form_state) {
    $values = $form_state->get('form_values');
    $product_type_id = $values['product_type'];
    $template_product_id = $values['template_product_id'];
    $selected_products = $values['selected_products'];
    $selected_langcode = $values['langcode'] ?? NULL;
    $apply_all_translations = $values['apply_all_translations'] ?? FALSE;
    $override_existing = $values['override_existing'] ?? FALSE;

    $product_type = ProductType::load($product_type_id);
    $template_product = Product::load($template_product_id);

    $product_count = count($selected_products);

    // Calculate dry run statistics.
    $dry_run_stats = $this->calculateDryRunStats(
      $template_product_id,
      $selected_products,
      $selected_langcode,
      $apply_all_translations,
      $override_existing
    );

    // Get list of products that will actually be updated.
    $products_to_update = [];
    $products = Product::loadMultiple($selected_products);
    foreach ($products as $product) {
      $result = $this->applyTemplateLayout(
        $template_product_id,
        $product,
        $selected_langcode,
        $apply_all_translations,
        $override_existing,
        // dry_run = TRUE.
        TRUE
      );

      if ($result['translations_updated'] > 0) {
        $products_to_update[] = $product->id();
      }
    }

    // Calculate translation stats only for products that will be updated.
    $translation_stats = $this->calculateTranslationStats($products_to_update, $selected_langcode, $apply_all_translations);

    // Add override warning if override is enabled.
    if ($override_existing) {
      $form['override_warning'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        '#weight' => -10,
        'message' => [
          '#markup' => '<strong>' . $this->t('WARNING: Override Mode Active') . '</strong><br>' .
          $this->t('Layouts will be overwritten even if they are already valid. This action cannot be undone!'),
        ],
      ];
    }

    $form['confirmation'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--warning']],
    ];

    $form['confirmation']['title'] = [
      '#markup' => '<h2>' . $this->t('Confirm Layout Update') . '</h2>',
    ];

    $template_layout_url = Url::fromRoute('layout_builder.overrides.commerce_product.view', [
      'commerce_product' => $template_product_id,
    ])->toString();

    $language_name = $this->getLanguageName($selected_langcode);

    $details_html = '<div class="confirmation-details">
      <p><strong>' . $this->t('You are about to apply the following changes:') . '</strong></p>
      <ul>
        <li><strong>' . $this->t('Product Type:') . '</strong> ' . htmlspecialchars($product_type->label()) . ' (' . htmlspecialchars($product_type_id) . ')</li>
        <li><strong>' . $this->t('Template Product:') . '</strong> ' . htmlspecialchars($template_product->getTitle()) . ' (ID: ' . $template_product_id . ') →
          <strong>
            <a href="' . $template_layout_url . '" target="_blank" class="preview-layout-btn">' . $this->t('Preview Layout') . '</a>
          </strong>
        </li>
        <li><strong>' . $this->t('Language:') . '</strong> ' . htmlspecialchars($language_name) . '</li>';

    // Add override existing status.
    $override_status = $override_existing ? 'ON' : 'OFF';
    $details_html .= '<li><strong>' . $this->t('Override existing layouts:') . '</strong> ' . $override_status . '</li>';

    if ($apply_all_translations) {
      $details_html .= '<li>' . $this->t('<strong>Apply to all translations:</strong> @status', [
        '@status' => 'ON',
      ]) . '</li>';

      $details_html .= '<li>' . $this->t('<strong>Total translations to update:</strong> @count', [
        '@count' => $translation_stats['total_translations'],
      ]) . '</li>';
    }
    else {
      $details_html .= '<li>' . $this->t('<strong>Apply to all translations:</strong> @status', [
        '@status' => 'OFF',
      ]) . '</li>';
    }

    $details_html .= '<li><strong>' . $this->t('Products selected:') . '</strong> ' . $product_count . '</li>';

    // Add dry run statistics.
    $details_html .= '<li><strong>' . $this->t('Products that will be updated:') . '</strong> ' . $dry_run_stats['will_update'] . '</li>';
    $details_html .= '<li><strong>' . $this->t('Products that will be skipped:') . '</strong> ' . $dry_run_stats['will_skip'] . '</li>';

    // Add total translations info if apply all translations is ON.
    if ($apply_all_translations && !empty($translation_stats['total_translations'])) {
      $details_html .= '<li><strong>' . $this->t('Total translations that will be updated:') . '</strong> ' . $translation_stats['total_translations'] . '</li>';
    }

    $details_html .= '</ul>
      <div class="messages messages--warning margin-bottom-0">
        <strong>' . $this->t('Warning: This action will modify the layouts of all selected products. This action cannot be undone automatically.') . '</strong>
      </div>
    </div>';

    $form['confirmation']['details'] = [
      '#type' => 'processed_text',
      '#text' => $details_html,
      '#format' => 'full_html',
      '#allowed_tags' => ['a', 'div', 'ul', 'li', 'p', 'strong', 'span', 'style'],
    ];

    if ($apply_all_translations && !empty($translation_stats['by_language'])) {
      $form['translation_breakdown'] = [
        '#type' => 'details',
        '#title' => $this->t('Translation Breakdown'),
        '#collapsed' => TRUE,
      ];

      $lang_rows = [];
      foreach ($translation_stats['by_language'] as $langcode => $count) {
        $lang_name = $this->getLanguageName($langcode);
        $lang_rows[] = [
          htmlspecialchars($lang_name),
          $count . ' ' . $this->formatPlural($count, 'product', 'products'),
        ];
      }

      $form['translation_breakdown']['table'] = [
        '#type' => 'table',
        '#header' => [$this->t('Language'), $this->t('Products')],
        '#rows' => $lang_rows,
      ];
    }

    if (!empty($selected_products)) {
      $products = Product::loadMultiple($selected_products);

      // Build entities data for table - only include products
      // that will actually be updated.
      $entities_with_data = [];
      foreach ($products as $product) {
        // Check if this product will actually be updated using dry run.
        $result = $this->applyTemplateLayout(
          $template_product_id,
          $product,
          $selected_langcode,
          $apply_all_translations,
          $override_existing,
          // dry_run = TRUE.
          TRUE
        );

        // Only add to table if product will be updated.
        if ($result['translations_updated'] > 0) {
          $original_product = $product->getUntranslated();
          $original_lang = $original_product->language()->getId();
          $viewing_lang = $product->language()->getId();
          $is_translation = $viewing_lang !== $original_lang;

          $languages_to_update = [];
          if ($apply_all_translations && $product->isTranslatable()) {
            $translation_languages = $product->getTranslationLanguages();
            foreach ($translation_languages as $lang) {
              $languages_to_update[] = $lang->getId();
            }
          }
          else {
            $languages_to_update[] = $selected_langcode;
          }

          $entities_with_data[] = [
            'product' => $product,
            'empty_fields' => [],
            'is_translation' => $is_translation,
            'languages_to_update' => $languages_to_update,
          ];
        }
      }

      // Only show the table if there are products to update.
      if (!empty($entities_with_data)) {
        $form['affected_products'] = [
          '#type' => 'details',
          '#title' => $this->t('Products that will be updated (@count)', ['@count' => count($entities_with_data)]),
          '#collapsed' => TRUE,
        ];

        // Build header.
        $header = [
          $this->t('Product ID'),
          $this->t('Title'),
          $this->t('Language'),
          $this->t('Translations to Update'),
          $this->t('Actions'),
        ];

        // Build rows.
        $rows = [];
        foreach ($entities_with_data as $entity_data) {
          $product = $entity_data['product'];
          $languages_to_update = $entity_data['languages_to_update'];
          $is_translation = $entity_data['is_translation'];

          $product_url = Url::fromRoute('entity.commerce_product.canonical', ['commerce_product' => $product->id()]);
          $title = $product->getTitle();
          $max_length = 50;
          $truncated_title = mb_strlen($title) > $max_length
            ? mb_substr($title, 0, $max_length) . '...'
            : $title;

          $title_link = [
            '#type' => 'link',
            '#title' => $truncated_title,
            '#url' => $product_url,
            '#attributes' => [
              'target' => '_blank',
              'title' => $title,
            ],
          ];

          // Build language cell.
          $langcode = $product->language()->getId();
          $language = $this->languageManager->getLanguage($langcode);
          $language_name = $language ? $language->getName() : $langcode;
          $badge_color = $is_translation ? '#0074bd' : '#5cb85c';
          $badge_text = $is_translation ? $this->t('Translation') : $this->t('Original');

          $lang_cell = [
            'data' => [
              '#markup' => '<div style="white-space: nowrap;">' .
              '<strong>' . htmlspecialchars($language_name) . '</strong>' .
              '<br><span style="display: inline-block; padding: 2px 6px; background: ' . $badge_color . '; color: white; border-radius: 3px; font-size: 11px; margin-top: 2px;">' .
              $badge_text .
              '</span></div>',
            ],
          ];

          // Build translations to update cell.
          $language_names = [];
          foreach ($languages_to_update as $langcode) {
            $lang = $this->languageManager->getLanguage($langcode);
            $language_names[] = $lang ? $lang->getName() : $langcode;
          }
          $count = count($language_names);
          $display_text = implode(', ', $language_names);

          $translations_cell = [
            'data' => [
              '#markup' => htmlspecialchars($display_text) . ' (' . $count . ')',
            ],
          ];

          // Use buildProductOperations for consistent dropdown.
          $operations = $this->buildProductOperations($product, $languages_to_update);

          $rows[] = [
            $product->id(),
            ['data' => $title_link],
            $lang_cell,
            $translations_cell,
            $operations,
          ];
        }

        $form['affected_products']['table_wrapper'] = [
          '#type' => 'container',
          '#attributes' => [
            'class' => 'table-responsive layout-manager-table',
          ],
        ];

        $form['affected_products']['table_wrapper']['table'] = [
          '#type' => 'table',
          '#header' => $header,
          '#rows' => $rows,
          '#attributes' => ['class' => ['admin-list', 'sticky-table']],
        ];
      }
    }

    $form['actions'] = ['#type' => 'actions'];

    // Disable confirm button if no products will be updated.
    $no_changes = ($dry_run_stats['will_update'] === 0);
    $classes = ['button--danger'];
    if (!$no_changes) {
      $classes[] = 'confirm-countdown-button';
    }

    $form['actions']['confirm'] = [
      '#type' => 'submit',
      '#value' => $no_changes ? $this->t('No Changes to Apply') : $this->t('Yes, Update All Layouts'),
      '#button_type' => 'primary',
      '#submit' => ['::confirmSubmit'],
      '#attributes' => [
        'class' => $classes,
        'id' => 'confirm-update-button',
      ],
      '#disabled' => $no_changes,
    ];

    $form['actions']['back'] = [
      '#type' => 'submit',
      '#value' => $this->t('Back to Selection'),
      '#submit' => ['::backToEdit'],
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['button']],
      '#ajax' => [
        'callback' => '::ajaxBackToEdit',
        'wrapper' => 'field-main-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
    ];

    $form['actions']['cancel'] = [
      '#type' => 'submit',
      '#value' => $this->t('Cancel'),
      '#attributes' => ['class' => ['button--danger']],
      '#submit' => ['::cancelSubmit'],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::ajaxCancelSubmit',
        'wrapper' => 'field-main-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
    ];

    return $form;
  }

  /**
   * Calculate translation statistics.
   */
  private function calculateTranslationStats(array $product_ids, $selected_langcode, $apply_all_translations) {
    $stats = [
      'total_translations' => 0,
      'by_language' => [],
    ];

    if (!$apply_all_translations) {
      return $stats;
    }

    $products = Product::loadMultiple($product_ids);

    foreach ($products as $product) {
      if (!$product->isTranslatable()) {
        continue;
      }

      $translation_languages = $product->getTranslationLanguages();
      foreach ($translation_languages as $lang) {
        $langcode = $lang->getId();

        if (!isset($stats['by_language'][$langcode])) {
          $stats['by_language'][$langcode] = 0;
        }

        $stats['by_language'][$langcode]++;
        $stats['total_translations']++;
      }
    }

    return $stats;
  }

  /**
   * Build product section with language selector.
   */
  private function buildProductSection(&$form, FormStateInterface $form_state, $product_type_id) {
    $languages = $this->languageManager->getLanguages();
    $language_options = [];

    foreach ($languages as $langcode => $language) {
      $language_options[$langcode] = $language->getName();
    }

    // Get selected language with fallback to stored value (for back button).
    $selected_langcode = $form_state->getValue('layout_language_selector')
      ?? $form_state->get('form_values')['langcode']
      ?? $this->languageManager->getCurrentLanguage()->getId();

    $form['language_options'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['language-options-container'],
      ],
    ];

    $form['language_options']['layout_language_selector'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Language / Translation'),
      '#options' => $language_options,
      '#default_value' => $form_state->getValue('layout_language_selector') ?? $form_state->get('form_values')['langcode'] ?? $selected_langcode,
      '#description' => $this->t('Select which language version of products to check for layout issues.'),
      '#ajax' => [
        'callback' => '::updateProductTypeAndTemplate',
        'wrapper' => 'product-list-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
      '#weight' => -10,
    ];

    $form['language_options']['apply_all_translations'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Apply layout to all translations'),
      '#description' => $this->t('If checked, the layout will be applied to all available translations of the selected products.'),
      '#default_value' => $form_state->getValue('apply_all_translations') ?? $form_state->get('form_values')['apply_all_translations'] ?? FALSE,
    ];

    $form['language_options']['include_valid_layouts'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include products with already valid layouts'),
      '#description' => $this->t('If checked, products that already have valid layouts will also be shown in the list.'),
      '#default_value' => $form_state->getValue('include_valid_layouts') ?? $form_state->get('form_values')['include_valid_layouts'] ?? FALSE,
      '#ajax' => [
        'callback' => '::updateProductTypeAndTemplate',
        'wrapper' => 'product-list-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
    ];

    $include_valid = $form_state->getValue('include_valid_layouts') ?? $form_state->get('form_values')['include_valid_layouts'] ?? FALSE;

    // If include_valid is OFF, force override_existing to OFF.
    $override_existing = FALSE;
    if ($include_valid) {
      $override_existing = $form_state->getValue('override_existing') ?? $form_state->get('form_values')['override_existing'] ?? FALSE;
    }

    $form['language_options']['override_existing'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Override existing valid layouts'),
      '#description' => $this->t('If checked, layouts will be overridden even if they are already valid. <strong>Use with caution!</strong>'),
      '#default_value' => $override_existing,
    ];

    // Hide override checkbox if include_valid is not checked.
    if (!$include_valid) {
      $form['language_options']['override_existing']['#access'] = FALSE;
    }

    $problematic_products = $this->getProblematicProducts($product_type_id, $selected_langcode, $include_valid, $form_state);

    $product_type = ProductType::load($product_type_id);
    $product_type_label = $product_type ? $product_type->label() : $product_type_id;

    $language_name = $this->getLanguageName($selected_langcode);

    $form['product_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Step 2: Select Products to Update (@count) - Language: @lang', [
        '@count' => count($problematic_products),
        '@lang' => $language_name,
      ]),
      '#open' => TRUE,
    ];

    if (empty($problematic_products)) {
      $form['product_section']['no_issues'] = [
        '#markup' => '<div class="messages messages--status">' .
        $this->t('All @type products have valid layouts in @lang.', [
          '@type' => $product_type_label,
          '@lang' => $language_name,
        ]) . '</div>',
      ];
      return;
    }

    $form['product_section']['products_count'] = [
      '#markup' => '<p><strong>' .
      $this->formatPlural(count($problematic_products),
          '1 ' . $product_type_label . ' product found:',
          '@count ' . $product_type_label . ' products found:') .
      '</strong></p>',
    ];

    $form['product_section']['selection_controls'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['selection-controls'],
      ],
    ];

    $form['product_section']['selection_controls']['select_all'] = [
      '#type' => 'button',
      '#value' => $this->t('Select All'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'id' => 'product-select-all',
      ],
    ];

    $form['product_section']['selection_controls']['deselect_all'] = [
      '#type' => 'button',
      '#value' => $this->t('Deselect All'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'id' => 'product-deselect-all',
      ],
    ];

    $header = [
      $this->t('Select'),
      $this->t('Product ID'),
      $this->t('Title'),
      $this->t('Language'),
      $this->t('Translations'),
      $this->t('Issue'),
      $this->t('Created'),
      $this->t('Operations'),
    ];

    $rows = [];

    // Get stored selected products.
    $stored_selected_products = $form_state->get('selected_products') ?? [];

    // If empty, try user input (for initial render).
    if (empty($stored_selected_products)) {
      $user_input = $form_state->getUserInput();
      if (isset($user_input['selected_products']) && is_array($user_input['selected_products'])) {
        $stored_selected_products = array_values(array_filter($user_input['selected_products']));
      }
    }

    // CRITICAL: Store visible product IDs for checkbox persistence logic
    // This must be done BEFORE saveSelectedProducts is called.
    $visible_product_ids = [];
    foreach ($problematic_products as $product_data) {
      $visible_product_ids[] = $product_data['product']->id();
    }
    $form_state->set('visible_product_ids', $visible_product_ids);

    foreach ($problematic_products as $product_data) {
      $product = $product_data['product'];

      // Check if this product is in the stored selection.
      $is_selected = in_array($product->id(), $stored_selected_products);

      $checkbox = [
        'data' => [
          '#type' => 'checkbox',
          '#name' => 'selected_products[' . $product->id() . ']',
          '#attributes' => ['class' => ['entity-checkbox'], 'data-product-id' => $product->id()],
          '#return_value' => $product->id(),
          '#default_value' => $is_selected ? $product->id() : 0,
        ],
      ];

      // CRITICAL: For table checkboxes, we need to manually set the checked attribute
      // because Drupal's table rendering doesn't always respect #default_value.
      if ($is_selected) {
        $checkbox['data']['#attributes']['checked'] = 'checked';
      }
      else {
        unset($checkbox['data']['#attributes']['checked']);
      }

      $product_url = Url::fromRoute('entity.commerce_product.canonical', ['commerce_product' => $product->id()]);

      // Truncate long titles.
      $title = $product->getTitle();
      $max_length = 40;
      $truncated_title = mb_strlen($title) > $max_length
        ? mb_substr($title, 0, $max_length) . '...'
        : $title;

      $title_link = [
        '#type' => 'link',
        '#title' => $truncated_title,
        '#url' => $product_url,
        '#attributes' => [
          'target' => '_blank',
          // Full title on hover.
          'title' => $title,
        ],
      ];

      // Get the viewing language for proper URL generation.
      $viewing_lang = $product->language()->getId();

      // Collect all translation languages for operations dropdown.
      $languages_for_operations = [];
      if ($product->isTranslatable()) {
        $translation_languages = $product->getTranslationLanguages();
        foreach ($translation_languages as $lang) {
          $languages_for_operations[] = $lang->getId();
        }
      }
      else {
        $languages_for_operations = [$viewing_lang];
      }

      // Use buildProductOperations for consistent dropdown.
      $operations = $this->buildProductOperations($product, $languages_for_operations);

      $original_product = $product->getUntranslated();
      $original_lang = $original_product->language()->getId();
      $original_lang_name = $this->getLanguageName($original_lang);

      $viewing_lang = $product->language()->getId();
      $is_translation = $viewing_lang !== $original_lang;
      $badge_text = $is_translation ? $this->t('Translation') : $this->t('Original');

      if ($is_translation) {
        $viewing_lang_name = $this->getLanguageName($viewing_lang);
        $lang_cell = '<div>' .
          '<strong>' . htmlspecialchars($viewing_lang_name) . '</strong>' .
          '<br><span>' .
          htmlspecialchars($badge_text) .
          '</span>' .
          '<br><span >Original: ' . htmlspecialchars($original_lang_name) . '</span>' .
          '</div>';
      }
      else {
        $lang_cell = '<div>' .
          '<strong>' . htmlspecialchars($original_lang_name) . '</strong>' .
          '<br><span>' .
          htmlspecialchars($badge_text) .
          '</span></div>';
      }

      $translations = [];
      if ($product->isTranslatable()) {
        $translation_languages = $product->getTranslationLanguages();
        foreach ($translation_languages as $lang) {
          $translations[] = $this->getLanguageName($lang->getId());
        }
      }
      $translations_text = !empty($translations) ? implode(', ', $translations) . ' (' . count($translations) . ')' : $this->t('None');

      $issue_text = $product_data['issue'];
      $has_valid = $product_data['has_valid_layout'] ?? FALSE;

      if ($has_valid) {
        $issue_cell = '<span class="valid-layout">' . $issue_text . '</span>';
      }
      else {
        $issue_cell = '<span class="invalid-layout">' . $issue_text . '</span>';
      }

      $rows[] = [
        $checkbox,
        $product->id(),
        ['data' => $title_link],
        ['data' => ['#markup' => $lang_cell]],
        $translations_text,
        ['data' => ['#markup' => $issue_cell]],
        $this->dateFormatter->format($product->getCreatedTime(), 'short'),
        ['data' => $operations],
      ];
    }

    $form['product_section']['table_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['table-responsive layout-manager-table'],
      ],
    ];

    $form['product_section']['table_container']['products_table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No products found.'),
      '#attributes' => [
        'class' => ['admin-list', 'sticky-table'],
      ],
      '#attached' => [
        'library' => ['system/admin'],
      ],
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['fix_layouts'] = [
      '#type' => 'submit',
      '#value' => $this->t('Apply Template to Selected Products'),
      '#button_type' => 'primary',
      '#ajax' => [
        'callback' => '::ajaxSubmitForm',
        'wrapper' => 'field-main-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
    ];
  }

  /**
   * AJAX callback for product type and template update.
   */
  public function updateProductTypeAndTemplate(array &$form, FormStateInterface $form_state) {
    // Save selected checkboxes before AJAX refresh.
    $this->saveSelectedProducts($form_state);

    // Only return the product container, not the entire form.
    return $form['product_container'];
  }

  /**
   * Get layout enabled product types.
   */
  private function getLayoutEnabledProductTypes() {
    $product_types = ProductType::loadMultiple();
    $layout_enabled = [];
    foreach ($product_types as $type_id => $type) {
      $display = LayoutBuilderEntityViewDisplay::load('commerce_product.' . $type_id . '.default');
      if ($display && $display->isLayoutBuilderEnabled()) {
        $layout_enabled[$type_id] = $type->label();
      }
    }
    return $layout_enabled;
  }

  /**
   * Build operations dropdown for a product.
   *
   * @param object $product
   *   The product entity.
   * @param array $languages_to_update
   *   Array of language codes that will be updated.
   *
   * @return array
   *   Operations render array with dropbutton.
   */
  private function buildProductOperations($product, array $languages_to_update = []) {
    $viewing_lang = $product->language()->getId();
    $language_object = $this->languageManager->getLanguage($viewing_lang);

    $edit_url = Url::fromRoute('entity.commerce_product.edit_form', [
      'commerce_product' => $product->id(),
    ], [
      'language' => $language_object,
    ]);

    // If no languages specified, use current viewing language.
    if (empty($languages_to_update)) {
      $languages_to_update = [$viewing_lang];
    }

    $links = [
      'edit' => [
        'title' => $this->t('Edit'),
        'url' => $edit_url,
        'attributes' => ['target' => '_blank'],
      ],
    ];

    // Add layout edit links.
    if (count($languages_to_update) === 1) {
      // Single language.
      $langcode = reset($languages_to_update);
      $lang_object = $this->languageManager->getLanguage($langcode);
      $layout_url = Url::fromRoute('layout_builder.overrides.commerce_product.view', [
        'commerce_product' => $product->id(),
      ], [
        'language' => $lang_object,
      ]);

      $links['layout'] = [
        'title' => $this->t('Edit Layout'),
        'url' => $layout_url,
        'attributes' => ['target' => '_blank'],
      ];
    }
    else {
      // Multiple languages - add each as separate link with language name.
      foreach ($languages_to_update as $langcode) {
        $lang = $this->languageManager->getLanguage($langcode);
        $lang_name = $lang ? $lang->getName() : $langcode;
        $lang_object = $this->languageManager->getLanguage($langcode);

        $layout_url = Url::fromRoute('layout_builder.overrides.commerce_product.view', [
          'commerce_product' => $product->id(),
        ], [
          'language' => $lang_object,
        ]);

        $links['layout_' . $langcode] = [
          'title' => $this->t('Edit Layout (@lang)', ['@lang' => $lang_name]),
          'url' => $layout_url,
          'attributes' => ['target' => '_blank'],
        ];
      }
    }

    return [
      'data' => [
        '#type' => 'operations',
        '#links' => $links,
        '#attributes' => ['class' => ['dropbutton--extrasmall']],
      ],
    ];
  }

  /**
   * Get problematic products with language support.
   */
  private function getProblematicProducts($product_type_id, $langcode = NULL, $include_valid = FALSE, ?FormStateInterface $form_state = NULL) {
    $problematic = [];

    $template_product_id = $form_state?->get('template_product_id');

    if ($langcode === NULL) {
      $langcode = $this->languageManager->getCurrentLanguage()->getId();
    }

    $product_ids = $this->entityTypeManager->getStorage('commerce_product')
      ->getQuery()
      ->condition('type', $product_type_id)
      ->accessCheck(TRUE)
      ->execute();

    if (empty($product_ids)) {
      return $problematic;
    }

    $products = Product::loadMultiple($product_ids);
    foreach ($products as $product) {
      if ($template_product_id && $product->id() == $template_product_id) {
        continue;
      }

      if (strpos($product->getTitle(), '[TEMPLATE]') !== FALSE) {
        continue;
      }

      $original_product = $product->getUntranslated();
      $original_langcode = $original_product->language()->getId();

      $check_product = NULL;

      if ($langcode === $original_langcode) {
        // User selected the original language - check the original.
        $check_product = $original_product;
      }
      elseif ($product->isTranslatable() && $product->hasTranslation($langcode)) {
        // User selected a different language and translation exists.
        $check_product = $product->getTranslation($langcode);
      }
      else {
        // Translation doesn't exist for selected language - skip this product.
        continue;
      }

      $issue = $this->checkProductLayoutIssues($check_product);

      if ($include_valid || $issue) {
        $is_translation = ($check_product->language()->getId() !== $original_langcode);
        $problematic[] = [
          'product' => $check_product,
          'issue' => $issue ?: $this->t('Valid layout'),
          'has_valid_layout' => !$issue,
          'is_translation' => $is_translation,
          'original_langcode' => $original_langcode,
        ];
      }
    }
    return $problematic;
  }

  /**
   * Check product layout issues.
   */
  private function checkProductLayoutIssues($product) {
    if (!$product->hasField('layout_builder__layout')) {
      return $this->t('No layout field');
    }

    $layout_sections = $product->get('layout_builder__layout')->getValue();
    if (empty($layout_sections)) {
      return $this->t('No sections defined or not overridden');
    }

    $has_valid_section = FALSE;
    $all_sections_blank = TRUE;
    $missing_blocks = [];

    foreach ($layout_sections as $section_item) {
      $section_data = $section_item['section'] ?? NULL;

      if (!$section_data) {
        continue;
      }

      $section = $section_data;
      $layout_id = $section->getLayoutId();

      if ($layout_id !== 'layout_builder_blank') {
        $all_sections_blank = FALSE;
      }

      $components = $section->getComponents();
      if (!empty($components)) {
        $section_has_missing_blocks = FALSE;

        foreach ($components as $component) {
          $plugin_id = $component->getPluginId();
          if (!$this->blockManager->hasDefinition($plugin_id)) {
            $missing_blocks[] = $plugin_id;
            $section_has_missing_blocks = TRUE;
          }
        }

        if (!$section_has_missing_blocks) {
          $has_valid_section = TRUE;
        }
      }
    }

    if ($has_valid_section) {
      return FALSE;
    }

    if ($all_sections_blank) {
      return $this->t('All sections use blank layout');
    }

    if (!empty($missing_blocks)) {
      return $this->t('Missing blocks: @blocks', ['@blocks' => implode(', ', array_unique($missing_blocks))]);
    }

    return $this->t('No components in any section');
  }

  /**
   * Get language name from language code.
   */
  private function getLanguageName($langcode) {
    if (!$langcode) {
      return $this->languageManager->getCurrentLanguage()->getName();
    }

    $language = $this->languageManager->getLanguage($langcode);
    return $language ? $language->getName() : $langcode;
  }

  /**
   * AJAX callback for form submission.
   */
  public function ajaxSubmitForm(array &$form, FormStateInterface $form_state) {
    // Save selected checkboxes before proceeding to confirmation.
    $this->saveSelectedProducts($form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $template_mode = $form_state->getValue('template_mode');
    $product_type_id = $form_state->getValue('product_type');
    $selected_langcode = $form_state->getValue('layout_language_selector')
      ?? $this->languageManager->getCurrentLanguage()->getId();
    $apply_all_translations = $form_state->getValue('apply_all_translations') ?? FALSE;
    $include_valid_layouts = $form_state->getValue('include_valid_layouts') ?? FALSE;
    $override_existing = $form_state->getValue('override_existing') ?? FALSE;

    $template_product_id = NULL;

    if ($template_mode === 'existing') {
      $template_product_id = $form_state->getValue('existing_product');

      if (!$template_product_id) {
        $this->saveSelectedProducts($form_state);
        $this->messenger()->addError($this->t('Please select a template product.'));
        $form_state->setRebuild(TRUE);
        return;
      }
    }
    else {
      $template_product_id = $product_type_id ? $form_state->get('template_product_id_' . $product_type_id) : NULL;

      if (!$template_product_id) {
        $this->saveSelectedProducts($form_state);
        $this->messenger()->addError($this->t('Please create a template product first.'));
        $form_state->setRebuild(TRUE);
        return;
      }

      $template_ready = $form_state->getValue('template_ready') ?? FALSE;
      if (!$template_ready) {
        $this->saveSelectedProducts($form_state);
        $this->messenger()->addWarning($this->t('Please confirm that you have finished editing the template layout.'));
        $form_state->setRebuild(TRUE);
        return;
      }
    }

    // Save selected products first.
    $this->saveSelectedProducts($form_state);

    // Then get from form state.
    $selected_products = $form_state->get('selected_products') ?? [];

    if (empty($selected_products)) {
      $this->messenger()->addError($this->t('Please select at least one product to update.'));
      $form_state->setRebuild(TRUE);
      return;
    }

    $template_product = Product::load($template_product_id);
    if (!$template_product || !$template_product->hasField('layout_builder__layout')) {
      $this->messenger()->addError($this->t('Template product has no layout field.'));
      return;
    }

    $layout_data = $template_product->get('layout_builder__layout')->getValue();
    if (empty($layout_data)) {
      $this->messenger()->addError($this->t('Template product has no layout configured. Please edit the layout first.'));
      return;
    }

    $form_state->set('form_values', [
      'product_type' => $product_type_id,
      'template_product_id' => $template_product_id,
      'template_mode' => $template_mode,
      'template_ready' => $template_mode === 'create' ? ($template_ready ?? TRUE) : TRUE,
      'selected_products' => $selected_products,
      'langcode' => $selected_langcode,
      'apply_all_translations' => $apply_all_translations,
      'include_valid_layouts' => $include_valid_layouts,
      'override_existing' => $override_existing,
    ]);

    $form_state->set('step', 'confirm');
    $form_state->setRebuild(TRUE);
  }

  /**
   * Confirm submit handler with translation support.
   */
  public function confirmSubmit(array &$form, FormStateInterface $form_state) {
    $values = $form_state->get('form_values');
    $product_type_id = $values['product_type'];
    $template_product_id = $values['template_product_id'];
    $template_mode = $values['template_mode'];
    $selected_products = $values['selected_products'];
    $selected_langcode = $values['langcode'] ?? NULL;
    $apply_all_translations = $values['apply_all_translations'] ?? FALSE;
    $override_existing = $values['override_existing'] ?? FALSE;

    if (empty($selected_products)) {
      $this->messenger()->addStatus($this->t('No products selected.'));
      $this->clearFormState($form_state);
      return;
    }

    $fixed_count = 0;
    $failed_count = 0;
    $skipped_count = 0;
    $total_translations_updated = 0;

    foreach ($selected_products as $product_id) {
      $product = Product::load($product_id);

      if (!$product) {
        $failed_count++;
        continue;
      }

      $result = $this->applyTemplateLayout($template_product_id, $product, $selected_langcode, $apply_all_translations, $override_existing);

      if ($result['success']) {
        if ($result['translations_updated'] > 0) {
          $fixed_count++;
          $total_translations_updated += $result['translations_updated'];
        }
        else {
          $skipped_count++;
        }
      }
      else {
        $failed_count++;
      }
    }

    $language_name = $this->getLanguageName($selected_langcode);

    if ($fixed_count > 0) {
      if ($apply_all_translations) {
        $this->messenger()->addStatus($this->t(
          'Applied template layout to @count product(s) across @translations translation(s).',
          [
            '@count' => $fixed_count,
            '@translations' => $total_translations_updated,
          ]
        ));
      }
      else {
        $this->messenger()->addStatus($this->t(
          'Applied template layout to @count product(s) in @lang.',
          [
            '@count' => $fixed_count,
            '@lang' => $language_name,
          ]
        ));
      }
    }

    if ($skipped_count > 0) {
      $this->messenger()->addStatus($this->t(
        '@count product(s) skipped (already have valid layouts and override not enabled).',
        ['@count' => $skipped_count]
      ));
    }

    if ($failed_count > 0) {
      $this->messenger()->addWarning($this->formatPlural($failed_count,
        'Failed to update 1 product.',
        'Failed to update @count products.'
      ));
    }

    if ($template_mode === 'create_new') {
      // Use type-specific key based on product type.
      $template_id = $product_type_id ? $form_state->get('template_product_id_' . $product_type_id) : NULL;
      if ($template_id) {
        $template = Product::load($template_id);
        if ($template) {
          $template->delete();
        }
        // Clear type-specific template ID from form state.
        if ($product_type_id) {
          $form_state->set('template_product_id_' . $product_type_id, NULL);
        }
      }
    }

    $this->clearFormState($form_state);
  }

  /**
   * Save selected products from user input to form state.
   *
   * This method extracts checkbox selections from the current user input
   * and stores them in form state, so they persist across AJAX rebuilds.
   *
   * IMPORTANT: Only products that are BOTH visible AND selected are kept.
   * If a product becomes invisible (e.g., due to language/filter change),
   * it is removed from selections.
   */
  private function saveSelectedProducts(FormStateInterface $form_state): void {
    $user_input = $form_state->getUserInput();
    $previously_stored = $form_state->get('selected_products') ?? [];
    $visible_product_ids = $form_state->get('visible_product_ids') ?? [];

    // Get current checkbox selections.
    $user_input_selected = [];
    if (isset($user_input['selected_products']) && is_array($user_input['selected_products'])) {
      $user_input_selected = array_filter($user_input['selected_products']);
    }

    $products_to_keep = [];

    // Keep invisible products.
    foreach ($previously_stored as $product_id) {
      if (!in_array($product_id, $visible_product_ids)) {
        $products_to_keep[$product_id] = $product_id;
      }
    }

    // Add visible selected products.
    foreach ($user_input_selected as $product_id) {
      if (in_array($product_id, $visible_product_ids)) {
        $products_to_keep[$product_id] = $product_id;
      }
    }

    // Save merged selections.
    $all_selected = array_values($products_to_keep);
    $form_state->set('selected_products', $all_selected);

    // Keep in user input for rebuild.
    $current_input = $form_state->getUserInput();
    $current_input['selected_products'] = [];
    foreach ($all_selected as $product_id) {
      $current_input['selected_products'][$product_id] = $product_id;
    }

    $form_state->setUserInput($current_input);
  }

  /**
   * Clear form state.
   */
  private function clearFormState(FormStateInterface $form_state) {
    $form_state->set('step', NULL);
    $form_state->set('form_values', NULL);
    $form_state->set('selected_products', NULL);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Calculate dry run statistics for layout updates.
   */
  private function calculateDryRunStats($template_product_id, $selected_products, $langcode = NULL, $apply_all_translations = FALSE, $override_existing = FALSE) {
    $stats = [
      'will_update' => 0,
      'will_skip' => 0,
      'total_products' => count($selected_products),
    ];

    $products = Product::loadMultiple($selected_products);

    foreach ($products as $product) {
      $result = $this->applyTemplateLayout(
        $template_product_id,
        $product,
        $langcode,
        $apply_all_translations,
        $override_existing,
        // dry_run = TRUE.
        TRUE
      );

      if ($result['translations_updated'] > 0) {
        $stats['will_update']++;
      }
      else {
        $stats['will_skip']++;
      }
    }

    return $stats;
  }

  /**
   * Apply template layout to target product with translation support.
   */
  private function applyTemplateLayout($template_product_id, $target_product, $langcode = NULL, $apply_all_translations = FALSE, $override_existing = FALSE, $dry_run = FALSE) {
    $result = [
      'success' => FALSE,
      'translations_updated' => 0,
    ];

    try {
      $template_product = Product::load($template_product_id);

      if (!$template_product || !$template_product->hasField('layout_builder__layout')) {
        $this->loggerFactory->get('product_manager_tool')->error('Template product @id has no layout field', [
          '@id' => $template_product_id,
        ]);
        return $result;
      }

      $layout_data = $template_product->get('layout_builder__layout')->getValue();

      if (empty($layout_data)) {
        $this->loggerFactory->get('product_manager_tool')->error('Template product @id has no layout data', [
          '@id' => $template_product_id,
        ]);
        return $result;
      }

      $languages_to_update = [];

      if ($apply_all_translations && $target_product->isTranslatable()) {
        $translation_languages = $target_product->getTranslationLanguages();
        foreach ($translation_languages as $lang) {
          $languages_to_update[] = $lang->getId();
        }
      }
      else {
        $languages_to_update[] = $langcode ?? $target_product->language()->getId();
      }

      foreach ($languages_to_update as $update_langcode) {
        $working_product = $target_product;

        if ($target_product->isTranslatable() && $target_product->hasTranslation($update_langcode)) {
          $working_product = $target_product->getTranslation($update_langcode);
        }

        if (!$override_existing) {
          $issue = $this->checkProductLayoutIssues($working_product);
          if (!$issue) {
            // Only log if not in dry run mode.
            if (!$dry_run) {
              $this->loggerFactory->get('product_manager_tool')->info('Skipped product @id translation @lang (already valid)', [
                '@id' => $target_product->id(),
                '@lang' => $update_langcode,
              ]);
            }
            continue;
          }
        }

        // If dry run, don't actually modify the product, just count.
        if (!$dry_run) {
          $working_product->set('layout_builder__layout', []);

          foreach ($layout_data as $item) {
            if (isset($item['section'])) {
              $original_section = $item['section'];
              $new_section = new Section(
                $original_section->getLayoutId(),
                $original_section->getLayoutSettings()
              );

              foreach ($original_section->getComponents() as $component) {
                $new_uuid = $this->uuid->generate();
                $config = $component->get('configuration');

                $new_component = new SectionComponent(
                  $new_uuid,
                  $component->getRegion(),
                  $config
                );

                if (method_exists($component, 'getWeight')) {
                  $new_component->setWeight($component->getWeight());
                }

                $new_section->appendComponent($new_component);
              }

              $working_product->get('layout_builder__layout')->appendSection($new_section);
            }
          }
        }

        $result['translations_updated']++;
      }

      if ($result['translations_updated'] > 0) {
        // Only save if not in dry run mode.
        if (!$dry_run) {
          $target_product->save();

          $tempstore = $this->sharedTempStoreFactory->get('layout_builder.section_storage.overrides');
          foreach ($languages_to_update as $update_langcode) {
            $key = 'commerce_product.' . $target_product->id() . '.full.' . $update_langcode;
            $tempstore->delete($key);
          }

          // Log only when actually applying (not in dry run)
          $this->loggerFactory->get('product_manager_tool')->info('Successfully applied template layout to product @id (@count translations)', [
            '@id' => $target_product->id(),
            '@count' => $result['translations_updated'],
          ]);
        }

        $result['success'] = TRUE;
      }
      else {
        // Only log if not in dry run mode.
        if (!$dry_run) {
          $this->loggerFactory->get('product_manager_tool')->info('No translations updated for product @id (all already valid)', [
            '@id' => $target_product->id(),
          ]);
        }
        $result['success'] = TRUE;
      }

      return $result;

    }
    catch (\Exception $e) {
      $this->loggerFactory->get('product_manager_tool')->error('Error applying template layout to product @id: @error', [
        '@id' => $target_product->id(),
        '@error' => $e->getMessage(),
      ]);
      return $result;
    }
  }

  /**
   * Cancel submit handler.
   */
  public function cancelSubmit(array &$form, FormStateInterface $form_state) {
    $this->clearFormState($form_state);
  }

  /**
   * AJAX callback for cancel submit.
   */
  public function ajaxCancelSubmit(array &$form, FormStateInterface $form_state) {
    $this->clearFormState($form_state);

    $response = new AjaxResponse();
    $response->addCommand(new MessageCommand(
      $this->t('Layout update has been cancelled.'),
      NULL,
      ['type' => 'status']
    ));

    $response->addCommand(new ReplaceCommand('#tab-content-wrapper', $form['tab_content']));

    return $response;
  }

  /**
   * AJAX callback for template preview.
   */
  public function previewTemplateLayout(array &$form, FormStateInterface $form_state) {
    return $form['template_section']['template_options']['preview'];
  }

  /**
   * AJAX callback for template options update.
   */
  public function updateTemplateOptions(array &$form, FormStateInterface $form_state) {
    // Save selected checkboxes before AJAX refresh.
    $this->saveSelectedProducts($form_state);

    return $form['template_section']['template_options'];
  }

  /**
   * Back to edit handler - returns to selection screen with values preserved.
   */
  public function backToEdit(array &$form, FormStateInterface $form_state) {
    // Get the stored form values from confirmation step.
    $form_values = $form_state->get('form_values');

    // If we have selected products in form_values
    // restore them to selected_products.
    if (!empty($form_values['selected_products'])) {
      $form_state->set('selected_products', $form_values['selected_products']);
    }

    // Clear confirmation step to go back to selection.
    $form_state->set('step', 'select');
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for back to edit.
   */
  public function ajaxBackToEdit(array &$form, FormStateInterface $form_state) {
    return $form;
  }

}
