<?php

namespace Drupal\product_manager_tool\Form;

use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductType;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\commerce_product\Entity\ProductVariationType;
use Drupal\Component\Utility\Html;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\product_manager_tool\Service\FieldAccessValidator;
use Drupal\product_manager_tool\Service\FieldEntityProcessor;
use Drupal\product_manager_tool\Service\FieldTableBuilder;
use Drupal\product_manager_tool\Service\FieldValueProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Product/Variation Field Manager Form for bulk field updates.
 */
class ProductFieldManagerForm extends FormBase {

  /**
   * Logger channel name constant.
   */
  const LOGGER_CHANNEL = 'product_manager_tool';

  /**
   * Entity type constants.
   */
  const ENTITY_TYPE_PRODUCT = 'product';
  const ENTITY_TYPE_VARIATION = 'variation';

  /**
   * Storage type constants.
   */
  const STORAGE_PRODUCT = 'commerce_product';
  const STORAGE_VARIATION = 'commerce_product_variation';

  /**
   * Form state key constants.
   */
  const STATE_STEP = 'field_step';
  const STATE_EDIT_VALUES = 'field_edit_values';
  const STATE_CONFIRM_VALUES = 'field_confirm_values';
  const STATE_FIELD_STATUS = 'field_status';
  const STATE_SELECTED_ENTITIES = 'selected_entities';

  /**
   * Step value constants.
   */
  const STEP_SELECT = 'select';
  const STEP_EDIT = 'edit';
  const STEP_CONFIRM = 'confirm';

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The field entity processor service.
   *
   * @var \Drupal\product_manager_tool\Service\FieldEntityProcessor
   */
  protected $fieldEntityProcessor;

  /**
   * The field value processor service.
   *
   * @var \Drupal\product_manager_tool\Service\FieldValueProcessor
   */
  protected $fieldValueProcessor;

  /**
   * The field table builder service.
   *
   * @var \Drupal\product_manager_tool\Service\FieldTableBuilder
   */
  protected $fieldTableBuilder;

  /**
   * The field access validator service.
   *
   * @var \Drupal\product_manager_tool\Service\FieldAccessValidator
   */
  protected $fieldAccessValidator;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a ProductFieldManagerForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\product_manager_tool\Service\FieldEntityProcessor $field_entity_processor
   *   The field entity processor.
   * @param \Drupal\product_manager_tool\Service\FieldValueProcessor $field_value_processor
   *   The field value processor.
   * @param \Drupal\product_manager_tool\Service\FieldTableBuilder $field_table_builder
   *   The field table builder.
   * @param \Drupal\product_manager_tool\Service\FieldAccessValidator $field_access_validator
   *   The field access validator.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    DateFormatterInterface $date_formatter,
    LoggerChannelFactoryInterface $logger_factory,
    FieldEntityProcessor $field_entity_processor,
    FieldValueProcessor $field_value_processor,
    FieldTableBuilder $field_table_builder,
    FieldAccessValidator $field_access_validator,
    LanguageManagerInterface $language_manager,
    RendererInterface $renderer,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->dateFormatter = $date_formatter;
    $this->loggerFactory = $logger_factory;
    $this->fieldEntityProcessor = $field_entity_processor;
    $this->fieldValueProcessor = $field_value_processor;
    $this->fieldTableBuilder = $field_table_builder;
    $this->fieldAccessValidator = $field_access_validator;
    $this->languageManager = $language_manager;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // @phpstan-ignore new.static
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('date.formatter'),
      $container->get('logger.factory'),
      $container->get('product_manager_tool.field_entity_processor'),
      $container->get('product_manager_tool.field_value_processor'),
      $container->get('product_manager_tool.field_table_builder'),
      $container->get('product_manager_tool.field_access_validator'),
      $container->get('language_manager'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'product_field_manager_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'core/drupal.ajax';
    $form['#attached']['library'][] = 'system/admin';

    $step = $form_state->get(self::STATE_STEP) ?: self::STEP_SELECT;

    if ($step === self::STEP_CONFIRM) {
      return $this->buildConfirmationStep($form, $form_state);
    }

    if ($step === self::STEP_EDIT) {
      return $this->buildEditStep($form, $form_state);
    }

    return $this->buildSelectionStep($form, $form_state);
  }

  /**
   * Build selection step.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  private function buildSelectionStep(array $form, FormStateInterface $form_state) {
    $form['field_description'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['messages', 'messages--info'],
      ],
      'text' => [
        '#markup' => $this->t('
      <div>
        <strong>Field Manager</strong><br><br>
        <strong>Step 1:</strong> Select the <em>entity type</em> (Product or Variation).<br>
        <strong>Step 2:</strong> Choose the language / translation and optional field settings.<br>
        <strong>Step 3:</strong> Update the missing fields in bulk using the provided actions.<br><br>
        <em>This tool helps you efficiently manage and update fields across multiple entities.</em>
      </div>
    '),
      ],
    ];

    $entity_type = $form_state->getValue('entity_type') ?: self::ENTITY_TYPE_PRODUCT;

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity Type'),
      '#options' => [
        self::ENTITY_TYPE_PRODUCT => $this->t('Product'),
        self::ENTITY_TYPE_VARIATION => $this->t('Product Variation'),
      ],
      '#default_value' => $entity_type,
      '#ajax' => [
        'callback' => '::updateFieldProductList',
        'wrapper' => 'field-product-list-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
      '#required' => TRUE,
    ];

    $form['field_product_container'] = [
      '#type' => 'container',
      '#prefix' => '<div id="field-product-list-wrapper">',
      '#suffix' => '</div>',
    ];

    if ($entity_type === self::ENTITY_TYPE_VARIATION) {
      $this->buildVariationTypeSection($form['field_product_container'], $form_state);
    }
    else {
      $this->buildProductTypeSection($form['field_product_container'], $form_state);
    }

    return $form;
  }

  /**
   * Build edit step.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  private function buildEditStep(array $form, FormStateInterface $form_state) {
    $values = $form_state->get(self::STATE_EDIT_VALUES);
    $entity_type = $values['entity_type'];
    $type_id = $values['type_id'];
    $selected_entities = $values['selected_entities'];
    $include_disabled = $values['include_disabled'] ?? FALSE;

    $form['#prefix'] = '<div id="field-edit-form-wrapper">';
    $form['#suffix'] = '</div>';

    $entity_label = $entity_type === self::ENTITY_TYPE_PRODUCT
      ? $this->t('Products')
      : $this->t('Variations');

    $form['field_edit_description'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--info', 'product-field-update']],
      '#weight' => 0,
      'text' => [
        '#markup' => $this->t('
          <strong>Bulk Field Update - @entity_label</strong><br>
          Set values for empty fields. These values will be applied to all selected entities.<br>
          <em>Only the fields you fill will be updated. Empty fields will be skipped.</em>
        ', ['@entity_label' => $entity_label]),
      ],
    ];

    $field_status = [];
    $all_available_fields = $this->fieldEntityProcessor->getAllAvailableFieldsForEntities(
      $entity_type,
      $type_id,
      $selected_entities,
      $field_status,
      $include_disabled,
      $values['langcode'] ?? NULL
    );
    $form_state->set(self::STATE_FIELD_STATUS, $field_status);

    if (empty($all_available_fields)) {
      $form['no_common_fields'] = [
        '#markup' => '<div class="messages messages--warning">No empty fields found in selected entities.</div>',
      ];

      $form['field_actions'] = $this->buildCancelActions($form_state);
      return $form;
    }

    $this->buildFieldWidgets($form, $form_state, $entity_type, $type_id, $all_available_fields, $field_status, count($selected_entities));
    $this->buildSelectedEntitiesTable($form, $form_state, $entity_type, $selected_entities, $include_disabled);
    $form['field_actions'] = $this->buildEditActions($form_state);

    return $form;
  }

  /**
   * Build confirmation step.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  private function buildConfirmationStep(array $form, FormStateInterface $form_state) {
    $values = $form_state->get(self::STATE_CONFIRM_VALUES);
    $entity_type = $values['entity_type'];
    $type_id = $values['type_id'];
    $selected_entities = $values['selected_entities'];
    $field_values = $values['field_values'];
    $field_overrides = $values['field_overrides'];
    $apply_all_languages = $values['apply_all_languages'] ?? [];
    $langcode = $values['langcode'] ?? NULL;

    $type_entity = $this->loadTypeEntity($entity_type, $type_id);
    $sample_entity = $this->createSampleEntity($entity_type, $type_id);
    $entity_labels = $this->getEntityLabels($entity_type);

    $stats = $this->fieldEntityProcessor->applyFieldValues(
      $entity_type,
      $selected_entities,
      $field_values,
      $field_overrides,
      TRUE,
      $langcode,
      $apply_all_languages
    );

    $this->buildConfirmationHeader($form, $entity_labels);
    $this->buildConfirmationDetails($form, $type_entity, $stats, $field_values, $field_overrides, $sample_entity, $entity_labels, $selected_entities, $langcode, $apply_all_languages);
    $this->buildAffectedEntitiesTable($form, $form_state, $entity_type, $stats, $field_values, $field_overrides, $entity_labels, $apply_all_languages);

    // Check if any entities will actually be updated.
    if ($stats['updated_count'] == 0) {
      $form['no_changes_warning'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        '#weight' => -5,
        'message' => [
          '#markup' => '<strong>' . $this->t('No Changes Will Be Made') . '</strong><br>' .
          $this->t('All selected fields are already filled in the selected entities. No updates will be performed.') .
          '<br>' .
          $this->t('To update these fields, please enable the "Override" option for the fields you want to change.'),
        ],
      ];
    }

    $form['actions'] = $this->buildConfirmationActions($entity_labels, $stats['updated_count']);

    return $form;
  }

  /**
   * Build product type section.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  private function buildProductTypeSection(array &$form, FormStateInterface $form_state) {
    $product_types = $this->getAllProductTypes();

    if (empty($product_types)) {
      $form['no_types'] = [
        '#markup' => '<div class="messages messages--warning">No product types found.</div>',
      ];
      return;
    }

    $default_product_type = $form_state->getValue('field_product_type') ?: array_key_first($product_types);

    $form['field_product_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Product Type'),
      '#options' => $product_types,
      '#default_value' => $default_product_type,
      '#ajax' => [
        'callback' => '::updateFieldProductList',
        'wrapper' => 'field-product-list-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
      '#required' => TRUE,
    ];

    if ($default_product_type) {
      $this->buildEntityListSection($form, $form_state, self::ENTITY_TYPE_PRODUCT, $default_product_type);
    }
  }

  /**
   * Build variation type section.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  private function buildVariationTypeSection(array &$form, FormStateInterface $form_state) {
    $variation_types = $this->getAllVariationTypes();

    if (empty($variation_types)) {
      $form['no_types'] = [
        '#markup' => '<div class="messages messages--warning">No variation types found.</div>',
      ];
      return;
    }

    $default_variation_type = $form_state->getValue('field_product_type') ?: array_key_first($variation_types);

    $form['field_product_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Variation Type'),
      '#options' => $variation_types,
      '#default_value' => $default_variation_type,
      '#ajax' => [
        'callback' => '::updateFieldProductList',
        'wrapper' => 'field-product-list-wrapper',
        'effect' => 'fade',
        'progress' => ['type' => 'fullscreen'],
      ],
      '#required' => TRUE,
    ];

    if ($default_variation_type) {
      $this->buildEntityListSection($form, $form_state, self::ENTITY_TYPE_VARIATION, $default_variation_type);
    }
  }

  /**
   * Build entity list section (unified for products and variations).
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param string $type_id
   *   The bundle/type ID.
   */
  private function buildEntityListSection(array &$form, FormStateInterface $form_state, string $entity_type, string $type_id): void {
    $type_entity = $this->loadTypeEntity($entity_type, $type_id);
    $type_label = $type_entity ? $type_entity->label() : $type_id;

    $include_disabled = $form_state->getValue('include_disabled_fields') ?? FALSE;
    $include_filled = $form_state->getValue('include_filled_fields') ?? ($entity_type === self::ENTITY_TYPE_PRODUCT ? TRUE : FALSE);

    // Get selected language.
    $selected_langcode = $form_state->getValue('language_selector') ?? $this->languageManager->getCurrentLanguage()->getId();
    $language_name = $this->getLanguageName($selected_langcode);

    $form['filter_options'] = $this->buildFilterOptions($include_disabled, $include_filled, $entity_type, $form_state);

    $entities_with_empty = $this->fieldEntityProcessor->getEntitiesWithEmptyFields(
      $entity_type,
      $type_id,
      $include_disabled,
      $include_filled,
      $selected_langcode
    );

    $form['field_section'] = [
      '#type' => 'details',
      '#title' => $this->t('@type - @entities with Empty Fields (@count) - Language: @lang', [
        '@type' => $type_label,
        '@entities' => $entity_type === self::ENTITY_TYPE_PRODUCT ? $this->t('Products') : $this->t('Variations'),
        '@count' => count($entities_with_empty),
        '@lang' => $language_name,
      ]),
      '#open' => TRUE,
    ];

    if (empty($entities_with_empty)) {
      $form['field_section']['no_issues'] = [
        '#markup' => $this->getNoIssuesMessage($entity_type, $type_label, $include_disabled, $language_name),
      ];
      return;
    }

    $this->fieldTableBuilder->buildEntityTable(
      $form['field_section'],
      $entities_with_empty,
      $entity_type,
      TRUE,
      500,
      NULL,
      FALSE,
      $form_state
    );

    $form['field_actions'] = $this->buildUpdateActions($entity_type);
  }

  /**
   * Build filter options.
   *
   * @param bool $include_disabled
   *   Include disabled fields flag.
   * @param bool $include_filled
   *   Include filled fields flag.
   * @param string $entity_type
   *   The entity type.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   Filter options render array.
   */
  private function buildFilterOptions(bool $include_disabled, bool $include_filled, string $entity_type, FormStateInterface $form_state): array {
    $entity_label = $entity_type === self::ENTITY_TYPE_PRODUCT
      ? $this->t('Products')
      : $this->t('variations');

    // Get available languages.
    $languages = $this->languageManager->getLanguages();
    $language_options = [];

    foreach ($languages as $langcode => $language) {
      $language_options[$langcode] = $language->getName();
    }

    // Get the default language value from form state.
    $default_langcode = $form_state->getValue('language_selector')
      ?? $this->languageManager->getCurrentLanguage()->getId();

    return [
      '#type' => 'container',
      '#attributes' => [
        'style' => 'border: 1px solid var(--gin-border-color, #ddd); border-radius: var(--gin-border-m, 10px); padding: 0 15px 0 15px;',
      ],
      'language_selector' => [
        '#type' => 'select',
        '#title' => $this->t('Select Language / Translation'),
        '#options' => $language_options,
        '#default_value' => $default_langcode,
        '#description' => $this->t('Select which language version of entities to update.'),
        '#ajax' => [
          'callback' => '::updateFieldProductList',
          'wrapper' => 'field-product-list-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
        '#weight' => -10,
      ],

      'include_disabled_fields' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Include disabled fields'),
        '#description' => $this->t('Check this to include fields that are disabled by the Disable Field module.'),
        '#default_value' => $include_disabled,
        '#ajax' => [
          'callback' => '::updateFieldProductList',
          'wrapper' => 'field-product-list-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],

      'include_filled_fields' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Include @entities with filled fields', ['@entities' => $entity_label]),
        '#description' => $this->t('Check this to also show @entities that have all fields filled.', ['@entities' => $entity_label]),
        '#default_value' => $include_filled,
        '#ajax' => [
          'callback' => '::updateFieldProductList',
          'wrapper' => 'field-product-list-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],
    ];
  }

  /**
   * Build field widgets section.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $entity_type
   *   The entity type.
   * @param string $type_id
   *   The bundle/type ID.
   * @param array $all_available_fields
   *   All available field names.
   * @param array $field_status
   *   Field status information.
   * @param int $total_entities
   *   Total number of entities.
   */
  private function buildFieldWidgets(array &$form, FormStateInterface $form_state, $entity_type, $type_id, array $all_available_fields, array $field_status, $total_entities) {
    $sample_entity = $this->createSampleEntity($entity_type, $type_id);
    $form_display = $this->loadFormDisplay($entity_type, $type_id);

    if (!$form_display) {
      $form['error'] = [
        '#markup' => '<div class="messages messages--error">Could not load form display.</div>',
      ];
      return;
    }

    // Check if we're restoring values from a "Back" action.
    $restore_values = $form_state->get('restore_field_values');
    $restore_overrides = $form_state->get('restore_field_overrides');
    $restore_all_languages = $form_state->get('restore_apply_all_languages');

    // Get current language from edit values.
    $edit_values = $form_state->get(self::STATE_EDIT_VALUES);
    $current_langcode = $edit_values['langcode'] ?? NULL;

    $form['field_values'] = [
      '#type' => 'details',
      '#title' => $this->t('Field Values'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#parents' => ['field_values'],
      '#weight' => 10,
    ];

    foreach ($all_available_fields as $field_name) {
      $field_definition = $sample_entity->getFieldDefinition($field_name);
      if (!$field_definition) {
        continue;
      }

      $widget = $form_display->getRenderer($field_name);
      if (!$widget) {
        $form['field_values'][$field_name] = [
          '#type' => 'textfield',
          '#title' => $field_definition->getLabel(),
          '#description' => $this->t('Field: @name (No widget configured)', ['@name' => $field_name]),
        ];
        continue;
      }

      // Restore previous values if coming back from confirmation.
      if (!empty($restore_values[$field_name])) {
        try {
          // Set the restored value on the entity for the widget.
          $sample_entity->set($field_name, $restore_values[$field_name]);
          $items = $sample_entity->get($field_name);
        }
        catch (\Exception $e) {
          // If restore fails, use empty items.
          $items = $sample_entity->get($field_name);
        }
        $widget_form = $widget->form($items, $form['field_values'], $form_state);
      }
      else {
        $items = $sample_entity->get($field_name);
        $widget_form = $widget->form($items, $form['field_values'], $form_state);
      }

      $field_container = $this->buildFieldContainer(
        $sample_entity,
        $field_name,
        $field_definition,
        $widget,
        $form['field_values'],
        $form_state,
        $field_status,
        $total_entities,
        $widget_form,
        $restore_overrides,
        $restore_all_languages,
        $current_langcode
      );

      $form['field_values'][$field_name . '_container'] = $field_container;
    }

    // Clear restore values after use.
    if (!empty($restore_values)) {
      $form_state->set('restore_field_values', NULL);
      $form_state->set('restore_field_overrides', NULL);
      $form_state->set('restore_apply_all_languages', NULL);
    }
  }

  /**
   * Build individual field container with widget and override checkbox.
   *
   * @param mixed $sample_entity
   *   Sample entity for widget building.
   * @param string $field_name
   *   The field name.
   * @param mixed $field_definition
   *   The field definition.
   * @param mixed $widget
   *   The field widget.
   * @param array $parent_form
   *   Parent form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $field_status
   *   Field status information.
   * @param int $total_entities
   *   Total number of entities.
   * @param array $widget_form
   *   Pre-built widget form (optional, for restore).
   * @param array|null $restore_overrides
   *   Restore override values (optional).
   * @param array|null $restore_all_languages
   *   Restore apply all languages values (optional).
   * @param string|null $current_langcode
   *   The currently selected language code.
   *
   * @return array
   *   Field container render array.
   */
  private function buildFieldContainer(
    mixed $sample_entity,
    string $field_name,
    mixed $field_definition,
    mixed $widget,
    array &$parent_form,
    FormStateInterface $form_state,
    array $field_status,
    int $total_entities,
    ?array $widget_form = NULL,
    ?array $restore_overrides = NULL,
    ?array $restore_all_languages = NULL,
    ?string $current_langcode = NULL,
  ): array {
    $field_container = [
      '#type' => 'container',
      '#attributes' => [
        'style' => 'border: 1px solid var(--gin-border-color, #ddd); border-radius: var(--gin-border-m, 10px); padding: 15px; margin-bottom: 15px;',
      ],
      '#tree' => TRUE,
    ];

    if (isset($field_status[$field_name])) {
      $field_container['info'] = $this->buildFieldStatusInfo(
        $field_name,
        $field_definition,
        $field_status[$field_name],
        $total_entities,
        $sample_entity,
        $current_langcode
      );
    }

    // Use pre-built widget form if provided (for restore).
    if ($widget_form === NULL) {
      $items = $sample_entity->get($field_name);
      $widget_form = $widget->form($items, $parent_form, $form_state);
    }

    $this->makeWidgetOptional($widget_form);

    foreach ($widget_form as $key => $value) {
      if (strpos($key, '#') !== 0) {
        $field_container[$key] = $value;
      }
    }

    // Restore override value if available.
    $override_default = FALSE;
    if (!empty($restore_overrides[$field_name])) {
      $override_default = TRUE;
    }

    $field_container['override_' . $field_name] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Override already filled fields for @label', ['@label' => $field_definition->getLabel()]),
      '#description' => $this->t('If checked, existing values for this field will be overwritten with the new ones.'),
      '#default_value' => $override_default,
      '#weight' => 100,
    ];

    // Restore apply all languages value if available.
    $apply_all_default = FALSE;
    if (!empty($restore_all_languages[$field_name])) {
      $apply_all_default = TRUE;
    }

    if ($sample_entity->isTranslatable()) {
      $field_container['apply_all_languages_' . $field_name] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Apply to all translations for @label', ['@label' => $field_definition->getLabel()]),
        '#description' => $this->t('If checked, this field will be updated in all available translations of the entity.'),
        '#default_value' => $apply_all_default,
        '#weight' => 101,
      ];
    }

    return $field_container;
  }

  /**
   * Build field status info message with translation details.
   *
   * @param string $field_name
   *   The field name.
   * @param mixed $field_definition
   *   The field definition.
   * @param array $status_data
   *   Status data (empty_count, total_count, by_language).
   * @param int $total_entities
   *   Total number of entities.
   * @param mixed $sample_entity
   *   Sample entity for checking if field is disabled.
   * @param string|null $current_langcode
   *   The currently selected language code.
   *
   * @return array
   *   Info message render array.
   */
  private function buildFieldStatusInfo(
    string $field_name,
    mixed $field_definition,
    array $status_data,
    int $total_entities,
    mixed $sample_entity,
    ?string $current_langcode = NULL,
  ): array {
    $empty_count = $status_data['empty_count'];
    $filled_count = $total_entities - $empty_count;
    $field_label = $field_definition->getLabel();
    $by_language = $status_data['by_language'] ?? [];

    $info_message = '';

    if ($this->fieldEntityProcessor->isFieldDisabled($sample_entity, $field_name)) {
      $info_message .= '<em>' . $this->t('This is a disabled field.') . '</em><br><br>';
    }

    $current_language_name = $this->getLanguageName($current_langcode);

    // Main status for the selected language.
    if ($current_langcode && isset($by_language[$current_langcode])) {
      $current_counts = $by_language[$current_langcode];
      $current_empty = $current_counts['empty'];
      $current_filled = $current_counts['filled'];
      $current_total = $current_empty + $current_filled;

      $info_message .= '<strong>' . $this->t('@label (Selected Language: @lang):', [
        '@label' => $field_label,
        '@lang' => $current_language_name,
      ]) . '</strong> ';

      if ($current_empty === 0 && $current_filled > 0) {
        $info_message .= $this->formatPlural(
          $current_filled,
          'All 1 entity already has this field filled.',
          'All @count entities already have this field filled.'
        );
      }
      elseif ($current_filled === 0 && $current_empty > 0) {
        $info_message .= $this->formatPlural(
          $current_empty,
          'Empty in 1 entity.',
          'Empty in all @count entities.'
              );
      }
      elseif ($current_empty === 0 && $current_filled === 0) {
        $info_message .= $this->t('No entities have this translation.');
      }
      else {
        $info_message .= $this->t('Empty in @empty of @total entities. @filled entity/entities already filled.', [
          '@empty' => $current_empty,
          '@total' => $current_total,
          '@filled' => $current_filled,
        ]);
      }
    }
    else {
      // Fallback to original message if no language context.
      if ($empty_count === $total_entities) {
        $info_message .= $this->t(
          '<strong>@label:</strong> Empty in all @total entity(s).',
          ['@label' => $field_label, '@total' => $total_entities]
        );
      }
      elseif ($empty_count === 0) {
        $info_message .= $this->t(
          '<strong>@label:</strong> All @total entity(s) already have this field filled.',
          ['@label' => $field_label, '@total' => $total_entities]
              );
      }
      else {
        $info_message .= $this->t(
          '<strong>@label:</strong> Empty in @empty of @total entities. @filled entity/entities already have this field filled.',
          ['@label' => $field_label, '@empty' => $empty_count, '@total' => $total_entities, '@filled' => $filled_count]
              );
      }
    }

    // Add total across all translations if entity is translatable.
    if (!empty($by_language) && $sample_entity->isTranslatable() && count($by_language) > 0) {
      // Calculate totals across ALL translations.
      $total_empty_all_langs = 0;
      $total_filled_all_langs = 0;

      foreach ($by_language as $counts) {
        $total_empty_all_langs += $counts['empty'];
        $total_filled_all_langs += $counts['filled'];
      }

      $info_message .= '<br><br><strong>' . $this->t('Total Across All Translations:') . '</strong> ';

      if ($total_empty_all_langs === 0 && $total_filled_all_langs > 0) {
        $info_message .= $this->t('All @count field(s) are filled across @lang_count translation(s).', [
          '@count' => $total_filled_all_langs,
          '@lang_count' => count($by_language),
        ]);
      }
      elseif ($total_filled_all_langs === 0 && $total_empty_all_langs > 0) {
        $info_message .= $this->t('All @count field(s) are empty across @lang_count translation(s).', [
          '@count' => $total_empty_all_langs,
          '@lang_count' => count($by_language),
        ]);
      }
      else {
        $info_message .= $this->t('@empty empty, @filled filled across @lang_count translation(s).', [
          '@empty' => $total_empty_all_langs,
          '@filled' => $total_filled_all_langs,
          '@lang_count' => count($by_language),
        ]);
      }
    }

    // Add per-language breakdown if available.
    if (!empty($by_language) && $sample_entity->isTranslatable()) {
      if (count($by_language) > 1) {
        // Multiple languages - show breakdown.
        $info_message .= '<br><br><strong>' . $this->t('Other Available Translations:') . '</strong><ul>';

        foreach ($by_language as $langcode => $counts) {
          // Skip the current language as it's already shown above.
          if ($langcode === $current_langcode) {
            continue;
          }

          $language_name = $this->getLanguageName($langcode);
          $total_lang = $counts['empty'] + $counts['filled'];

          $info_message .= '<li><strong>' . Html::escape($language_name) . ':</strong> ';

          if ($counts['empty'] === 0 && $counts['filled'] > 0) {
            $info_message .= $this->formatPlural(
              $counts['filled'],
              '1 entity filled',
              'All @count entities filled'
            );
          }
          elseif ($counts['filled'] === 0 && $counts['empty'] > 0) {
            $info_message .= $this->formatPlural(
              $counts['empty'],
              '1 entity empty',
              'All @count entities empty'
                      );
          }
          elseif ($counts['empty'] === 0 && $counts['filled'] === 0) {
            $info_message .= $this->t('No entities have this translation');
          }
          else {
            $info_message .= $this->t('@empty empty, @filled filled (total: @total)', [
              '@empty' => $counts['empty'],
              '@filled' => $counts['filled'],
              '@total' => $total_lang,
            ]);
          }

          $info_message .= '</li>';
        }
        $info_message .= '</ul>';
      }
      elseif (count($by_language) === 1) {
        $info_message .= '<br><br><em>' . $this->t('No other translations available for this field.') . '</em>';
      }
    }

    return [
      '#markup' => '<div class="messages messages--warning">' . $info_message . '</div>',
      '#weight' => -10,
    ];
  }

  /**
   * Build selected entities table.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $entity_type
   *   The entity type.
   * @param array $selected_entities
   *   Selected entity IDs.
   * @param bool $include_disabled
   *   Include disabled fields flag.
   */
  private function buildSelectedEntitiesTable(array &$form, FormStateInterface $form_state, $entity_type, array $selected_entities, $include_disabled) {
    $storage_type = $entity_type === self::ENTITY_TYPE_PRODUCT ? self::STORAGE_PRODUCT : self::STORAGE_VARIATION;
    $entities = $this->entityTypeManager->getStorage($storage_type)->loadMultiple($selected_entities);

    // Get language from edit values.
    $edit_values = $form_state->get(self::STATE_EDIT_VALUES);
    $langcode = $edit_values['langcode'] ?? NULL;

    $entities_with_fields = [];
    foreach ($entities as $entity) {
      $translated_entity = $entity;
      if ($langcode && $entity->isTranslatable() && $entity->hasTranslation($langcode)) {
        $translated_entity = $entity->getTranslation($langcode);
      }

      $empty_fields = $this->fieldEntityProcessor->getEntityFields($translated_entity, $include_disabled, TRUE);

      $entities_with_fields[] = [
        $entity_type => $translated_entity,
        'empty_fields' => $empty_fields,
        'langcode' => $langcode,
        'is_translation' => $langcode && $langcode !== $entity->language()->getId(),
        // Add empty languages_to_update for edit screen (no dropdown needed).
        'languages_to_update' => [],
      ];
    }

    $entity_label = $entity_type === self::ENTITY_TYPE_PRODUCT ? $this->t('Products') : $this->t('Variations');

    $form['selected_entities_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Selected @entities for Update (@count)', [
        '@entities' => $entity_label,
        '@count' => count($entities_with_fields),
      ]),
      '#open' => FALSE,
      '#weight' => 20,
    ];

    $this->fieldTableBuilder->buildEntityTable(
      $form['selected_entities_section'],
      $entities_with_fields,
      $entity_type,
      FALSE,
      500,
      NULL,
      FALSE,
      $form_state
    );
  }

  /**
   * Build confirmation header.
   *
   * @param array $form
   *   The form array.
   * @param array $entity_labels
   *   Entity labels array.
   */
  private function buildConfirmationHeader(array &$form, array $entity_labels) {
    $form['confirmation'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--warning']],
      'title' => [
        '#markup' => '<h2>' . $this->t('Confirm Field Update') . '</h2>',
      ],
    ];
  }

  /**
   * Build confirmation details.
   *
   * @param array $form
   *   The form array.
   * @param mixed $type_entity
   *   Type entity (ProductType or ProductVariationType).
   * @param array $stats
   *   Statistics from dry run.
   * @param array $field_values
   *   Field values to apply.
   * @param array $field_overrides
   *   Override flags.
   * @param mixed $sample_entity
   *   Sample entity for getting field labels.
   * @param array $entity_labels
   *   Entity labels array.
   * @param array $selected_entities
   *   Array of selected entity IDs.
   * @param string|null $langcode
   *   The language code.
   * @param array $apply_all_languages
   *   Apply all languages flags.
   */
  private function buildConfirmationDetails(array &$form, $type_entity, array $stats, array $field_values, array $field_overrides, $sample_entity, array $entity_labels, array $selected_entities, $langcode = NULL, array $apply_all_languages = []) {
    $language_name = $this->getLanguageName($langcode);

    $preview_stats = $this->buildPreviewStats($stats, $entity_labels);
    $has_override = $this->hasOverrideEnabled($field_overrides);

    if ($has_override) {
      $form['override_warning'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        '#weight' => -10,
        'message' => [
          '#markup' => '<strong>' . $this->t('WARNING: Override Mode Active') . '</strong><br>' .
          $this->t('Fields with override ON will overwrite existing values. This action cannot be undone!'),
        ],
      ];
    }

    // Calculate actual fields that will be filled (not skipped).
    $actual_fields_to_fill = $stats['total_field_updates'];

    // Calculate total SELECTED translations (all available).
    $total_selected_translations = 0;
    if (!empty($stats['languages_selected'])) {
      foreach ($stats['languages_selected'] as $count) {
        $total_selected_translations += $count;
      }
    }

    // Calculate actual translations that will be updated (after dry run).
    $total_translations_to_update = 0;
    if (!empty($stats['languages_updated'])) {
      foreach ($stats['languages_updated'] as $count) {
        $total_translations_to_update += $count;
      }
    }

    // Build the main list items.
    $list_items = [
      [
        '#markup' => '<strong>' . $this->t('Type:') . '</strong> ' .
        Html::escape($type_entity->label()),
      ],
      [
        '#markup' => '<strong>' . $this->t('Language:') . '</strong> ' .
        Html::escape($language_name),
      ],
      [
        '#markup' => '<strong>' . $this->t('@entities selected:', ['@entities' => $entity_labels['plural']]) . '</strong> ' .
        (int) count($selected_entities),
      ],
      [
        '#markup' => '<strong>' . $this->t('@entities will be updated:', ['@entities' => $entity_labels['plural']]) . '</strong> ' .
        (int) $stats['updated_count'],
      ],
    ];

    // Show translations info if apply all languages is used.
    if ($total_selected_translations > 0) {
      $list_items[] = [
        '#markup' => '<strong>' . $this->t('Translations selected:') . '</strong> ' .
        (int) $total_selected_translations,
      ];

      $list_items[] = [
        '#markup' => '<strong>' . $this->t('Translations will be updated:') . '</strong> ' .
        (int) $total_translations_to_update,
      ];
    }

    $list_items[] = [
      '#markup' => '<strong>' . $this->t('Fields to be filled:') . '</strong> ' .
      (int) $actual_fields_to_fill,
    ];

    // Build suffix.
    $suffix = $preview_stats;

    if ($actual_fields_to_fill > 0) {
      $fields_list = $this->buildFieldsList($field_values, $field_overrides, $sample_entity, $apply_all_languages);
      $suffix .= '<p><strong>' . $this->t('Fields that will be updated:') . '</strong></p>' . $fields_list;
    }

    // Show languages breakdown with update and skip counts.
    if (!empty($stats['languages_updated']) || !empty($stats['languages_skipped'])) {
      $language_list = $this->buildLanguagesUpdatedInfo(
        $stats['languages_updated'] ?? [],
        $stats['languages_skipped'] ?? [],
        $stats['languages_field_updates'] ?? []
      );
      $suffix .= '<p><strong>' . $this->t('Translation Status:') . '</strong></p>' . $language_list;
    }

    $suffix .= '</div>';

    $form['confirmation']['details'] = [
      '#theme' => 'item_list',
      '#list_type' => 'ul',
      '#title' => $this->t('You are about to update the following:'),
      '#items' => $list_items,
      '#prefix' => '<div class="confirmation-details">',
      '#suffix' => $suffix,
    ];
  }

  /**
   * Build affected entities table.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $entity_type
   *   The entity type.
   * @param array $stats
   *   Statistics array.
   * @param array $field_values
   *   Field values.
   * @param array $field_overrides
   *   Override flags.
   * @param array $entity_labels
   *   Entity labels array.
   * @param array $apply_all_languages
   *   Apply all languages flags.
   */
  private function buildAffectedEntitiesTable(
    array &$form,
    FormStateInterface $form_state,
    string $entity_type,
    array $stats,
    array $field_values,
    array $field_overrides,
    array $entity_labels,
    array $apply_all_languages = [],
  ): void {
    if (empty($stats['updated_entity_ids'])) {
      return;
    }

    $storage_type = $entity_type === self::ENTITY_TYPE_PRODUCT ? self::STORAGE_PRODUCT : self::STORAGE_VARIATION;
    $entities = $this->entityTypeManager->getStorage($storage_type)->loadMultiple($stats['updated_entity_ids']);

    // Get language from confirm values.
    $confirm_values = $form_state->get(self::STATE_CONFIRM_VALUES);
    $langcode = $confirm_values['langcode'] ?? NULL;

    $entities_with_fields = [];
    foreach ($entities as $entity) {
      $fields_to_fill = [];

      // Check which fields will actually be filled for this entity.
      foreach ($field_values as $field_name => $field_value) {
        if (!$entity->hasField($field_name)) {
          continue;
        }

        $is_apply_all = !empty($apply_all_languages[$field_name]);
        $override_enabled = !empty($field_overrides[$field_name]);

        $will_update_this_field = FALSE;

        // Determine if this field will be processed.
        if ($is_apply_all && $entity->isTranslatable()) {
          // Apply all languages - check all translations.
          $translation_languages = $entity->getTranslationLanguages();
          foreach ($translation_languages as $lang) {
            $lang_id = $lang->getId();
            if ($entity->hasTranslation($lang_id)) {
              $translated = $entity->getTranslation($lang_id);
              $field_is_empty = $this->fieldEntityProcessor->isFieldEmpty($translated->get($field_name));

              // Update if empty OR override is enabled.
              if ($field_is_empty || $override_enabled) {
                $will_update_this_field = TRUE;
                break;
              }
            }
          }
        }
        else {
          // Single language mode.
          $check_entity = $entity;
          if ($langcode && $entity->isTranslatable() && $entity->hasTranslation($langcode)) {
            $check_entity = $entity->getTranslation($langcode);
          }

          $field_is_empty = $this->fieldEntityProcessor->isFieldEmpty($check_entity->get($field_name));

          // Update if empty OR override is enabled.
          if ($field_is_empty || $override_enabled) {
            $will_update_this_field = TRUE;
          }
        }

        if ($will_update_this_field) {
          $fields_to_fill[] = $field_name;
        }
      }

      if (!empty($fields_to_fill)) {
        // Get the entity in the display language.
        $display_entity = $entity;
        if ($langcode && $entity->isTranslatable() && $entity->hasTranslation($langcode)) {
          $display_entity = $entity->getTranslation($langcode);
        }

        // Get all languages that will be updated for this entity.
        $languages_to_update = $this->getLanguagesForEntity($entity, $langcode, $apply_all_languages, $fields_to_fill);

        $entities_with_fields[] = [
          $entity_type => $display_entity,
          'empty_fields' => $fields_to_fill,
          'langcode' => $langcode,
          'is_translation' => $langcode && $langcode !== $entity->language()->getId(),
          'languages_to_update' => $languages_to_update,
        ];
      }
    }

    $form['affected_entities'] = [
      '#type' => 'details',
      '#title' => $this->t('@entities that will be updated (@count)', [
        '@entities' => $entity_labels['plural'],
        '@count' => count($entities_with_fields),
      ]),
      '#collapsed' => TRUE,
    ];

    $this->fieldTableBuilder->buildEntityTable(
      $form['affected_entities'],
      $entities_with_fields,
      $entity_type,
      FALSE,
      500,
      $this->t('Fields to Update'),
      // Show languages to update column.
      TRUE,
      $form_state
    );
  }

  /**
   * Get languages that will be updated for an entity.
   *
   * @param mixed $entity
   *   The entity.
   * @param string|null $langcode
   *   Primary language code.
   * @param array $apply_all_languages
   *   Fields marked for all languages.
   * @param array $fields_to_fill
   *   Fields that will be filled.
   *
   * @return array
   *   Array of language codes.
   */
  private function getLanguagesForEntity($entity, $langcode, array $apply_all_languages, array $fields_to_fill) {
    $languages = [];

    if (!$entity->isTranslatable()) {
      return [$langcode];
    }

    // Check if any field to fill is marked for all languages.
    $has_all_lang_field = FALSE;
    foreach ($fields_to_fill as $field_name) {
      if (!empty($apply_all_languages[$field_name])) {
        $has_all_lang_field = TRUE;
        break;
      }
    }

    if ($has_all_lang_field) {
      $translation_languages = $entity->getTranslationLanguages();
      foreach ($translation_languages as $lang) {
        $languages[] = $lang->getId();
      }
    }
    else {
      $languages[] = $langcode;
    }

    return $languages;
  }

  /**
   * Build preview statistics.
   *
   * @param array $stats
   *   Statistics array.
   * @param array $entity_labels
   *   Entity labels array.
   *
   * @return string
   *   HTML preview stats.
   */
  private function buildPreviewStats(array $stats, array $entity_labels) {
    $preview_stats = '<div>';
    $preview_stats .= '<h3>' . $this->t('Update Preview') . '</h3>';
    $preview_stats .= '<ul>';

    if ($stats['updated_count'] > 0) {
      $preview_stats .= '<li><strong>' . $this->formatPlural(
          $stats['updated_count'],
          '1 @entity will be updated.',
          '@count @entities will be updated.',
          ['@entity' => $entity_labels['singular'], '@entities' => $entity_labels['plural']]
        ) . '</strong></li>';
    }

    if ($stats['total_field_updates'] > 0) {
      $preview_stats .= '<li>' . $this->formatPlural(
          $stats['total_field_updates'],
          '1 field value will be set.',
          '@count field values will be set.'
        ) . '</li>';
    }

    if ($stats['skipped_count'] > 0) {
      $preview_stats .= '<li>' . $this->formatPlural(
          $stats['skipped_count'],
          '1 already-filled field will be skipped (not overridden).',
          '@count already-filled fields will be skipped (not overridden).'
        ) . '</li>';
    }

    $preview_stats .= '</ul></div>';

    return $preview_stats;
  }

  /**
   * Build fields list with override status.
   *
   * @param array $field_values
   *   Field values array.
   * @param array $field_overrides
   *   Override flags array.
   * @param mixed $sample_entity
   *   Sample entity for getting field labels.
   * @param array $apply_all_languages
   *   Apply all languages flags.
   *
   * @return string
   *   Rendered HTML fields list.
   *
   * @throws \Exception
   */
  private function buildFieldsList(array $field_values, array $field_overrides, $sample_entity, array $apply_all_languages = []) {
    $items = [];

    foreach ($field_values as $field_name => $value) {
      if ($this->fieldValueProcessor->isFieldValueEmpty($value)) {
        continue;
      }

      $field_definition = $sample_entity->getFieldDefinition($field_name);
      if (!$field_definition) {
        continue;
      }

      $field_label = $field_definition->getLabel();
      $override_status = !empty($field_overrides[$field_name]) ? $this->t('ON') : $this->t('OFF');
      $all_lang_status = !empty($apply_all_languages[$field_name]) ? $this->t('ON') : $this->t('OFF');

      // Build status text.
      $status_parts = [];
      $status_parts[] = '<strong>' . $this->t('Override:') . ' ' . $override_status . '</strong>';

      if (!empty($apply_all_languages[$field_name])) {
        $status_parts[] = '<strong>' . $this->t('Apply All Languages:') . ' ' . $all_lang_status . '</strong>';
      }

      // Use Html::escape for all user-provided data.
      $items[] = [
        '#markup' => '<strong>' . Html::escape($field_label) . '</strong> (' .
        Html::escape($field_name) . ') - ' .
        implode(', ', $status_parts),
      ];
    }

    if (empty($items)) {
      return '';
    }

    $list = [
      '#theme' => 'item_list',
      '#list_type' => 'ul',
      '#items' => $items,
    ];

    return (string) $this->renderer->render($list);
  }

  /**
   * Build languages updated info with skip details and field counts.
   *
   * @param array $languages_updated
   *   Array of language codes and update counts.
   * @param array $languages_skipped
   *   Array of language codes and skip counts.
   * @param array $languages_field_updates
   *   Array of language codes and field update counts.
   *
   * @return string
   *   HTML info about languages.
   *
   * @throws \Exception
   */
  private function buildLanguagesUpdatedInfo(array $languages_updated, array $languages_skipped, array $languages_field_updates = []) {
    if (empty($languages_updated) && empty($languages_skipped)) {
      return '';
    }

    $items = [];

    // Combine both arrays to show all languages.
    $all_langcodes = array_unique(array_merge(
      array_keys($languages_updated),
      array_keys($languages_skipped)
    ));

    foreach ($all_langcodes as $langcode) {
      $language_name = $this->getLanguageName($langcode);

      $update_count = $languages_updated[$langcode] ?? 0;
      $skip_count = $languages_skipped[$langcode] ?? 0;
      $field_update_count = $languages_field_updates[$langcode] ?? 0;

      $status_parts = [];

      if ($update_count > 0) {
        $update_text = '<span>' .
          (int) $update_count . ' ' .
          $this->formatPlural($update_count, 'entity', 'entities') .
          ' will be updated';

        // Add field count if available.
        if ($field_update_count > 0) {
          $update_text .= ' (' . (int) $field_update_count . ' ' .
            $this->formatPlural($field_update_count, 'field', 'fields') . ')';
        }

        $update_text .= '</span>';
        $status_parts[] = $update_text;
      }

      if ($skip_count > 0) {
        $status_parts[] = '<span>' .
          (int) $skip_count . ' ' .
          $this->formatPlural($skip_count, 'entity', 'entities') .
          ' will be skipped</span>';
      }

      $items[] = [
        '#markup' => '<strong>' . Html::escape($language_name) . '</strong>: ' .
        implode(', ', $status_parts),
      ];
    }

    if (empty($items)) {
      return '';
    }

    $list = [
      '#theme' => 'item_list',
      '#list_type' => 'ul',
      '#items' => $items,
    ];

    return (string) $this->renderer->render($list);
  }

  /**
   * Check if any override is enabled.
   *
   * @param array $field_overrides
   *   Override flags array.
   *
   * @return bool
   *   TRUE if any override is enabled.
   */
  private function hasOverrideEnabled(array $field_overrides) {
    foreach ($field_overrides as $override_value) {
      if (!empty($override_value)) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Build update actions for entity list.
   *
   * @param string $entity_type
   *   The entity type.
   *
   * @return array
   *   Actions render array.
   */
  private function buildUpdateActions($entity_type) {
    $label = $entity_type === self::ENTITY_TYPE_PRODUCT ? $this->t('Products') : $this->t('Variations');

    return [
      '#type' => 'actions',
      'update_fields' => [
        '#type' => 'submit',
        '#value' => $this->t('Update Selected @entities', ['@entities' => $label]),
        '#button_type' => 'primary',
        '#submit' => ['::prepareFieldEdit'],
        '#ajax' => [
          'callback' => '::ajaxPrepareFieldEdit',
          'wrapper' => 'field-main-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],
    ];
  }

  /**
   * Build edit actions.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   Actions render array.
   */
  private function buildEditActions(FormStateInterface $form_state) {
    return [
      '#type' => 'actions',
      '#weight' => 30,
      'continue' => [
        '#type' => 'submit',
        '#value' => $this->t('Continue to Confirmation'),
        '#button_type' => 'primary',
        '#submit' => ['::prepareFieldConfirmation'],
        '#ajax' => [
          'callback' => '::ajaxPrepareFieldConfirmation',
          'wrapper' => 'field-main-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],
      'back' => [
        '#type' => 'submit',
        '#value' => $this->t('Back to Selection'),
        '#submit' => ['::backToSelection'],
        '#limit_validation_errors' => [],
        '#attributes' => ['class' => ['button']],
        '#ajax' => [
          'callback' => '::ajaxBackToSelection',
          'wrapper' => 'field-main-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],
      'cancel' => [
        '#type' => 'submit',
        '#value' => $this->t('Cancel'),
        '#submit' => ['::cancelFieldEdit'],
        '#limit_validation_errors' => [],
        '#attributes' => ['class' => ['button--danger']],
        '#ajax' => [
          'callback' => '::ajaxCancelFieldEdit',
          'wrapper' => 'field-main-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],
    ];
  }

  /**
   * Build cancel actions.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   Actions render array.
   */
  private function buildCancelActions(FormStateInterface $form_state) {
    return [
      '#type' => 'actions',
      'cancel' => [
        '#type' => 'submit',
        '#value' => $this->t('Cancel'),
        '#submit' => ['::cancelFieldEdit'],
        '#limit_validation_errors' => [],
        '#ajax' => [
          'callback' => '::ajaxCancelFieldEdit',
          'wrapper' => 'field-main-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],
    ];
  }

  /**
   * Build confirmation actions.
   *
   * @param array $entity_labels
   *   Entity labels array.
   * @param int $updated_count
   *   Number of entities that will be updated.
   *
   * @return array
   *   Actions render array.
   */
  private function buildConfirmationActions(array $entity_labels, $updated_count = 0) {
    $actions = [
      '#type' => 'actions',
      'confirm' => [
        '#type' => 'submit',
        '#value' => $this->t('Yes, Update All @entities', ['@entities' => $entity_labels['plural']]),
        '#button_type' => 'primary',
        '#submit' => ['::confirmFieldUpdate'],
        '#attributes' => [
          'class' => ['button--danger', 'confirm-countdown-button'],
          'id' => 'confirm-update-button',
        ],
      ],
      'back' => [
        '#type' => 'submit',
        '#value' => $this->t('Back to Edit'),
        '#submit' => ['::backToFieldEdit'],
        '#limit_validation_errors' => [],
        '#attributes' => ['class' => ['button']],
        '#ajax' => [
          'callback' => '::ajaxBackToFieldEdit',
          'wrapper' => 'field-main-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],
      'cancel' => [
        '#type' => 'submit',
        '#value' => $this->t('Cancel All'),
        '#submit' => ['::cancelFieldEdit'],
        '#limit_validation_errors' => [],
        '#attributes' => ['class' => ['button--danger']],
        '#ajax' => [
          'callback' => '::ajaxCancelFieldEdit',
          'wrapper' => 'field-main-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
      ],
    ];

    // Disable confirm button if no entities will be updated.
    if ($updated_count == 0) {
      $actions['confirm']['#disabled'] = TRUE;
      $actions['confirm']['#value'] = $this->t('No Changes to Apply');
      $actions['confirm']['#attributes']['class'] = ['button--disabled'];
      // Remove countdown.
      unset($actions['confirm']['#attributes']['id']);
    }

    return $actions;
  }

  /**
   * Go back to field edit step.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function backToFieldEdit(array &$form, FormStateInterface $form_state) {
    // Get confirmation values to restore edit state.
    $confirm_values = $form_state->get(self::STATE_CONFIRM_VALUES);

    if (!empty($confirm_values)) {
      // Restore edit state with all previous data.
      $form_state->set(self::STATE_EDIT_VALUES, [
        'entity_type' => $confirm_values['entity_type'],
        'type_id' => $confirm_values['type_id'],
        'selected_entities' => $confirm_values['selected_entities'],
        'include_disabled' => $confirm_values['include_disabled'] ?? FALSE,
        'langcode' => $confirm_values['langcode'] ?? NULL,
      ]);

      // Keep field values and overrides for restoration.
      // These will be used once by buildFieldWidgets and then cleared.
      $form_state->set('restore_field_values', $confirm_values['field_values']);
      $form_state->set('restore_field_overrides', $confirm_values['field_overrides']);
      $form_state->set('restore_apply_all_languages', $confirm_values['apply_all_languages'] ?? []);
    }

    // Clear confirmation step data.
    $form_state->set(self::STATE_CONFIRM_VALUES, NULL);

    // Go back to edit step.
    $form_state->set(self::STATE_STEP, self::STEP_EDIT);
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for back to edit action.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function ajaxBackToFieldEdit(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * Back to selection handler - returns to entity selection screen.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function backToSelection(array &$form, FormStateInterface $form_state) {
    // Get the current edit values to restore entity type and type_id.
    $edit_values = $form_state->get(self::STATE_EDIT_VALUES);

    // Restore entity_type, type_id, langcode, and include_disabled.
    $entity_type = $edit_values['entity_type'] ?? self::ENTITY_TYPE_PRODUCT;
    $type_id = $edit_values['type_id'] ?? NULL;
    $langcode = $edit_values['langcode'] ?? $this->languageManager->getCurrentLanguage()->getId();
    $include_disabled = $edit_values['include_disabled'] ?? FALSE;

    // Restore form values.
    $form_state->setValue('entity_type', $entity_type);
    if ($type_id) {
      $form_state->setValue('field_product_type', $type_id);
    }
    $form_state->setValue('language_selector', $langcode);
    $form_state->setValue('include_disabled_fields', $include_disabled);

    // Ensure checkbox selections are preserved for each unique combination.
    $selection_key = $this->getSelectionKey($entity_type, $type_id, $langcode, $include_disabled);
    $stored_selections = $form_state->get($selection_key) ?? [];

    if (!empty($stored_selections)) {
      // Restore to generic key so checkboxes render correctly.
      $form_state->set(self::STATE_SELECTED_ENTITIES, $stored_selections);
    }

    // Clear edit step data.
    $form_state->set(self::STATE_EDIT_VALUES, NULL);
    $form_state->set(self::STATE_FIELD_STATUS, NULL);

    // Go back to selection step.
    $form_state->set(self::STATE_STEP, self::STEP_SELECT);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Get a combination-specific key for storing checkbox selections.
   *
   * This ensures that checkbox selections are preserved separately for each
   * combination of entity_type + type_id + langcode + include_disabled.
   *
   * @param string $entity_type
   *   The entity type (product/variation).
   * @param string $type_id
   *   The product/variation type ID.
   * @param string $langcode
   *   The language code.
   * @param bool $include_disabled
   *   Whether disabled fields are included.
   *
   * @return string
   *   A unique key for this combination.
   */
  private function getSelectionKey($entity_type, $type_id, $langcode, $include_disabled) {
    return implode('_', [
      self::STATE_SELECTED_ENTITIES,
      $entity_type,
      $type_id,
      $langcode,
      $include_disabled ? 'with_disabled' : 'no_disabled',
    ]);
  }

  /**
   * AJAX callback for back to selection action.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function ajaxBackToSelection(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * Make widget optional (remove required flags).
   *
   * @param array $widget_form
   *   Widget form array.
   */
  private function makeWidgetOptional(array &$widget_form) {
    if (isset($widget_form['#required'])) {
      $widget_form['#required'] = FALSE;
    }
    if (isset($widget_form['widget']['#required'])) {
      $widget_form['widget']['#required'] = FALSE;
    }
    if (isset($widget_form['widget'][0]['#required'])) {
      $widget_form['widget'][0]['#required'] = FALSE;
    }
    if (isset($widget_form['widget'][0]['value']['#required'])) {
      $widget_form['widget'][0]['value']['#required'] = FALSE;
    }
  }

  /**
   * Get entity labels (singular and plural).
   *
   * @param string $entity_type
   *   The entity type.
   *
   * @return array
   *   Array with 'singular' and 'plural' keys.
   */
  private function getEntityLabels($entity_type) {
    if ($entity_type === self::ENTITY_TYPE_PRODUCT) {
      return [
        'singular' => $this->t('product'),
        'plural' => $this->t('Products'),
      ];
    }

    return [
      'singular' => $this->t('variation'),
      'plural' => $this->t('Variations'),
    ];
  }

  /**
   * Get no issues message.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $type_label
   *   The type label.
   * @param bool $include_disabled
   *   Include disabled flag.
   * @param string $language_name
   *   The language name.
   *
   * @return string
   *   HTML message.
   */
  private function getNoIssuesMessage($entity_type, $type_label, $include_disabled, $language_name) {
    $entity_label = $entity_type === self::ENTITY_TYPE_PRODUCT ? $this->t('products') : $this->t('variations');

    $message = $include_disabled
      ? $this->t('All @type @entities have their fields (including disabled) filled.',
        ['@type' => $type_label, '@entities' => $entity_label])
      : $this->t('All @type @entities have their non-disabled fields filled.',
        ['@type' => $type_label, '@entities' => $entity_label]);

    return '<div class="messages messages--status">' . $message . '</div>';
  }

  /**
   * Get language name from language code.
   *
   * @param string|null $langcode
   *   The language code.
   *
   * @return string
   *   The language name or the langcode if not found.
   */
  private function getLanguageName($langcode) {
    if (!$langcode) {
      return $this->languageManager->getCurrentLanguage()->getName();
    }

    $language = $this->languageManager->getLanguage($langcode);
    return $language ? $language->getName() : $langcode;
  }

  /**
   * Load type entity (ProductType or ProductVariationType).
   *
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param string $type_id
   *   The bundle/type ID.
   *
   * @return mixed|null
   *   The type entity or NULL.
   */
  private function loadTypeEntity($entity_type, $type_id) {
    if ($entity_type === self::ENTITY_TYPE_PRODUCT) {
      return ProductType::load($type_id);
    }

    return ProductVariationType::load($type_id);
  }

  /**
   * Create sample entity.
   *
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param string $type_id
   *   The bundle/type ID.
   *
   * @return mixed
   *   Sample entity.
   */
  private function createSampleEntity($entity_type, $type_id) {
    if ($entity_type === self::ENTITY_TYPE_PRODUCT) {
      return Product::create(['type' => $type_id]);
    }

    return ProductVariation::create(['type' => $type_id]);
  }

  /**
   * Load form display.
   *
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param string $type_id
   *   The bundle/type ID.
   *
   * @return mixed|null
   *   Form display entity or NULL.
   */
  private function loadFormDisplay($entity_type, $type_id) {
    $form_display_id = $entity_type === self::ENTITY_TYPE_PRODUCT
      ? 'commerce_product.' . $type_id . '.default'
      : 'commerce_product_variation.' . $type_id . '.default';

    return $this->entityTypeManager
      ->getStorage('entity_form_display')
      ->load($form_display_id);
  }

  /**
   * Get all product types.
   *
   * @return array
   *   Array of product type options.
   */
  private function getAllProductTypes() {
    $product_types = ProductType::loadMultiple();
    $options = [];
    foreach ($product_types as $type_id => $type) {
      $options[$type_id] = $type->label();
    }
    return $options;
  }

  /**
   * Get all variation types.
   *
   * @return array
   *   Array of variation type options.
   */
  private function getAllVariationTypes() {
    $variation_types = ProductVariationType::loadMultiple();
    $options = [];
    foreach ($variation_types as $type_id => $type) {
      $options[$type_id] = $type->label();
    }
    return $options;
  }

  /**
   * Prepare field edit step.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function prepareFieldEdit(array &$form, FormStateInterface $form_state) {
    // Check permissions.
    if (!$this->fieldAccessValidator->canBulkUpdateFields()) {
      $this->messenger()->addError($this->t('You do not have permission to bulk update fields.'));
      return;
    }

    $entity_type = $form_state->getValue('entity_type') ?: self::ENTITY_TYPE_PRODUCT;
    $type_id = $form_state->getValue('field_product_type');
    $include_disabled = $form_state->getValue('include_disabled_fields') ?? FALSE;
    $selected_langcode = $form_state->getValue('language_selector') ?? $this->languageManager->getCurrentLanguage()->getId();

    $user_input = $form_state->getUserInput();
    $selected_entities = [];

    // Check for selected entities in user input.
    if (isset($user_input['selected_entities']) && is_array($user_input['selected_entities'])) {
      $selected_entities = $this->fieldAccessValidator->sanitizeEntityIds(
        array_filter($user_input['selected_entities'])
      );
    }

    if (empty($selected_entities)) {
      $this->messenger()->addError($this->t('Please select at least one entity to update.'));
      return;
    }

    // Validate access to selected entities.
    $storage_type = $entity_type === self::ENTITY_TYPE_PRODUCT ? self::STORAGE_PRODUCT : self::STORAGE_VARIATION;
    $entities = $this->entityTypeManager->getStorage($storage_type)->loadMultiple($selected_entities);

    $access_check = $this->fieldAccessValidator->validateBulkAccess($entities);

    if (!empty($access_check['denied'])) {
      $this->messenger()->addWarning($this->t(
        'You do not have access to update @count entity/entities. They have been excluded.',
        ['@count' => count($access_check['denied'])]
      ));
      $selected_entities = $access_check['allowed'];
    }

    if (empty($selected_entities)) {
      $this->messenger()->addError($this->t('You do not have permission to update any of the selected entities.'));
      return;
    }

    // Store selected entities in BOTH generic and combination-specific keys
    // Generic key for current workflow.
    $form_state->set(self::STATE_EDIT_VALUES, [
      'entity_type' => $entity_type,
      'type_id' => $type_id,
      'selected_entities' => $selected_entities,
      'include_disabled' => $include_disabled,
      'langcode' => $selected_langcode,
    ]);

    // Combination-specific key for persistence across different combinations.
    $selection_key = $this->getSelectionKey($entity_type, $type_id, $selected_langcode, $include_disabled);
    $form_state->set($selection_key, $selected_entities);

    $form_state->set(self::STATE_STEP, self::STEP_EDIT);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Prepare field confirmation step.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function prepareFieldConfirmation(array &$form, FormStateInterface $form_state) {
    $values = $form_state->get(self::STATE_EDIT_VALUES);
    $field_values_input = $form_state->getValue('field_values');

    $filled_values = [];
    $field_overrides = [];
    $apply_all_languages = [];

    $all_field_names = $this->extractAllFieldNames($field_values_input);

    foreach ($all_field_names as $field_name => $dummy) {
      // Validate field name.
      if (!preg_match('/^[a-z][a-z0-9_]*$/', $field_name)) {
        $this->messenger()->addError($this->t('Invalid field name detected: @name', ['@name' => $field_name]));
        return;
      }

      $widget_value = $field_values_input[$field_name] ?? NULL;

      // First, try to extract IEF entities from form state.
      $ief_data = $this->fieldValueProcessor->extractIefEntitiesFromFormState($form_state, $field_name);
      if (!empty($ief_data)) {
        // IEF data found - use it directly, don't sanitize.
        $widget_value = ['_ief_entities' => $ief_data];
      }
      elseif (is_string($widget_value)) {
        $widget_value = $this->fieldAccessValidator->sanitizeString($widget_value);
      }
      elseif (is_array($widget_value) && !isset($widget_value['_ief_entities'])) {
        // No IEF in array - sanitize it (but preserve IEF if already present).
        $widget_value = $this->sanitizeArrayRecursive($widget_value);
      }

      $container_key = $field_name . '_container';
      $override_key = 'override_' . $field_name;
      $apply_all_key = 'apply_all_languages_' . $field_name;
      $override_value = FALSE;
      $apply_all_value = FALSE;

      if (isset($field_values_input[$container_key][$override_key])) {
        $override_value = $field_values_input[$container_key][$override_key];
      }

      if (isset($field_values_input[$container_key][$apply_all_key])) {
        $apply_all_value = $field_values_input[$container_key][$apply_all_key];
      }

      if ($widget_value !== NULL && !$this->fieldValueProcessor->isFieldValueEmpty($widget_value)) {
        $filled_values[$field_name] = $widget_value;
        if ($override_value) {
          $field_overrides[$field_name] = TRUE;
        }
        if ($apply_all_value) {
          $apply_all_languages[$field_name] = TRUE;
        }
      }
    }

    if (empty($filled_values)) {
      $this->messenger()->addError($this->t('No values were entered. Please fill at least one field with an actual value before continuing.'));
      return;
    }

    // Final validation of all values.
    if (!$this->fieldAccessValidator->validateFieldValues($filled_values)) {
      $this->messenger()->addError($this->t('Invalid or potentially malicious field values detected. Operation cancelled.'));
      return;
    }

    $form_state->set(self::STATE_CONFIRM_VALUES, [
      'entity_type' => $values['entity_type'],
      'type_id' => $values['type_id'],
      'selected_entities' => $values['selected_entities'],
      'field_values' => $filled_values,
      'field_overrides' => $field_overrides,
      'apply_all_languages' => $apply_all_languages,
      'include_disabled' => $values['include_disabled'] ?? FALSE,
      'langcode' => $values['langcode'] ?? NULL,
    ]);
    $form_state->set(self::STATE_STEP, self::STEP_CONFIRM);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Recursively sanitize array values.
   *
   * @param array $array
   *   Array to sanitize.
   * @param int $depth
   *   Current depth.
   *
   * @return array
   *   Sanitized array.
   */
  private function sanitizeArrayRecursive(array $array, $depth = 0): array {
    if ($depth > 10) {
      return [];
    }

    // Preserve IEF entities structure - don't sanitize these.
    if (isset($array['_ief_entities']) && is_array($array['_ief_entities'])) {
      return $array;
    }

    // If this array contains entity objects with needs_save, preserve it.
    if (isset($array['entity']) && is_object($array['entity'])) {
      return $array;
    }

    $sanitized = [];
    foreach ($array as $key => $value) {
      // Keep objects as-is (entity references, TranslatableMarkup, etc).
      if (is_object($value)) {
        $sanitized[$key] = $value;
        continue;
      }

      if (is_string($value)) {
        $sanitized[$key] = $this->fieldAccessValidator->sanitizeString($value);
        continue;
      }

      if (is_array($value)) {
        $sanitized[$key] = $this->sanitizeArrayRecursive($value, $depth + 1);
        continue;
      }

      // Keep numeric and other scalar values.
      $sanitized[$key] = $value;
    }

    return $sanitized;
  }

  /**
   * Extract all field names from field values input.
   *
   * @param array $field_values_input
   *   Field values input array.
   *
   * @return array
   *   Array of field names.
   */
  private function extractAllFieldNames(array $field_values_input) {
    $all_field_names = [];

    foreach ($field_values_input as $key => $value) {
      if (strpos($key, '_container') !== FALSE) {
        $field_name = str_replace('_container', '', $key);
        $all_field_names[$field_name] = TRUE;
      }
    }

    return $all_field_names;
  }

  /**
   * Confirm and apply field values.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function confirmFieldUpdate(array &$form, FormStateInterface $form_state) {
    // Re-check permissions before applying changes.
    if (!$this->fieldAccessValidator->canBulkUpdateFields()) {
      $this->messenger()->addError($this->t('You do not have permission to bulk update fields.'));
      $this->clearFieldFormState($form_state);
      return;
    }

    $values = $form_state->get(self::STATE_CONFIRM_VALUES);

    if (empty($values)) {
      $this->messenger()->addError($this->t('Session expired. Please start over.'));
      $this->clearFieldFormState($form_state);
      return;
    }

    $entity_type = $values['entity_type'];
    $selected_entities = $values['selected_entities'];
    $field_values = $values['field_values'];
    $field_overrides = $values['field_overrides'];
    $apply_all_languages = $values['apply_all_languages'] ?? [];
    $langcode = $values['langcode'] ?? NULL;

    // Validate field values.
    if (!$this->fieldAccessValidator->validateFieldValues($field_values)) {
      $this->messenger()->addError($this->t('Invalid field values detected. Operation cancelled.'));
      $this->clearFieldFormState($form_state);
      return;
    }

    // Re-validate entity access.
    $storage_type = $entity_type === self::ENTITY_TYPE_PRODUCT ? self::STORAGE_PRODUCT : self::STORAGE_VARIATION;
    $entities = $this->entityTypeManager->getStorage($storage_type)->loadMultiple($selected_entities);

    $field_names = array_keys($field_values);
    $access_check = $this->fieldAccessValidator->validateBulkAccess($entities, $field_names);

    if (!empty($access_check['denied'])) {
      $this->messenger()->addWarning($this->t(
        '@count entity/entities have been excluded due to access restrictions.',
        ['@count' => count($access_check['denied'])]
      ));
      $selected_entities = $access_check['allowed'];
    }

    if (empty($selected_entities)) {
      $this->messenger()->addError($this->t('No entities can be updated due to permission restrictions.'));
      $this->clearFieldFormState($form_state);
      return;
    }

    // Apply updates.
    try {
      $stats = $this->fieldEntityProcessor->applyFieldValues(
        $entity_type,
        $selected_entities,
        $field_values,
        $field_overrides,
        FALSE,
        $langcode,
        $apply_all_languages
      );

      $entity_labels = $this->getEntityLabels($entity_type);

      $language_message = '';

      if (!empty($stats['languages_updated'])) {
        $lang_names = [];
        foreach ($stats['languages_updated'] as $lang_code => $count) {
          $lang_names[] = $this->getLanguageName($lang_code);
        }
        $language_message = ' ' . $this->t('(Languages: @langs)', ['@langs' => implode(', ', $lang_names)]);
      }
      elseif ($langcode) {
        $language_name = $this->getLanguageName($langcode);
        $language_message = ' ' . $this->t('(Language: @lang)', ['@lang' => $language_name]);
      }

      if ($stats['updated_count'] > 0) {
        $this->messenger()->addStatus($this->formatPlural(
          $stats['updated_count'],
          'Updated 1 @entity successfully@lang.',
          'Updated @count @entities successfully@lang.',
          [
            '@entity' => $entity_labels['singular'],
            '@entities' => $entity_labels['plural'],
            '@lang' => $language_message,
          ]
        ));
      }

      if ($stats['skipped_count'] > 0) {
        $this->messenger()->addStatus($this->t('Skipped @count already-filled fields (not overridden).', ['@count' => $stats['skipped_count']]));
      }

      if ($stats['failed_count'] > 0) {
        $this->messenger()->addWarning($this->formatPlural(
          $stats['failed_count'],
          'Failed to update 1 @entity.',
          'Failed to update @count @entities.',
          ['@entity' => $entity_labels['singular'], '@entities' => $entity_labels['plural']]
        ));
      }

      if ($stats['updated_count'] == 0 && $stats['failed_count'] == 0) {
        $this->messenger()->addWarning($this->t('No entities were updated. All selected fields were already filled.'));
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get(self::LOGGER_CHANNEL)->error('Bulk field update failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      $this->messenger()->addError($this->t('An error occurred during the update. Please check the logs.'));
    }

    $this->clearFieldFormState($form_state);
  }

  /**
   * Cancel field edit operation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function cancelFieldEdit(array &$form, FormStateInterface $form_state) {
    $this->messenger()->addStatus($this->t('Field update has been cancelled.'));
    $this->clearFieldFormState($form_state);
  }

  /**
   * Save selected entities from user input to form state.
   *
   * This method extracts checkbox selections from the current user input
   * and stores them in form state so they persist across AJAX rebuilds.
   */
  private function saveSelectedEntities(FormStateInterface $form_state) {
    $user_input = $form_state->getUserInput();

    // Get selected entities from user input.
    $selected_entities = [];
    if (isset($user_input['selected_entities']) && is_array($user_input['selected_entities'])) {
      // Filter out empty values and keep only the selected entity IDs.
      $selected_entities = array_filter($user_input['selected_entities']);
      // Convert to array values to ensure clean array.
      $selected_entities = array_values($selected_entities);
    }

    // Store in form state for persistence across AJAX calls.
    $form_state->set(self::STATE_SELECTED_ENTITIES, $selected_entities);

    // Also keep it in user input for Drupal's rebuild process.
    $current_input = $form_state->getUserInput();
    if (!isset($current_input['selected_entities']) || !is_array($current_input['selected_entities'])) {
      $current_input['selected_entities'] = [];
    }

    // Preserve the selected entities in user input.
    foreach ($selected_entities as $entity_id) {
      $current_input['selected_entities'][$entity_id] = $entity_id;
    }

    $form_state->setUserInput($current_input);
  }

  /**
   * Clear field form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  private function clearFieldFormState(FormStateInterface $form_state): void {
    $form_state->set(self::STATE_STEP, NULL);
    $form_state->set(self::STATE_EDIT_VALUES, NULL);
    $form_state->set(self::STATE_CONFIRM_VALUES, NULL);
    $form_state->set(self::STATE_FIELD_STATUS, NULL);
    $form_state->set(self::STATE_SELECTED_ENTITIES, NULL);
    // Removing widget related states.
    $form_state->set('inline_entity_form', NULL);
    $form_state->set('field_storage', NULL);
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for field product list update.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The field product container element.
   */
  public function updateFieldProductList(array &$form, FormStateInterface $form_state) {
    // Save selected checkboxes before AJAX refresh.
    $this->saveSelectedEntities($form_state);

    return $form['field_product_container'];
  }

  /**
   * AJAX callback for field edit preparation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function ajaxPrepareFieldEdit(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * AJAX callback for field edit cancellation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function ajaxCancelFieldEdit(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * AJAX callback for field confirmation preparation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function ajaxPrepareFieldConfirmation(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Handled by specific submit handlers.
  }

}
