<?php

namespace Drupal\product_manager_tool\Form;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Block\BlockManagerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\TempStore\SharedTempStoreFactory;
use Drupal\layout_builder\LayoutTempstoreRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Product Manager Form - Main wrapper for Layout, Field and Variation managers.
 */
class ProductManagerForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The block manager.
   *
   * @var \Drupal\Core\Block\BlockManagerInterface
   */
  protected $blockManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The layout tempstore repository.
   *
   * @var \Drupal\layout_builder\LayoutTempstoreRepositoryInterface
   */
  protected $layoutTempstoreRepository;

  /**
   * The shared tempstore factory.
   *
   * @var \Drupal\Core\TempStore\SharedTempStoreFactory
   */
  protected $sharedTempStoreFactory;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuid;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The layout form instance.
   *
   * @var \Drupal\product_manager_tool\Form\LayoutTemplateForm|null
   */
  protected $layoutFormInstance;

  /**
   * The field form instance.
   *
   * @var \Drupal\product_manager_tool\Form\ProductFieldManagerForm|null
   */
  protected $fieldFormInstance;

  /**
   * Constructs a ProductManagerForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Block\BlockManagerInterface $block_manager
   *   The block manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\layout_builder\LayoutTempstoreRepositoryInterface $layout_tempstore_repository
   *   The layout tempstore repository.
   * @param \Drupal\Core\TempStore\SharedTempStoreFactory $shared_tempstore_factory
   *   The shared tempstore factory.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    BlockManagerInterface $block_manager,
    ModuleHandlerInterface $module_handler,
    LayoutTempstoreRepositoryInterface $layout_tempstore_repository,
    SharedTempStoreFactory $shared_tempstore_factory,
    LanguageManagerInterface $language_manager,
    DateFormatterInterface $date_formatter,
    UuidInterface $uuid,
    LoggerChannelFactoryInterface $logger_factory,
    EntityFieldManagerInterface $entity_field_manager,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->blockManager = $block_manager;
    $this->moduleHandler = $module_handler;
    $this->layoutTempstoreRepository = $layout_tempstore_repository;
    $this->sharedTempStoreFactory = $shared_tempstore_factory;
    $this->languageManager = $language_manager;
    $this->dateFormatter = $date_formatter;
    $this->uuid = $uuid;
    $this->loggerFactory = $logger_factory;
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.block'),
      $container->get('module_handler'),
      $container->get('layout_builder.tempstore_repository'),
      $container->get('tempstore.shared'),
      $container->get('language_manager'),
      $container->get('date.formatter'),
      $container->get('uuid'),
      $container->get('logger.factory'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'product_manager_tool_main_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'core/drupal.ajax';
    $form['#attached']['library'][] = 'system/admin';
    $form['#attached']['library'][] = 'core/drupal.dropbutton';
    $form['#attached']['library'][] = 'product_manager_tool/product_manager_scroll';
    $form['#attached']['library'][] = 'product_manager_tool/product_manager_reload';
    $form['#attached']['library'][] = 'product_manager_tool/product_manager_confirm_countdown';
    $form['#attached']['library'][] = 'product_manager_tool/product_manager_select_boxes';
    $form['#attached']['library'][] = 'product_manager_tool/product_manager_tool_styles';

    if ($this->moduleHandler->moduleExists('gin')) {
      $form['#attached']['library'][] = 'gin/gin_dropbutton';
    }

    $form['#attached']['html_head'][] = [
      [
        '#tag' => 'style',
        '#value' => '.product-manager-tabs-wrapper { margin-bottom: 0.75rem; }',
      ],
      'product-manager-tabs-style',
    ];

    $active_tab = $form_state->get('active_tab') ?: 'layout';

    $form['#prefix'] = '<div id="field-main-wrapper">';
    $form['#suffix'] = '</div>';

    $form['tabs_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['product-manager-tabs-wrapper']],
      '#weight' => -100,
    ];

    $form['tabs_wrapper']['navigation'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['product-manager-tabs-navigation']],
    ];

    $tabs = [
      'layout' => $this->t('Layout Template Manager'),
      'field' => $this->t('Field Manager'),
    ];

    foreach ($tabs as $tab_key => $tab_label) {
      $button_classes = ['button'];
      $attributes = ['class' => $button_classes];

      if ($active_tab === $tab_key) {
        $button_classes[] = 'button--primary';
        $attributes['class'] = $button_classes;
        $attributes['style'] = 'pointer-events: none; cursor: default;';
        $attributes['data-active'] = 'true';
      }

      $form['tabs_wrapper']['navigation'][$tab_key . '_tab'] = [
        '#type' => 'submit',
        '#value' => $tab_label,
        '#name' => 'switch_to_' . $tab_key,
        '#submit' => ['::switchTab'],
        '#attributes' => $attributes,
        '#ajax' => [
          'callback' => '::switchTabCallback',
          'wrapper' => 'tab-content-wrapper',
          'effect' => 'fade',
          'progress' => ['type' => 'fullscreen'],
        ],
        '#limit_validation_errors' => [],
      ];
    }

    $form['active_tab'] = [
      '#type' => 'hidden',
      '#value' => $active_tab,
    ];

    $form['tab_content'] = [
      '#type' => 'container',
      '#prefix' => '<div id="tab-content-wrapper">',
      '#suffix' => '</div>',
    ];

    if ($active_tab === 'field') {
      $this->fieldFormInstance = ProductFieldManagerForm::create(\Drupal::getContainer());
      $field_form = $this->fieldFormInstance->buildForm([], $form_state);
      $this->updateCallbacksRecursive($field_form);

      unset($field_form['#prefix']);
      unset($field_form['#suffix']);
      $form['tab_content'] = array_merge($form['tab_content'], $field_form);
    }
    else {
      $this->layoutFormInstance = LayoutTemplateForm::create(\Drupal::getContainer());
      $layout_form = $this->layoutFormInstance->buildForm([], $form_state);
      $this->updateCallbacksRecursive($layout_form);

      if (isset($layout_form['#prefix'])) {
        unset($layout_form['#prefix']);
      }
      if (isset($layout_form['#suffix'])) {
        unset($layout_form['#suffix']);
      }

      $form['tab_content'] = array_merge($form['tab_content'], $layout_form);
    }

    return $form;
  }

  /**
   * Updates callbacks recursively to use proxy methods.
   *
   * @param array $element
   *   The form element to update.
   */
  private function updateCallbacksRecursive(&$element) {
    if (!is_array($element)) {
      return;
    }

    if (isset($element['#ajax']['callback'])) {
      $callback_map = [
        '::updateFieldProductList' => '::proxyFieldProductList',
        '::ajaxPrepareFieldEdit' => '::proxyAjaxPrepareFieldEdit',
        '::ajaxCancelFieldEdit' => '::proxyAjaxCancelFieldEdit',
        '::ajaxPrepareFieldConfirmation' => '::proxyAjaxPrepareFieldConfirmation',
        '::ajaxBackToFieldEdit' => '::proxyAjaxBackToFieldEdit',
        '::updateProductTypeAndTemplate' => '::proxyLayoutProductType',
        '::updateTemplateOptions' => '::proxyTemplateOptions',
        '::previewTemplateLayout' => '::proxyTemplatePreview',
        '::ajaxCreateTemplate' => '::proxyCreateTemplate',
        '::ajaxDeleteTemplate' => '::proxyDeleteTemplate',
        '::ajaxSubmitForm' => '::proxyAjaxSubmitForm',
        '::ajaxCancelSubmit' => '::proxyAjaxCancelSubmit',
        '::ajaxBackToEdit' => '::proxyAjaxBackToEdit',
        '::ajaxBackToSelection' => '::proxyAjaxBackToSelection',
      ];
      if (is_string($element['#ajax']['callback']) && isset($callback_map[$element['#ajax']['callback']])) {
        $element['#ajax']['callback'] = $callback_map[$element['#ajax']['callback']];
      }
    }

    if (isset($element['#submit']) && is_array($element['#submit'])) {
      $submit_map = [
        '::prepareFieldEdit' => '::proxyPrepareFieldEdit',
        '::prepareFieldConfirmation' => '::proxyPrepareFieldConfirmation',
        '::confirmFieldUpdate' => '::proxyConfirmFieldUpdate',
        '::cancelFieldEdit' => '::proxyCancelFieldEdit',
        '::backToFieldEdit' => '::proxyBackToFieldEdit',
        '::backToSelection' => '::proxyBackToSelection',
        '::createTemplateProduct' => '::proxyCreateTemplateSubmit',
        '::deleteTemplateProduct' => '::proxyDeleteTemplateSubmit',
        '::confirmSubmit' => '::proxyConfirmSubmit',
        '::cancelSubmit' => '::proxyCancelSubmit',
        '::backToEdit' => '::proxyBackToEdit',
      ];
      foreach ($element['#submit'] as $key => $handler) {
        if (is_string($handler) && isset($submit_map[$handler])) {
          $element['#submit'][$key] = $submit_map[$handler];
        }
      }
    }

    foreach ($element as $key => $value) {
      if (is_array($value) && strpos($key, '#') !== 0) {
        $this->updateCallbacksRecursive($element[$key]);
      }
    }
  }

  /**
   * Switches between tabs.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function switchTab(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $button_name = $triggering_element['#name'];

    $new_tab = ($button_name === 'switch_to_field') ? 'field' : 'layout';

    $form_state->set('active_tab', $new_tab);
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for tab switching.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function switchTabCallback(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    if (isset($form['tabs_wrapper']['navigation'])) {
      $response->addCommand(new ReplaceCommand('.product-manager-tabs-navigation', $form['tabs_wrapper']['navigation']));
    }

    if (isset($form['tab_content'])) {
      $response->addCommand(new ReplaceCommand('#tab-content-wrapper', $form['tab_content']));
    }

    return $response;
  }

  /**
   * Proxy callback for field product list update.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The field product container element.
   */
  public function proxyFieldProductList(array &$form, FormStateInterface $form_state) {
    return $form['tab_content']['field_product_container'];
  }

  /**
   * Proxy AJAX callback for field edit preparation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function proxyAjaxPrepareFieldEdit(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * Proxy submit handler for field edit cancellation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyAjaxCancelFieldEdit(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * Proxy AJAX callback for field confirmation preparation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function proxyAjaxPrepareFieldConfirmation(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * Proxy submit handler for field edit preparation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyPrepareFieldEdit(array &$form, FormStateInterface $form_state) {
    if ($this->fieldFormInstance) {
      $this->fieldFormInstance->prepareFieldEdit($form, $form_state);
    }
  }

  /**
   * Proxy submit handler for field confirmation preparation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyPrepareFieldConfirmation(array &$form, FormStateInterface $form_state) {
    if ($this->fieldFormInstance) {
      $this->fieldFormInstance->prepareFieldConfirmation($form, $form_state);
    }
  }

  /**
   * Proxy submit handler for field update confirmation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyConfirmFieldUpdate(array &$form, FormStateInterface $form_state) {
    if ($this->fieldFormInstance) {
      $this->fieldFormInstance->confirmFieldUpdate($form, $form_state);
    }
  }

  /**
   * Proxy submit handler for field edit cancellation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyCancelFieldEdit(array &$form, FormStateInterface $form_state) {
    if ($this->fieldFormInstance) {
      $this->fieldFormInstance->cancelFieldEdit($form, $form_state);
    }
  }

  /**
   * Proxy submit handler for back to field edit.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyBackToFieldEdit(array &$form, FormStateInterface $form_state) {
    if ($this->fieldFormInstance) {
      $this->fieldFormInstance->backToFieldEdit($form, $form_state);
    }
  }

  /**
   * Proxy submit handler for back to selection.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyBackToSelection(array &$form, FormStateInterface $form_state) {
    if ($this->fieldFormInstance) {
      $this->fieldFormInstance->backToSelection($form, $form_state);
    }
  }

  /**
   * Proxy AJAX callback for back to field edit.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function proxyAjaxBackToFieldEdit(array &$form, FormStateInterface $form_state) {
    if ($this->fieldFormInstance) {
      return $this->fieldFormInstance->ajaxBackToFieldEdit($form, $form_state);
    }
    return $form;
  }

  /**
   * Proxy AJAX callback for back to selection.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function proxyAjaxBackToSelection(array &$form, FormStateInterface $form_state) {
    if ($this->fieldFormInstance) {
      return $this->fieldFormInstance->ajaxBackToSelection($form, $form_state);
    }
    return $form;
  }

  /**
   * Proxy callback for layout product type update.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function proxyLayoutProductType(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#template-section-wrapper', $form['tab_content']['template_section']));
    $response->addCommand(new ReplaceCommand('#product-list-wrapper', $form['tab_content']['product_container']));
    return $response;
  }

  /**
   * Proxy callback for template options update.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The template options element.
   */
  public function proxyTemplateOptions(array &$form, FormStateInterface $form_state) {
    return $form['tab_content']['template_section']['template_options'];
  }

  /**
   * Proxy callback for template preview.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The preview element.
   */
  public function proxyTemplatePreview(array &$form, FormStateInterface $form_state) {
    return $form['tab_content']['template_section']['template_options']['preview'];
  }

  /**
   * Proxy callback for template creation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function proxyCreateTemplate(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#template-options-wrapper', $form['tab_content']['template_section']['template_options']));
    return $response;
  }

  /**
   * Proxy callback for template deletion.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function proxyDeleteTemplate(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#template-section-wrapper', $form['tab_content']['template_section']));
    return $response;
  }

  /**
   * Proxy submit handler for template creation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyCreateTemplateSubmit(array &$form, FormStateInterface $form_state) {
    if ($this->layoutFormInstance) {
      $this->layoutFormInstance->createTemplateProduct($form, $form_state);
    }
  }

  /**
   * Proxy submit handler for template deletion.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyDeleteTemplateSubmit(array &$form, FormStateInterface $form_state) {
    if ($this->layoutFormInstance) {
      $this->layoutFormInstance->deleteTemplateProduct($form, $form_state);
    }
  }

  /**
   * Proxy submit handler for layout confirmation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyConfirmSubmit(array &$form, FormStateInterface $form_state) {
    if ($this->layoutFormInstance) {
      $this->layoutFormInstance->confirmSubmit($form, $form_state);
    }
  }

  /**
   * Proxy submit handler for layout cancellation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyCancelSubmit(array &$form, FormStateInterface $form_state) {
    if ($this->layoutFormInstance) {
      $this->layoutFormInstance->cancelSubmit($form, $form_state);
    }
  }

  /**
   * Proxy AJAX callback for layout cancellation.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array|\Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response or form array.
   */
  public function proxyAjaxCancelSubmit(array &$form, FormStateInterface $form_state) {
    if ($this->layoutFormInstance) {
      return $this->layoutFormInstance->ajaxCancelSubmit($form, $form_state);
    }
    return $form;
  }

  /**
   * Proxy submit handler for back to edit.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function proxyBackToEdit(array &$form, FormStateInterface $form_state) {
    if ($this->layoutFormInstance) {
      $this->layoutFormInstance->backToEdit($form, $form_state);
    }
  }

  /**
   * Proxy AJAX callback for back to edit.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function proxyAjaxBackToEdit(array &$form, FormStateInterface $form_state) {
    if ($this->layoutFormInstance) {
      return $this->layoutFormInstance->ajaxBackToEdit($form, $form_state);
    }
    return $form;
  }

  /**
   * Proxy AJAX callback for form submission.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array|\Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response or form array.
   */
  public function proxyAjaxSubmitForm(array &$form, FormStateInterface $form_state) {
    if ($this->layoutFormInstance) {
      return $this->layoutFormInstance->ajaxSubmitForm($form, $form_state);
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $active_tab = $form_state->get('active_tab') ?: 'layout';

    if ($active_tab === 'field' && $this->fieldFormInstance) {
      $this->fieldFormInstance->submitForm($form, $form_state);
    }
    elseif ($active_tab === 'layout' && $this->layoutFormInstance) {
      $this->layoutFormInstance->submitForm($form, $form_state);
    }
  }

}
