<?php

namespace Drupal\product_manager_tool\Service;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Service for validating field access and permissions.
 */
class FieldAccessValidator {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a FieldAccessValidator object.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(AccountProxyInterface $current_user) {
    $this->currentUser = $current_user;
  }

  /**
   * Check if user can bulk update fields.
   *
   * @return bool
   *   TRUE if user has permission.
   */
  public function canBulkUpdateFields(): bool {
    return $this->currentUser->hasPermission('administer commerce_product')
      || $this->currentUser->hasPermission('bulk update product fields');
  }

  /**
   * Validate entity access for field updates.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE if user can update entity.
   */
  public function canUpdateEntity(EntityInterface $entity): bool {
    return $entity->access('update', $this->currentUser);
  }

  /**
   * Validate field access for entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   *
   * @return bool
   *   TRUE if user can edit field.
   */
  public function canEditField(EntityInterface $entity, string $field_name): bool {
    if (!$entity->hasField($field_name)) {
      return FALSE;
    }

    $field_access = $entity->get($field_name)->access('edit', $this->currentUser, TRUE);
    return $field_access->isAllowed();
  }

  /**
   * Validate multiple entities for bulk update with IDOR protection.
   *
   * @param array $entities
   *   Array of entities to validate.
   * @param array $field_names
   *   Array of field names to update.
   *
   * @return array
   *   Array with 'allowed' and 'denied' entity IDs and reasons.
   */
  public function validateBulkAccess(array $entities, array $field_names = []): array {
    $allowed = [];
    $denied = [];
    $denial_reasons = [];

    foreach ($entities as $entity) {
      $entity_id = $entity->id();

      // IDOR Protection: Verify entity actually belongs to the expected bundle.
      $expected_bundle = $entity->getEntityType()->getBundleEntityType();
      if ($expected_bundle && method_exists($entity, 'bundle')) {
        $actual_bundle = $entity->bundle();
        // Additional verification could be added here if needed.
      }

      // Check entity update access.
      if (!$this->canUpdateEntity($entity)) {
        $denied[] = $entity_id;
        $denial_reasons[$entity_id] = 'No update access';
        continue;
      }

      // IDOR Protection: Verify entity ownership or permissions.
      if (!$this->verifyEntityOwnership($entity)) {
        $denied[] = $entity_id;
        $denial_reasons[$entity_id] = 'Ownership verification failed';
        continue;
      }

      // Check field-level access.
      $field_access_ok = TRUE;
      foreach ($field_names as $field_name) {
        if (!$this->canEditField($entity, $field_name)) {
          $field_access_ok = FALSE;
          $denial_reasons[$entity_id] = "No access to field: $field_name";
          break;
        }
      }

      if ($field_access_ok) {
        $allowed[] = $entity_id;
      }
      else {
        $denied[] = $entity_id;
      }
    }

    return [
      'allowed' => $allowed,
      'denied' => $denied,
      'reasons' => $denial_reasons,
    ];
  }

  /**
   * Verify entity ownership or admin permissions (IDOR protection).
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to verify.
   *
   * @return bool
   *   TRUE if verification passes.
   */
  private function verifyEntityOwnership(EntityInterface $entity): bool {
    // Admin users bypass ownership check.
    if ($this->currentUser->hasPermission('administer commerce_product')) {
      return TRUE;
    }

    // Check if entity has owner field.
    if ($entity->hasField('uid')) {
      $owner_id = $entity->get('uid')->target_id;

      // User owns the entity.
      if ($owner_id == $this->currentUser->id()) {
        return TRUE;
      }

      // Check for "edit any" permission.
      $entity_type_id = $entity->getEntityTypeId();
      $bundle = $entity->bundle();

      if ($this->currentUser->hasPermission("edit any $bundle $entity_type_id")) {
        return TRUE;
      }

      // User doesn't own and doesn't have "edit any" permission.
      return FALSE;
    }

    // No ownership field, rely on access system.
    return TRUE;
  }

  /**
   * Sanitize entity IDs input.
   *
   * @param array $entity_ids
   *   Raw entity IDs from user input.
   *
   * @return array
   *   Sanitized entity IDs.
   */
  public function sanitizeEntityIds(array $entity_ids): array {
    return array_filter(array_map('intval', $entity_ids), function ($id) {
      return $id > 0;
    });
  }

  /**
   * Validate field values for XSS and injection attacks.
   *
   * @param array $field_values
   *   Field values to validate.
   *
   * @return bool
   *   TRUE if values are safe.
   */
  public function validateFieldValues(array $field_values): bool {
    foreach ($field_values as $field_name => $value) {
      if (!$this->isValidFieldName($field_name)) {
        return FALSE;
      }

      if (is_array($value) && !$this->validateArrayValue($value)) {
        return FALSE;
      }

      if (is_string($value) && !$this->isStringSafe($value)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Check if string is safe from XSS and injection.
   *
   * @param string $value
   *   String value to check.
   *
   * @return bool
   *   TRUE if safe.
   */
  private function isStringSafe(string $value): bool {
    // Check for script tags.
    if (preg_match('/<script[^>]*>.*?<\/script>/is', $value)) {
      return FALSE;
    }

    // Check for event handlers.
    if (preg_match('/on\w+\s*=/i', $value)) {
      return FALSE;
    }

    // Check for javascript: protocol.
    if (preg_match('/javascript:/i', $value)) {
      return FALSE;
    }

    // Check for data: protocol with base64.
    if (preg_match('/data:.*base64/i', $value)) {
      return FALSE;
    }

    // Check for SQL injection patterns.
    $sql_patterns = [
      '/union.*select/i',
      '/insert.*into/i',
      '/delete.*from/i',
      '/drop.*table/i',
      '/update.*set/i',
    ];

    foreach ($sql_patterns as $pattern) {
      if (preg_match($pattern, $value)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Sanitize string value for safe storage using Drupal APIs.
   *
   * @param string $value
   *   Raw string value.
   *
   * @return string
   *   Sanitized value.
   */
  public function sanitizeString(string $value): string {
    // Use Drupal's Xss::filter for HTML content.
    $value = Xss::filter($value);

    // Remove null bytes.
    $value = str_replace("\0", '', $value);

    // Trim whitespace.
    $value = trim($value);

    // Limit length.
    if (strlen($value) > 50000) {
      $value = substr($value, 0, 50000);
    }

    return $value;
  }

  /**
   * Sanitize for HTML display.
   *
   * @param string $value
   *   Raw string value.
   *
   * @return string
   *   HTML-escaped value.
   */
  public function sanitizeForDisplay(string $value): string {
    return Html::escape($value);
  }

  /**
   * Sanitize plain text (no HTML allowed).
   *
   * @param string $value
   *   Raw string value.
   *
   * @return string
   *   Plain text value.
   */
  public function sanitizePlainText(string $value): string {
    // Strip all HTML tags.
    $value = strip_tags($value);

    // Remove null bytes.
    $value = str_replace("\0", '', $value);

    // Trim whitespace.
    $value = trim($value);

    return $value;
  }

  /**
   * Validate entity ownership for IEF operations.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE if current user can create/edit this entity.
   */
  public function canModifyIefEntity(EntityInterface $entity): bool {
    // Check if entity has owner field.
    if ($entity->hasField('uid')) {
      $owner_id = $entity->get('uid')->target_id;

      // New entities or entities owned by current user.
      if (empty($owner_id) || $owner_id == $this->currentUser->id()) {
        return TRUE;
      }

      // Check if user has permission to edit any entity.
      $entity_type_id = $entity->getEntityTypeId();
      $bundle = $entity->bundle();

      if ($this->currentUser->hasPermission("edit any $bundle $entity_type_id")) {
        return TRUE;
      }
    }

    // Default to entity access check.
    return $entity->access('update', $this->currentUser);
  }

  /**
   * Validate field name format.
   *
   * @param string $field_name
   *   Field name to validate.
   *
   * @return bool
   *   TRUE if valid field name.
   */
  private function isValidFieldName(string $field_name): bool {
    return preg_match('/^[a-z][a-z0-9_]*$/', $field_name) === 1;
  }

  /**
   * Recursively validate array values.
   *
   * @param array $value
   *   Array value to validate.
   * @param int $depth
   *   Current recursion depth.
   *
   * @return bool
   *   TRUE if array is safe.
   */
  private function validateArrayValue(array $value, int $depth = 0): bool {
    if ($depth > 10) {
      return FALSE;
    }

    foreach ($value as $key => $item) {
      if (is_array($item)) {
        if (!$this->validateArrayValue($item, $depth + 1)) {
          return FALSE;
        }
      }
      elseif (is_string($item) && strlen($item) > 50000) {
        return FALSE;
      }
    }

    return TRUE;
  }

}
