<?php

namespace Drupal\product_manager_tool\Service;

use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for processing field operations on products and variations.
 */
class FieldEntityProcessor {

  /**
   * Fields to skip during processing.
   *
   * @var array
   */
  private const SKIP_FIELDS = [
    'layout_selection',
    'layout_builder__layout',
    'stores',
    'uid',
    'created',
    'changed',
    'default_langcode',
    'path',
    'sku',
    'product_id',
    'price',
    'list_price',
    'field_remote_id',
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The field value processor service.
   *
   * @var \Drupal\product_manager_tool\Service\FieldValueProcessor
   */
  protected $fieldValueProcessor;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs a FieldEntityProcessor object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\product_manager_tool\Service\FieldValueProcessor $field_value_processor
   *   The field value processor.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory,
    FieldValueProcessor $field_value_processor,
    LanguageManagerInterface $language_manager,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
    $this->fieldValueProcessor = $field_value_processor;
    $this->languageManager = $language_manager;
  }

  /**
   * Get entities with empty fields.
   *
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param string $type_id
   *   The bundle/type ID.
   * @param bool $include_disabled
   *   Whether to include disabled fields.
   * @param bool $include_filled
   *   Whether to include entities with all fields filled.
   * @param string|null $langcode
   *   The language code to filter by.
   *
   * @return array
   *   Array of entities with empty fields.
   */
  public function getEntitiesWithEmptyFields(
    string $entity_type,
    string $type_id,
    bool $include_disabled = FALSE,
    bool $include_filled = FALSE,
    ?string $langcode = NULL,
  ): array {
    $storage_type = $entity_type === 'product' ? 'commerce_product' : 'commerce_product_variation';
    $bundle_key = 'type';

    // Get current language if not specified.
    if ($langcode === NULL) {
      $langcode = $this->languageManager->getCurrentLanguage()->getId();
    }

    $entity_ids = $this->entityTypeManager->getStorage($storage_type)
      ->getQuery()
      ->condition($bundle_key, $type_id)
      ->accessCheck(TRUE)
      ->execute();

    if (empty($entity_ids)) {
      return [];
    }

    $entities = $entity_type === 'product'
      ? Product::loadMultiple($entity_ids)
      : ProductVariation::loadMultiple($entity_ids);

    $entities_with_empty = [];

    foreach ($entities as $entity) {
      // Skip template products.
      if ($entity_type === 'product' && strpos($entity->getTitle(), '[TEMPLATE]') !== FALSE) {
        continue;
      }

      // Load translation if available and different from current.
      $translated_entity = $this->getEntityTranslation($entity, $langcode);

      // Skip if translation doesn't exist.
      if (!$translated_entity) {
        continue;
      }

      $empty_fields = $this->getEntityFields($translated_entity, $include_disabled, TRUE);

      if ($include_filled || !empty($empty_fields)) {
        $entities_with_empty[] = [
          $entity_type => $translated_entity,
          'empty_fields' => $empty_fields,
          'langcode' => $langcode,
          'is_translation' => $langcode !== $entity->language()->getId(),
        ];
      }
    }

    return $entities_with_empty;
  }

  /**
   * Get entity translation or original if translation doesn't exist.
   *
   * @param mixed $entity
   *   The entity object to retrieve the translation from.
   * @param string $langcode
   *   The language code for the desired translation (e.g., 'en', 'tr').
   *
   * @return mixed
   *   The translated entity object, the original entity if not translatable,
   *   or NULL if the requested translation does not exist.
   */
  private function getEntityTranslation(mixed $entity, string $langcode): mixed {
    // If entity is not translatable, return original.
    if (!$entity->isTranslatable()) {
      return $entity;
    }

    // If requesting original language, return entity as-is.
    if ($entity->language()->getId() === $langcode) {
      return $entity;
    }

    // Check if translation exists.
    if ($entity->hasTranslation($langcode)) {
      return $entity->getTranslation($langcode);
    }

    // Translation doesn't exist.
    return NULL;
  }

  /**
   * Get entity fields (empty or all).
   *
   * @param mixed $entity
   *   The entity object.
   * @param bool $include_disabled
   *   Whether to include disabled fields.
   * @param bool $only_empty
   *   If TRUE, return only empty fields. If FALSE, return all valid fields.
   *
   * @return array
   *   Array of field names.
   */
  public function getEntityFields(mixed $entity, bool $include_disabled = FALSE, bool $only_empty = TRUE): array {
    $result_fields = [];
    $field_definitions = $entity->getFieldDefinitions();
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    foreach ($field_definitions as $field_name => $field_definition) {
      if (in_array($field_name, self::SKIP_FIELDS)) {
        continue;
      }

      if (method_exists($field_definition, 'isBaseField') && $field_definition->isBaseField()) {
        continue;
      }

      if ($field_definition->isComputed() || $field_definition->isReadOnly()) {
        continue;
      }

      if ($field_definition->getType() === 'boolean') {
        continue;
      }

      if (!$this->isFieldVisibleInForm($entity_type_id, $bundle, $field_name)) {
        continue;
      }

      if (!$include_disabled && $this->isFieldDisabled($entity, $field_name)) {
        continue;
      }

      if ($only_empty) {
        if ($entity->hasField($field_name) && $this->isFieldEmpty($entity->get($field_name))) {
          $result_fields[] = $field_name;
        }
      }
      else {
        $result_fields[] = $field_name;
      }
    }

    return $result_fields;
  }

  /**
   * Check if field is visible in form.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle
   *   The bundle.
   * @param string $field_name
   *   The field name.
   *
   * @return bool
   *   TRUE if visible, FALSE otherwise.
   */
  private function isFieldVisibleInForm(string $entity_type_id, string $bundle, string $field_name): bool {
    $form_display = $this->entityTypeManager
      ->getStorage('entity_form_display')
      ->load($entity_type_id . '.' . $bundle . '.default');

    if (!$form_display) {
      return FALSE;
    }

    $component = $form_display->getComponent($field_name);

    return $component && (!isset($component['region']) || $component['region'] !== 'hidden');
  }

  /**
   * Check if a field is disabled.
   *
   * @param mixed $entity
   *   The entity object.
   * @param string $field_name
   *   The field name.
   *
   * @return bool
   *   TRUE if disabled, FALSE otherwise.
   */
  public function isFieldDisabled(mixed $entity, string $field_name): bool {
    $field_definition = $entity->getFieldDefinition($field_name);

    if (!$field_definition || !method_exists($field_definition, 'getThirdPartySettings')) {
      return FALSE;
    }

    $third_party_settings = $field_definition->getThirdPartySettings('disable_field');

    if (empty($third_party_settings)) {
      return FALSE;
    }

    return (!empty($third_party_settings['add_disable']) && $third_party_settings['add_disable'] === 'all')
      || (!empty($third_party_settings['edit_disable']) && $third_party_settings['edit_disable'] === 'all');
  }

  /**
   * Check if a field items object is empty.
   *
   * @param mixed $field_items
   *   The field items object.
   *
   * @return bool
   *   TRUE if empty, FALSE otherwise.
   */
  public function isFieldEmpty(mixed $field_items): bool {
    if ($field_items->isEmpty()) {
      return TRUE;
    }

    foreach ($field_items as $item) {
      if (isset($item->target_id) && !empty($item->target_id)) {
        return FALSE;
      }

      if (isset($item->value) && $item->value !== NULL && $item->value !== '') {
        return FALSE;
      }

      if (!isset($item->target_id) && !isset($item->value)) {
        $properties = $item->getProperties();
        foreach ($properties as $property) {
          $prop_value = $property->getValue();
          if ($prop_value !== NULL && $prop_value !== '') {
            return FALSE;
          }
        }
      }
    }

    return TRUE;
  }

  /**
   * Get all available fields with status information and translations.
   *
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param string $type_id
   *   The bundle/type ID.
   * @param array $entity_ids
   *   Array of entity IDs.
   * @param array $field_status
   *   Field status array (passed by reference).
   * @param bool $include_disabled
   *   Whether to include disabled fields.
   * @param string|null $langcode
   *   Primary language code.
   *
   * @return array
   *   Array of common field names.
   */
  public function getAllAvailableFieldsForEntities(
    string $entity_type,
    string $type_id,
    array $entity_ids,
    array &$field_status,
    bool $include_disabled = FALSE,
    ?string $langcode = NULL,
  ): array {
    if (empty($entity_ids)) {
      return [];
    }

    $field_status = [];
    $storage_type = $entity_type === 'product' ? 'commerce_product' : 'commerce_product_variation';
    $storage = $this->entityTypeManager->getStorage($storage_type);
    $all_fields_set = [];

    foreach ($entity_ids as $entity_id) {
      $entity = $storage->load($entity_id);
      if (!$entity) {
        continue;
      }

      // Get translation or original.
      $check_entity = $entity;
      if ($langcode && $entity->isTranslatable() && $entity->hasTranslation($langcode)) {
        $check_entity = $entity->getTranslation($langcode);
      }

      $all_available = $this->getEntityFields($check_entity, $include_disabled, FALSE);
      $empty_fields = $this->getEntityFields($check_entity, $include_disabled, TRUE);

      foreach ($all_available as $field_name) {
        $all_fields_set[$field_name] = TRUE;

        if (!isset($field_status[$field_name])) {
          $field_status[$field_name] = [
            'empty_count' => 0,
            'total_count' => 0,
            // NEW: Per-language statistics.
            'by_language' => [],
          ];
        }

        $field_status[$field_name]['total_count']++;

        if (in_array($field_name, $empty_fields)) {
          $field_status[$field_name]['empty_count']++;
        }

        // Collect per-language statistics if entity is translatable.
        if ($entity->isTranslatable()) {
          $translation_languages = $entity->getTranslationLanguages();
          foreach ($translation_languages as $lang) {
            $lang_id = $lang->getId();

            if (!isset($field_status[$field_name]['by_language'][$lang_id])) {
              $field_status[$field_name]['by_language'][$lang_id] = [
                'empty' => 0,
                'filled' => 0,
              ];
            }

            if ($entity->hasTranslation($lang_id)) {
              $translated = $entity->getTranslation($lang_id);
              if ($translated->hasField($field_name)) {
                $is_empty = $this->isFieldEmpty($translated->get($field_name));
                if ($is_empty) {
                  $field_status[$field_name]['by_language'][$lang_id]['empty']++;
                }
                else {
                  $field_status[$field_name]['by_language'][$lang_id]['filled']++;
                }
              }
            }
          }
        }
      }
    }

    return array_keys($all_fields_set);
  }

  /**
   * Apply field values to selected entities.
   *
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param array $selected_entities
   *   Array of entity IDs.
   * @param array $field_values
   *   Array of field values to apply.
   * @param array $field_overrides
   *   Array of override flags.
   * @param bool $dry_run
   *   If TRUE, only calculate statistics without saving.
   * @param string|null $langcode
   *   Language code for single language update.
   * @param array $apply_all_languages
   *   Array of field names to apply to all languages.
   *
   * @return array
   *   Statistics array with counts.
   */
  public function applyFieldValues(
    string $entity_type,
    array $selected_entities,
    array $field_values,
    array $field_overrides,
    bool $dry_run = FALSE,
    ?string $langcode = NULL,
    array $apply_all_languages = [],
  ): array {
    $updated_count = 0;
    $failed_count = 0;
    $skipped_count = 0;
    $total_field_updates = 0;
    $updated_entity_ids = [];
    $languages_selected = [];
    $languages_updated = [];
    $languages_skipped = [];
    $languages_field_updates = [];

    $storage_type = $entity_type === 'product' ? 'commerce_product' : 'commerce_product_variation';
    $storage = $this->entityTypeManager->getStorage($storage_type);

    foreach ($selected_entities as $entity_id) {
      $entity = $storage->load($entity_id);

      if (!$entity) {
        $failed_count++;
        continue;
      }

      try {
        // Determine which languages to process.
        $languages_to_process = $this->getLanguagesToProcess($entity, $langcode, $field_values, $apply_all_languages);

        $entity_updated = FALSE;

        foreach ($languages_to_process as $process_langcode) {
          // Track selected languages (all that we check).
          if (!isset($languages_selected[$process_langcode])) {
            $languages_selected[$process_langcode] = 0;
          }
          $languages_selected[$process_langcode]++;

          $translated_entity = $entity;

          // Get translation if different from original.
          if ($process_langcode && $entity->isTranslatable()) {
            if ($entity->hasTranslation($process_langcode)) {
              $translated_entity = $entity->getTranslation($process_langcode);
            }
            else {
              // Skip if translation doesn't exist.
              continue;
            }
          }

          $result = $this->processEntityFieldUpdates(
            $translated_entity,
            $field_values,
            $field_overrides,
            $dry_run,
            $entity_type,
            $entity_id,
            $process_langcode,
            $apply_all_languages
          );

          if ($result['entity_will_update']) {
            if (!$dry_run) {
              $translated_entity->save();
            }
            $entity_updated = TRUE;
            $total_field_updates += $result['entity_field_updates'];

            // Track languages that WILL BE updated.
            if (!isset($languages_updated[$process_langcode])) {
              $languages_updated[$process_langcode] = 0;
            }
            $languages_updated[$process_langcode]++;

            if (!isset($languages_field_updates[$process_langcode])) {
              $languages_field_updates[$process_langcode] = 0;
            }
            $languages_field_updates[$process_langcode] += $result['entity_field_updates'];
          }
          else {
            // Track languages that will be SKIPPED (no updates needed).
            if (!isset($languages_skipped[$process_langcode])) {
              $languages_skipped[$process_langcode] = 0;
            }
            $languages_skipped[$process_langcode]++;
          }

          $skipped_count += $result['entity_field_skips'];
        }

        if ($entity_updated) {
          $updated_count++;
          $updated_entity_ids[] = $entity_id;
        }
      }
      catch (\Exception $e) {
        if (!$dry_run) {
          $this->loggerFactory->get('product_manager_tool')->error(
            'Error updating @type @id: @error',
            [
              '@type' => $entity_type,
              '@id' => $entity_id,
              '@error' => $e->getMessage(),
            ]
          );
        }
        $failed_count++;
      }
    }

    return [
      'updated_count' => $updated_count,
      'updated_entity_ids' => $updated_entity_ids,
      'failed_count' => $failed_count,
      'skipped_count' => $skipped_count,
      'total_field_updates' => $total_field_updates,
      'languages_selected' => $languages_selected,
      'languages_updated' => $languages_updated,
      'languages_skipped' => $languages_skipped,
      'languages_field_updates' => $languages_field_updates,
    ];
  }

  /**
   * Get languages to process for entity.
   *
   * @param mixed $entity
   *   The entity.
   * @param string|null $langcode
   *   Primary language code.
   * @param array $field_values
   *   Field values being applied.
   * @param array $apply_all_languages
   *   Fields to apply to all languages.
   *
   * @return array
   *   Array of language codes to process.
   */
  private function getLanguagesToProcess(mixed $entity, ?string $langcode, array $field_values, array $apply_all_languages): array {
    $languages = [$langcode];

    if (!$entity->isTranslatable()) {
      return $languages;
    }

    // Check if any field should be applied to all languages.
    $has_all_language_fields = FALSE;
    foreach ($field_values as $field_name => $value) {
      if (!empty($apply_all_languages[$field_name])) {
        $has_all_language_fields = TRUE;
        break;
      }
    }

    if ($has_all_language_fields) {
      $translation_languages = $entity->getTranslationLanguages();
      $languages = array_keys($translation_languages);
    }

    return $languages;
  }

  /**
   * Process field updates for a single entity.
   *
   * @param mixed $entity
   *   The entity object.
   * @param array $field_values
   *   Field values to apply.
   * @param array $field_overrides
   *   Override flags.
   * @param bool $dry_run
   *   Dry run flag.
   * @param string $entity_type
   *   Entity type for logging.
   * @param int $entity_id
   *   Entity ID for logging.
   * @param string|null $langcode
   *   Current language being processed.
   * @param array $apply_all_languages
   *   Fields to apply to all languages.
   *
   * @return array
   *   Array with update statistics.
   */
  private function processEntityFieldUpdates(
    mixed $entity,
    array $field_values,
    array $field_overrides,
    bool $dry_run,
    string $entity_type,
    int $entity_id,
    ?string $langcode = NULL,
    array $apply_all_languages = [],
  ): array {
    $entity_will_update = FALSE;
    $entity_field_updates = 0;
    $entity_field_skips = 0;

    foreach ($field_values as $field_name => $field_value) {
      if (!$entity->hasField($field_name)) {
        continue;
      }

      $is_apply_all = !empty($apply_all_languages[$field_name]);
      $current_lang = $entity->language()->getId();

      if (!$is_apply_all && $langcode !== $current_lang) {
        continue;
      }

      $override_enabled = !empty($field_overrides[$field_name]);
      $field_is_empty = $this->isFieldEmpty($entity->get($field_name));

      if (!$field_is_empty && !$override_enabled) {
        $entity_field_skips++;
        continue;
      }

      $entity_field_updates++;
      $entity_will_update = TRUE;

      if (!$dry_run) {
        try {
          $processed_value = $this->fieldValueProcessor->processFieldValue($field_value, $field_name);
          $entity->set($field_name, $processed_value);
        }
        catch (\Exception $e) {
          $this->loggerFactory->get('product_manager_tool')->warning(
            'Could not set field @field on @type @id (@lang): @error',
            [
              '@field' => $field_name,
              '@type' => $entity_type,
              '@id' => $entity_id,
              '@lang' => $current_lang,
              '@error' => $e->getMessage(),
            ]
                  );
        }
      }
    }

    return [
      'entity_will_update' => $entity_will_update,
      'entity_field_updates' => $entity_field_updates,
      'entity_field_skips' => $entity_field_skips,
    ];
  }

}
