<?php

namespace Drupal\product_manager_tool\Service;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Service for building entity tables in field manager.
 */
class FieldTableBuilder {

  use StringTranslationTrait;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs a FieldTableBuilder object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(LanguageManagerInterface $language_manager) {
    $this->languageManager = $language_manager;
  }

  /**
   * Build entity table (generic for both products and variations).
   *
   * @param array $form
   *   The form array.
   * @param array $entities_with_empty_fields
   *   Array of entities with empty fields.
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param bool $include_checkboxes
   *   Whether to include selection checkboxes (default TRUE).
   * @param int $max_height
   *   Maximum height for scrollable container (default 500px).
   * @param mixed|null $field_column_label
   *   Label for the fields column (default: 'Empty Fields').
   * @param bool $show_languages_to_update
   *   Whether to show Translations to Update column (default FALSE).
   * @param \Drupal\Core\Form\FormStateInterface|null $form_state
   *   The form state (optional, for checkbox persistence).
   */
  public function buildEntityTable(
    array &$form,
    array $entities_with_empty_fields,
    string $entity_type,
    bool $include_checkboxes = TRUE,
    int $max_height = 500,
    mixed $field_column_label = NULL,
    bool $show_languages_to_update = FALSE,
    $form_state = NULL,
  ): void {
    if ($field_column_label === NULL) {
      $field_column_label = $this->t('Empty Fields');
    }

    if ($include_checkboxes) {
      $form['selection_controls'] = $this->buildSelectionControls($entity_type);
    }

    $header = $this->buildTableHeader($entity_type, $include_checkboxes, $field_column_label, $show_languages_to_update);
    $rows = $this->buildTableRows($entities_with_empty_fields, $entity_type, $include_checkboxes, $show_languages_to_update, $form_state);

    $form['table_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['table-responsive'],
        'style' => 'max-height: ' . $max_height . 'px; overflow-y: auto;',
      ],
      'entities_table' => [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No entities found.'),
        '#attributes' => ['class' => ['admin-list', 'sticky-table']],
      ],
    ];
  }

  /**
   * Build selection controls (Select All / Deselect All buttons).
   *
   * @param string $entity_type
   *   The entity type.
   *
   * @return array
   *   Selection controls render array.
   */
  private function buildSelectionControls(string $entity_type): array {
    return [
      '#type' => 'container',
      '#attributes' => ['style' => 'margin-bottom: 10px;'],
      'select_all' => [
        '#type' => 'button',
        '#value' => $this->t('Select All'),
        '#attributes' => [
          'class' => ['button', 'button--small'],
          'id' => 'product-select-all',
        ],
      ],
      'deselect_all' => [
        '#type' => 'button',
        '#value' => $this->t('Deselect All'),
        '#attributes' => [
          'class' => ['button', 'button--small'],
          'id' => 'product-deselect-all',
          'style' => 'margin-left: 5px;',
        ],
      ],
    ];
  }

  /**
   * Build table header.
   *
   * @param string $entity_type
   *   The entity type.
   * @param bool $include_checkboxes
   *   Whether to include checkboxes.
   * @param mixed $field_column_label
   *   Label for fields column.
   * @param bool $show_languages_to_update
   *   Whether to show Translations to Update column.
   *
   * @return array
   *   Table header array.
   */
  private function buildTableHeader(
    string $entity_type,
    bool $include_checkboxes,
    mixed $field_column_label,
    bool $show_languages_to_update = FALSE,
  ): array {
    if ($entity_type === 'product') {
      if ($show_languages_to_update) {
        $header = $include_checkboxes
          ? [
            $this->t('Select'),
            $this->t('Product ID'),
            $this->t('Title'),
            $this->t('Language'),
            $this->t('Translations to Update'),
            $field_column_label,
            $this->t('Operations'),
          ]
          : [
            $this->t('Product ID'),
            $this->t('Title'),
            $this->t('Language'),
            $this->t('Translations to Update'),
            $field_column_label,
            $this->t('Operations'),
          ];
      }
      else {
        $header = $include_checkboxes
          ? [
            $this->t('Select'),
            $this->t('Product ID'),
            $this->t('Title'),
            $this->t('Language'),
            $field_column_label,
            $this->t('Operations'),
          ]
          : [
            $this->t('Product ID'),
            $this->t('Title'),
            $this->t('Language'),
            $field_column_label,
            $this->t('Operations'),
          ];
      }
    }
    else {
      if ($show_languages_to_update) {
        $header = $include_checkboxes
          ? [
            $this->t('Select'),
            $this->t('Variation ID'),
            $this->t('SKU'),
            $this->t('Title'),
            $this->t('Language'),
            $this->t('Translations to Update'),
            $field_column_label,
            $this->t('Operations'),
          ]
          : [
            $this->t('Variation ID'),
            $this->t('SKU'),
            $this->t('Title'),
            $this->t('Language'),
            $this->t('Translations to Update'),
            $field_column_label,
            $this->t('Operations'),
          ];
      }
      else {
        $header = $include_checkboxes
          ? [
            $this->t('Select'),
            $this->t('Variation ID'),
            $this->t('SKU'),
            $this->t('Title'),
            $this->t('Language'),
            $field_column_label,
            $this->t('Operations'),
          ]
          : [
            $this->t('Variation ID'),
            $this->t('SKU'),
            $this->t('Title'),
            $this->t('Language'),
            $field_column_label,
            $this->t('Operations'),
          ];
      }
    }

    return $header;
  }

  /**
   * Build table rows.
   *
   * @param array $entities_with_empty_fields
   *   Entities data.
   * @param string $entity_type
   *   The entity type.
   * @param bool $include_checkboxes
   *   Whether to include checkboxes.
   * @param bool $show_languages_to_update
   *   Whether to show Translations to Update column.
   * @param \Drupal\Core\Form\FormStateInterface|null $form_state
   *   The form state (optional).
   *
   * @return array
   *   Table rows array.
   */
  private function buildTableRows(
    array $entities_with_empty_fields,
    string $entity_type,
    bool $include_checkboxes,
    bool $show_languages_to_update = FALSE,
    ?FormStateInterface $form_state = NULL,
  ): array {
    $rows = [];

    foreach ($entities_with_empty_fields as $entity_data) {
      $entity = $entity_data[$entity_type === 'product' ? 'product' : 'variation'];
      $empty_fields = $entity_data['empty_fields'];

      if ($entity_type === 'product') {
        $rows[] = $this->buildProductRow($entity, $empty_fields, $include_checkboxes, $entity_data, $show_languages_to_update, $form_state);
      }
      else {
        $rows[] = $this->buildVariationRow($entity, $empty_fields, $include_checkboxes, $entity_data, $show_languages_to_update, $form_state);
      }
    }

    return $rows;
  }

  /**
   * Build product table row with language info.
   *
   * @param mixed $product
   *   The product entity.
   * @param array $empty_fields
   *   Empty field names.
   * @param bool $include_checkboxes
   *   Whether to include checkbox.
   * @param array $entity_data
   *   Entity data including language info.
   * @param bool $show_languages_to_update
   *   Whether to show Translations to Update column.
   * @param \Drupal\Core\Form\FormStateInterface|null $form_state
   *   The form state (optional).
   *
   * @return array
   *   Table row array.
   */
  private function buildProductRow(
    $product,
    array $empty_fields,
    bool $include_checkboxes,
    array $entity_data = [],
    bool $show_languages_to_update = FALSE,
    ?FormStateInterface $form_state = NULL,
  ): array {
    $row = [];

    if ($include_checkboxes) {
      $row[] = $this->buildCheckbox($product->id(), 'entity-checkbox', $form_state);
    }

    $product_url = Url::fromRoute('entity.commerce_product.canonical', ['commerce_product' => $product->id()]);
    $title_link = [
      '#type' => 'link',
      '#title' => $product->getTitle(),
      '#url' => $product_url,
      '#attributes' => ['target' => '_blank'],
    ];

    $row[] = $product->id();
    $row[] = ['data' => $title_link];

    // Add language column.
    $row[] = $this->buildLanguageCell($product, $entity_data);

    // Add Translations to Update column - only if requested.
    if ($show_languages_to_update) {
      $row[] = $this->buildLanguagesToUpdateCell($entity_data);
    }

    $row[] = $this->getFieldLabelsString($product, $empty_fields, 'product');

    // Build operations with dropdown support.
    $languages_to_update = $show_languages_to_update ? ($entity_data['languages_to_update'] ?? []) : [];
    $viewing_langcode = $entity_data['langcode'] ?? NULL;
    $row[] = $this->buildOperationsDropdown($product, 'product', $languages_to_update, $viewing_langcode);

    return $row;
  }

  /**
   * Build language cell with badge.
   *
   * @param mixed $entity
   *   The entity.
   * @param array $entity_data
   *   Entity data.
   *
   * @return array
   *   Language cell render array.
   */
  private function buildLanguageCell(mixed $entity, array $entity_data = []): array {
    $langcode = $entity->language()->getId();
    $language = $this->languageManager->getLanguage($langcode);
    $language_name = $language ? $language->getName() : $langcode;

    $is_translation = $entity_data['is_translation'] ?? FALSE;
    $badge_text = $is_translation ? $this->t('Translation') : $this->t('Original');

    return [
      'data' => [
        '#markup' => '<div style="white-space: nowrap;">' .
        '<strong>' . Html::escape($language_name) . '</strong>' .
        '<br><span >' .
        Html::escape($badge_text) .
        '</span></div>',
      ],
    ];
  }

  /**
   * Build Translations to Update cell.
   *
   * @param array $entity_data
   *   Entity data.
   *
   * @return array
   *   Translations to Update cell render array.
   */
  private function buildLanguagesToUpdateCell(array $entity_data = []): array {
    // Only show in confirmation screen - check if languages_to_update exists.
    if (empty($entity_data['languages_to_update'])) {
      return ['data' => ['#markup' => '-']];
    }

    $language_names = [];

    foreach ($entity_data['languages_to_update'] as $langcode) {
      $language = $this->languageManager->getLanguage($langcode);
      $language_names[] = $language ? $language->getName() : $langcode;
    }

    $count = count($language_names);
    $display_text = implode(', ', $language_names);

    return [
      'data' => [
        '#markup' => Html::escape($display_text) . ' (' . (int) $count . ')',
      ],
    ];
  }

  /**
   * Build variation table row with language info.
   *
   * @param mixed $variation
   *   The variation entity.
   * @param array $empty_fields
   *   Empty field names.
   * @param bool $include_checkboxes
   *   Whether to include checkbox.
   * @param array $entity_data
   *   Entity data including language info.
   * @param bool $show_languages_to_update
   *   Whether to show Translations to Update column.
   * @param \Drupal\Core\Form\FormStateInterface|null $form_state
   *   The form state (optional).
   *
   * @return array
   *   Table row array.
   */
  private function buildVariationRow(
    mixed $variation,
    array $empty_fields,
    bool $include_checkboxes,
    array $entity_data = [],
    bool $show_languages_to_update = FALSE,
    ?FormStateInterface $form_state = NULL,
  ): array {
    $row = [];

    if ($include_checkboxes) {
      $row[] = $this->buildCheckbox($variation->id(), 'entity-checkbox', $form_state);
    }

    $variation_title = $variation->getTitle() ?: $this->t('(No title)');
    $product_id = $variation->getProductId();

    if ($product_id) {
      $product_url = Url::fromRoute('entity.commerce_product.canonical', ['commerce_product' => $product_id]);
      $title_link = [
        '#type' => 'link',
        '#title' => $variation_title,
        '#url' => $product_url,
        '#attributes' => ['target' => '_blank'],
      ];
    }
    else {
      $title_link = ['#markup' => $variation_title];
    }

    $row[] = $variation->id();
    $row[] = $variation->getSku() ?: $this->t('(No SKU)');
    $row[] = ['data' => $title_link];
    $row[] = $this->buildLanguageCell($variation, $entity_data);

    // Add Translations to Update column - only if requested.
    if ($show_languages_to_update) {
      $row[] = $this->buildLanguagesToUpdateCell($entity_data);
    }

    $row[] = $this->getFieldLabelsString($variation, $empty_fields, 'variation');

    // Build operations with dropdown support.
    $languages_to_update = $show_languages_to_update ? ($entity_data['languages_to_update'] ?? []) : [];
    $viewing_langcode = $entity_data['langcode'] ?? NULL;
    $row[] = $this->buildOperationsDropdown($variation, 'variation', $languages_to_update, $viewing_langcode);

    return $row;
  }

  /**
   * Build checkbox element.
   *
   * @param int $entity_id
   *   Entity ID.
   * @param string $checkbox_class
   *   Checkbox CSS class.
   * @param \Drupal\Core\Form\FormStateInterface|null $form_state
   *   The form state (optional).
   *
   * @return array
   *   Checkbox render array.
   */
  private function buildCheckbox(int $entity_id, string $checkbox_class, ?FormStateInterface $form_state = NULL): array {
    $stored_selected_entities = [];

    if ($form_state !== NULL) {
      // Check form state.
      $stored_selected_entities = $form_state->get('selected_entities') ?? [];

      // Also check user input as a fallback.
      $user_input = $form_state->getUserInput();
      if (isset($user_input['selected_entities']) && is_array($user_input['selected_entities'])) {
        $user_input_selected = array_filter($user_input['selected_entities']);
        // Merge both sources, user input takes priority.
        $stored_selected_entities = array_unique(array_merge(
          $stored_selected_entities,
          array_values($user_input_selected)
        ));
      }
    }

    // Check if this entity is selected.
    $is_selected = in_array($entity_id, $stored_selected_entities);

    $checkbox = [
      'data' => [
        '#type' => 'checkbox',
        '#name' => 'selected_entities[' . $entity_id . ']',
        '#attributes' => ['class' => [$checkbox_class]],
        '#return_value' => $entity_id,
        '#default_value' => $is_selected ? $entity_id : 0,
      ],
    ];

    // For table checkboxes, we need to manually set the checked attribute
    // because Drupal's table rendering doesn't always respect #default_value.
    if ($is_selected) {
      $checkbox['data']['#attributes']['checked'] = 'checked';
    }

    return $checkbox;
  }

  /**
   * Get field labels as string.
   *
   * @param mixed $entity
   *   The entity object.
   * @param array $empty_fields
   *   Array of empty field names.
   * @param string $entity_type
   *   The entity type.
   *
   * @return string
   *   Comma-separated field labels with count.
   */
  private function getFieldLabelsString(mixed $entity, array $empty_fields, string $entity_type): string {
    if (empty($empty_fields)) {
      return $entity_type === 'product'
        ? $this->t('All product fields filled')
        : $this->t('All variation fields filled');
    }

    $empty_field_labels = array_map(function ($field_name) use ($entity) {
      $field_definition = $entity->getFieldDefinition($field_name);
      return $field_definition ? $field_definition->getLabel() : $field_name;
    }, $empty_fields);

    return implode(', ', $empty_field_labels) . ' (' . count($empty_fields) . ')';
  }

  /**
   * Build operations dropdown for entity with language support.
   *
   * @param mixed $entity
   *   The entity (product or variation).
   * @param string $entity_type
   *   The entity type ('product' or 'variation').
   * @param array $languages_to_update
   *   Array of language codes that will be updated.
   * @param string|null $viewing_langcode
   *   The language code being viewed (from selection screen).
   *
   * @return array
   *   Operations render array with dropbutton.
   */
  private function buildOperationsDropdown(
    mixed $entity,
    string $entity_type,
    array $languages_to_update = [],
    ?string $viewing_langcode = NULL,
  ): array {
    $current_langcode = $viewing_langcode ?? $entity->language()->getId();
    $current_language_object = $this->languageManager->getLanguage($current_langcode);

    $links = [];

    if ($entity_type === 'product') {
      // Primary edit link for current language.
      $edit_url = Url::fromRoute('entity.commerce_product.edit_form', [
        'commerce_product' => $entity->id(),
      ], [
        'language' => $current_language_object,
      ]);

      $links['edit'] = [
        'title' => $this->t('Edit'),
        'url' => $edit_url,
        'attributes' => ['target' => '_blank'],
      ];

      // Add edit links for additional languages.
      if (!empty($languages_to_update) && count($languages_to_update) > 1) {
        // If multiple languages to update, add dropdown for each.
        foreach ($languages_to_update as $langcode) {
          if ($langcode === $current_langcode) {
            // Skip current language as it's already the main Edit button.
            continue;
          }

          $lang = $this->languageManager->getLanguage($langcode);
          $lang_name = $lang ? $lang->getName() : $langcode;
          $lang_object = $this->languageManager->getLanguage($langcode);

          $edit_trans_url = Url::fromRoute('entity.commerce_product.edit_form', [
            'commerce_product' => $entity->id(),
          ], [
            'language' => $lang_object,
          ]);

          $links['edit_' . $langcode] = [
            'title' => $this->t('Edit (@lang)', ['@lang' => $lang_name]),
            'url' => $edit_trans_url,
            'attributes' => ['target' => '_blank'],
          ];
        }
      }
    }
    else {
      // Variation.
      $product_id = $entity->getProductId();

      if (!$product_id) {
        return ['data' => []];
      }

      // Primary edit link for current language.
      $edit_url = Url::fromRoute('entity.commerce_product_variation.edit_form', [
        'commerce_product' => $product_id,
        'commerce_product_variation' => $entity->id(),
      ], [
        'language' => $current_language_object,
      ]);

      $links['edit'] = [
        'title' => $this->t('Edit'),
        'url' => $edit_url,
        'attributes' => ['target' => '_blank'],
      ];

      // Add edit links for additional languages.
      if (!empty($languages_to_update) && count($languages_to_update) > 1) {
        foreach ($languages_to_update as $langcode) {
          if ($langcode === $current_langcode) {
            continue;
          }

          $lang = $this->languageManager->getLanguage($langcode);
          $lang_name = $lang ? $lang->getName() : $langcode;
          $lang_object = $this->languageManager->getLanguage($langcode);

          $edit_trans_url = Url::fromRoute('entity.commerce_product_variation.edit_form', [
            'commerce_product' => $product_id,
            'commerce_product_variation' => $entity->id(),
          ], [
            'language' => $lang_object,
          ]);

          $links['edit_' . $langcode] = [
            'title' => $this->t('Edit (@lang)', ['@lang' => $lang_name]),
            'url' => $edit_trans_url,
            'attributes' => ['target' => '_blank'],
          ];
        }
      }
    }

    // Return as operations dropdown.
    return [
      'data' => [
        '#type' => 'operations',
        '#links' => $links,
        '#attributes' => ['class' => ['dropbutton--extrasmall']],
      ],
    ];
  }

}
