<?php

namespace Drupal\product_manager_tool\Service;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\field\FieldConfigInterface;

/**
 * Service for processing field values (IEF, media, text fields).
 */
class FieldValueProcessor {

  /**
   * Module logger channel name.
   */
  const LOGGER_CHANNEL = 'product_manager_tool';

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The field access validator.
   *
   * @var \Drupal\product_manager_tool\Service\FieldAccessValidator
   */
  protected $fieldAccessValidator;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a FieldValueProcessor object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\product_manager_tool\Service\FieldAccessValidator $field_access_validator
   *   The field access validator.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    LoggerChannelFactoryInterface $logger_factory,
    FieldAccessValidator $field_access_validator,
    AccountProxyInterface $current_user,
  ) {
    $this->loggerFactory = $logger_factory;
    $this->fieldAccessValidator = $field_access_validator;
    $this->currentUser = $current_user;
  }

  /**
   * Check if a field value is empty.
   *
   * @param mixed $value
   *   The field value to check.
   *
   * @return bool
   *   TRUE if the value is empty, FALSE otherwise.
   */
  public function isFieldValueEmpty(mixed $value): bool {
    if ($value === NULL || $value === '' || $value === []) {
      return TRUE;
    }

    if (!is_array($value)) {
      return FALSE;
    }

    if (isset($value['_ief_entities'])) {
      return $this->isIefValueEmpty($value['_ief_entities']);
    }

    if (isset($value['selection'])) {
      return $this->isSelectionEmpty($value['selection']);
    }

    if (isset($value['media_library_selection'])) {
      return $this->isMediaLibraryEmpty($value['media_library_selection']);
    }

    if (isset($value['target_id'])) {
      return empty($value['target_id']);
    }

    if (isset($value['widget'])) {
      return $this->isFieldValueEmpty($value['widget']);
    }

    if (isset($value['value'])) {
      $text_value = is_array($value['value']) ? '' : trim($value['value']);
      return empty($text_value);
    }

    $meaningful_values = $this->extractMeaningfulValues($value);

    if (empty($meaningful_values)) {
      return TRUE;
    }

    foreach ($meaningful_values as $key => $item) {
      if (is_numeric($key) && is_array($item)) {
        if (!empty($item['target_id'])) {
          return FALSE;
        }
        if (!isset($item['target_id']) && !$this->isFieldValueEmpty($item)) {
          return FALSE;
        }
      }
      elseif (!$this->isFieldValueEmpty($item)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Check if IEF value is empty.
   *
   * @param array $ief_data
   *   IEF entities data.
   *
   * @return bool
   *   TRUE if empty.
   */
  private function isIefValueEmpty(array $ief_data): bool {
    if (empty($ief_data)) {
      return TRUE;
    }

    foreach ($ief_data as $entity_data) {
      if (isset($entity_data['entity']) && is_object($entity_data['entity'])) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Check if selection value is empty.
   *
   * @param mixed $selection
   *   Selection data.
   *
   * @return bool
   *   TRUE if empty.
   */
  private function isSelectionEmpty(mixed $selection): bool {
    if (!is_array($selection)) {
      return TRUE;
    }

    foreach ($selection as $item) {
      if (is_array($item) && !empty($item['target_id'])) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Check if media library selection is empty.
   *
   * @param mixed $selection
   *   Media library selection.
   *
   * @return bool
   *   TRUE if empty.
   */
  private function isMediaLibraryEmpty(mixed $selection): bool {
    if (is_array($selection)) {
      return $this->isFieldValueEmpty($selection);
    }

    $selection = trim($selection);
    if ($selection === '' || $selection === NULL) {
      return TRUE;
    }

    $ids = array_filter(explode(',', $selection), function ($id) {
      return !empty(trim($id));
    });

    return empty($ids);
  }

  /**
   * Extract meaningful values from array.
   *
   * @param array $value
   *   Value array.
   *
   * @return array
   *   Meaningful values.
   */
  private function extractMeaningfulValues(array $value): array {
    $control_keys = [
      'open_button',
      'media_library_update_widget',
      '_weight',
      'add_more',
      'remove_button',
      '_original_delta',
      '_actions',
      'actions',
      'info',
    ];

    $meaningful_values = [];

    foreach ($value as $key => $item) {
      if (in_array($key, $control_keys, TRUE)) {
        continue;
      }
      if (is_object($item)) {
        continue;
      }
      if (strpos($key, 'override_') === 0) {
        continue;
      }

      $meaningful_values[$key] = $item;
    }

    return $meaningful_values;
  }

  /**
   * Process field value for special widgets.
   *
   * @param mixed $value
   *   The raw field value.
   * @param string $field_name
   *   The field name.
   *
   * @return mixed
   *   The processed field value.
   */

  /**
   * Process field value with security validation.
   *
   * @param mixed $value
   *   The field value to process.
   * @param string $field_name
   *   The field name.
   *
   * @return mixed
   *   Processed and validated field value.
   */
  public function processFieldValue(mixed $value, string $field_name): mixed {
    // Validate field name to prevent injection attacks.
    if (!preg_match('/^[a-z][a-z0-9_]*$/', $field_name)) {
      $this->loggerFactory->get(self::LOGGER_CHANNEL)->error('Invalid field name: @field', [
        '@field' => $field_name,
      ]);
      return NULL;
    }

    if (!is_array($value)) {
      return $value;
    }

    if ($this->isTextFieldValue($value)) {
      return $this->processTextFieldValue($value);
    }

    if (isset($value['_ief_entities'])) {
      return $this->processIefValue($value['_ief_entities'], $field_name);
    }

    // Check media_library_selection but only if it has actual value.
    if (!empty($value['media_library_selection'])) {
      return $this->processMediaLibraryValue($value['media_library_selection']);
    }

    // Check selection array (media library widget uses this).
    if (!empty($value['selection'])) {
      return $this->processSelection($value['selection']);
    }

    if (isset($value['widget'])) {
      return $this->processFieldValue($value['widget'], $field_name);
    }

    if ($this->isSingleEntityReference($value)) {
      return $this->processSingleEntityReference($value);
    }

    if ($this->isMultiValueField($value)) {
      return $this->processMultiValueField($value);
    }

    return $value;
  }

  /**
   * Process selection value (media library widget).
   *
   * @param mixed $selection
   *   Selection value - should be array of items with target_id.
   *
   * @return array
   *   Processed selection as array of target_id items.
   */
  private function processSelection(mixed $selection): array {
    if (empty($selection)) {
      return [];
    }

    // If it's already an array, process each item.
    if (is_array($selection)) {
      $result = [];
      foreach ($selection as $key => $item) {
        // Skip non-numeric keys ('open_button', 'media_library_selection', etc)
        // Only process numeric indices which contain actual media items.
        if (!is_numeric($key)) {
          continue;
        }

        if (is_numeric($item)) {
          // Simple numeric ID - validate it's positive.
          $id = intval($item);
          if ($id > 0) {
            $result[] = ['target_id' => $id];
          }
        }
        elseif (is_array($item) && isset($item['target_id'])) {
          // Already in target_id format - validate and extract.
          $id = is_numeric($item['target_id']) ? intval($item['target_id']) : 0;
          if ($id > 0) {
            $field_item = ['target_id' => $id];

            // Preserve other important properties if they exist.
            if (isset($item['target_revision_id']) && is_numeric($item['target_revision_id'])) {
              $field_item['target_revision_id'] = intval($item['target_revision_id']);
            }
            if (isset($item['alt']) && is_string($item['alt'])) {
              // Sanitize alt text for XSS protection.
              $field_item['alt'] = $this->fieldAccessValidator->sanitizePlainText($item['alt']);
            }
            if (isset($item['title']) && is_string($item['title'])) {
              // Sanitize title text for XSS protection.
              $field_item['title'] = $this->fieldAccessValidator->sanitizePlainText($item['title']);
            }

            $result[] = $field_item;
          }
        }
        elseif (is_array($item)) {
          // Nested array without target_id, process recursively.
          $nested = $this->processSelection($item);
          if (!empty($nested)) {
            $result = array_merge($result, $nested);
          }
        }
      }

      return $result;
    }

    // If it's a string (comma-separated IDs)
    if (is_string($selection)) {
      $ids = array_filter(explode(',', $selection));
      return array_map(function ($id) {
        return ['target_id' => intval(trim($id))];
      }, $ids);
    }

    // Single numeric value.
    if (is_numeric($selection)) {
      return [['target_id' => intval($selection)]];
    }

    return [];
  }

  /**
   * Check if value is text field value.
   *
   * @param array $value
   *   Value array.
   *
   * @return bool
   *   TRUE if text field value.
   */
  private function isTextFieldValue(array $value): bool {
    return isset($value['value']) && count(array_filter(array_keys($value), 'is_numeric')) === 0;
  }

  /**
   * Process text field value.
   *
   * @param array $value
   *   Text field value.
   *
   * @return array
   *   Processed value.
   */
  private function processTextFieldValue(array $value): array {
    $item = ['value' => $value['value']];

    if (isset($value['format'])) {
      $item['format'] = $value['format'];
    }
    if (isset($value['summary'])) {
      $item['summary'] = $value['summary'];
    }

    return [$item];
  }

  /**
   * Process IEF value.
   *
   * @param array $ief_entities
   *   IEF entities data.
   * @param string $field_name
   *   Field name for error reporting.
   *
   * @return array
   *   Processed IEF values.
   */
  private function processIefValue(array $ief_entities, string $field_name): array {
    $clean_values = [];
    $save_errors = [];

    foreach ($ief_entities as $delta => $entity_data) {
      if (!isset($entity_data['entity']) || !is_object($entity_data['entity'])) {
        continue;
      }

      $entity = $entity_data['entity'];

      // Security: Check if user can modify this entity.
      if (!$this->fieldAccessValidator->canModifyIefEntity($entity)) {
        $save_errors[] = sprintf(
          'Access denied for entity at delta %d',
          $delta
        );
        continue;
      }

      if (method_exists($entity, 'isNew') && $entity->isNew()) {
        $needs_save = $entity_data['needs_save'] ?? TRUE;
        if ($needs_save) {
          try {
            // Set owner to current user for new entities.
            if ($entity->hasField('uid') && empty($entity->get('uid')->target_id)) {
              $entity->set('uid', $this->currentUser->id());
            }

            $entity->save();
          }
          catch (\Exception $e) {
            $error_msg = sprintf('Failed to save IEF entity at delta %d: %s', $delta, $e->getMessage());
            $save_errors[] = $error_msg;
            continue;
          }
        }
      }

      if (method_exists($entity, 'id') && $entity->id()) {
        $item = ['target_id' => $entity->id()];

        if (method_exists($entity, 'getRevisionId') && $entity->getRevisionId()) {
          $item['target_revision_id'] = $entity->getRevisionId();
        }
        $clean_values[] = $item;
      }
    }

    if (!empty($save_errors)) {
      \Drupal::messenger()->addWarning(t('Some entities in @field could not be saved: @errors', [
        '@field' => $field_name,
        '@errors' => implode('; ', $save_errors),
      ]));
    }

    return $clean_values;
  }

  /**
   * Process media library value.
   *
   * @param mixed $value
   *   Media library selection value.
   *
   * @return array
   *   Processed media values.
   */
  private function processMediaLibraryValue(mixed $value): array {
    if (empty($value)) {
      return [];
    }

    $media_ids = array_filter(explode(',', $value));
    return array_map(function ($id) {
      return ['target_id' => intval($id)];
    }, $media_ids);
  }

  /**
   * Check if value is single entity reference.
   *
   * @param array $value
   *   Value array.
   *
   * @return bool
   *   TRUE if single entity reference.
   */
  private function isSingleEntityReference(array $value): bool {
    return isset($value['target_id']) && count(array_filter(array_keys($value), 'is_numeric')) === 0;
  }

  /**
   * Process single entity reference.
   *
   * @param array $value
   *   Single entity reference value.
   *
   * @return array
   *   Processed value.
   */
  private function processSingleEntityReference(array $value): array {
    $item = ['target_id' => $value['target_id']];
    if (isset($value['target_revision_id'])) {
      $item['target_revision_id'] = $value['target_revision_id'];
    }
    return [$item];
  }

  /**
   * Check if value is multi-value field.
   *
   * @param array $value
   *   Value array.
   *
   * @return bool
   *   TRUE if multi-value field.
   */
  private function isMultiValueField(array $value): bool {
    $numeric_keys = array_filter(array_keys($value), 'is_numeric');
    return !empty($numeric_keys);
  }

  /**
   * Process multi-value field.
   *
   * @param array $value
   *   Multi-value field data.
   *
   * @return array
   *   Processed values.
   */
  private function processMultiValueField(array $value): array {
    $clean_values = [];
    $numeric_keys = array_filter(array_keys($value), 'is_numeric');

    foreach ($numeric_keys as $delta) {
      if (!is_array($value[$delta])) {
        continue;
      }

      if (!empty($value[$delta]['target_id'])) {
        $item = ['target_id' => $value[$delta]['target_id']];
        if (!empty($value[$delta]['target_revision_id'])) {
          $item['target_revision_id'] = $value[$delta]['target_revision_id'];
        }
        $clean_values[] = $item;
      }
      elseif (isset($value[$delta]['value'])) {
        $item = ['value' => $value[$delta]['value']];
        if (isset($value[$delta]['format'])) {
          $item['format'] = $value[$delta]['format'];
        }
        if (isset($value[$delta]['summary'])) {
          $item['summary'] = $value[$delta]['summary'];
        }
        $clean_values[] = $item;
      }
    }

    return empty($clean_values) ? [] : $clean_values;
  }

  /**
   * Extract IEF entities from form_state with nested support.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $field_name
   *   The field name.
   *
   * @return array
   *   Array of entity data.
   */
  public function extractIefEntitiesFromFormState(FormStateInterface $form_state, string $field_name): array {
    $all_ief_data = $form_state->get('inline_entity_form');

    if (empty($all_ief_data)) {
      return [];
    }

    $extracted_entities = [];
    $clean_field_name = str_replace('_container', '', $field_name);

    $field_patterns = [
      $clean_field_name,
      $field_name,
      "field_values-$clean_field_name-form",
    ];

    $main_ief_id = $this->findMainIefId($all_ief_data, $field_patterns);

    if (!$main_ief_id) {
      return [];
    }

    $ief_info = $all_ief_data[$main_ief_id];

    if (isset($ief_info['entities']) && is_array($ief_info['entities'])) {
      foreach ($ief_info['entities'] as $delta => $entity_item) {
        if (isset($entity_item['entity']) && is_object($entity_item['entity'])) {
          $extracted_entities[$delta] = [
            'entity' => $entity_item['entity'],
            'needs_save' => $entity_item['needs_save'] ?? TRUE,
          ];
        }
      }
    }

    if (empty($extracted_entities)) {
      return [];
    }

    $this->processNestedIefEntities($extracted_entities, $all_ief_data, $main_ief_id, $field_name);

    return $extracted_entities;
  }

  /**
   * Find main IEF ID.
   *
   * @param array $all_ief_data
   *   All IEF data.
   * @param array $field_patterns
   *   Field patterns to match.
   *
   * @return string|null
   *   IEF ID or NULL.
   */
  private function findMainIefId(array $all_ief_data, array $field_patterns): ?string {
    foreach ($all_ief_data as $ief_id => $ief_info) {
      foreach ($field_patterns as $pattern) {
        if ($ief_id === $pattern || strpos($ief_id, $pattern) === 0) {
          $after_pattern = str_replace($pattern, '', $ief_id);
          if (empty($after_pattern) || $after_pattern === '-form') {
            return $ief_id;
          }
        }
      }
    }

    return NULL;
  }

  /**
   * Process nested IEF entities.
   *
   * @param array $extracted_entities
   *   Extracted parent entities (passed by reference).
   * @param array $all_ief_data
   *   All IEF data.
   * @param string $main_ief_id
   *   Main IEF ID.
   * @param string $field_name
   *   Field name.
   */
  private function processNestedIefEntities(array &$extracted_entities, array $all_ief_data, string $main_ief_id, string $field_name): void {
    foreach ($extracted_entities as $delta => $entity_data) {
      $parent_entity = $entity_data['entity'];
      $field_definitions = $parent_entity->getFieldDefinitions();

      foreach ($field_definitions as $nested_field_name => $field_definition) {
        $is_base_field = method_exists($field_definition, 'isBaseField')
          ? $field_definition->isBaseField()
          : !($field_definition instanceof FieldConfigInterface);

        if ($is_base_field) {
          continue;
        }

        $field_type = $field_definition->getType();
        if (!in_array($field_type, ['entity_reference', 'entity_reference_revisions'])) {
          continue;
        }

        $nested_ief_id = $this->findNestedIefId($all_ief_data, $main_ief_id, $delta, $nested_field_name);

        if ($nested_ief_id && isset($all_ief_data[$nested_ief_id]['entities'])) {
          $nested_values = $this->processNestedEntities(
            $all_ief_data[$nested_ief_id]['entities'],
            $nested_field_name
          );

          if (!empty($nested_values)) {
            $parent_entity->set($nested_field_name, $nested_values);
            $extracted_entities[$delta]['needs_save'] = TRUE;
          }
        }
      }

      if ($entity_data['needs_save'] && method_exists($parent_entity, 'isNew') && $parent_entity->isNew()) {
        try {
          $parent_entity->save();
          $extracted_entities[$delta]['entity'] = $parent_entity;
        }
        catch (\Exception $e) {
          $this->loggerFactory->get(self::LOGGER_CHANNEL)->error(
            'Failed to save parent entity for @field: @error',
            ['@field' => $field_name, '@error' => $e->getMessage()]
                  );
        }
      }
    }
  }

  /**
   * Find nested IEF ID.
   *
   * @param array $all_ief_data
   *   All IEF data.
   * @param string $parent_ief_id
   *   Parent IEF ID.
   * @param int $delta
   *   Delta of parent entity.
   * @param string $nested_field_name
   *   Nested field name.
   *
   * @return string|null
   *   Nested IEF ID or NULL.
   */
  private function findNestedIefId(array $all_ief_data, string $parent_ief_id, int $delta, string $nested_field_name): ?string {
    $possible_patterns = [
      $parent_ief_id . '-' . $delta . '-' . $nested_field_name . '-form',
      $parent_ief_id . '-' . $delta . '-' . $nested_field_name,
    ];

    foreach ($all_ief_data as $ief_id => $ief_info) {
      if (in_array($ief_id, $possible_patterns, TRUE)) {
        return $ief_id;
      }

      if (str_contains($ief_id, $parent_ief_id) && str_contains($ief_id, $nested_field_name)) {
        if (str_contains($ief_id, '-' . $delta . '-') || str_contains($ief_id, '[' . $delta . ']')) {
          return $ief_id;
        }
      }
    }

    return NULL;
  }

  /**
   * Process nested entities.
   *
   * @param array $nested_entities
   *   Nested entities data.
   * @param string $nested_field_name
   *   Nested field name for error reporting.
   *
   * @return array
   *   Processed nested values.
   */
  private function processNestedEntities(array $nested_entities, string $nested_field_name): array {
    $nested_values = [];

    foreach ($nested_entities as $nested_delta => $nested_entity_item) {
      if (!isset($nested_entity_item['entity']) || !is_object($nested_entity_item['entity'])) {
        continue;
      }

      $nested_entity = $nested_entity_item['entity'];

      // Security: Check if user can modify this nested entity.
      if (!$this->fieldAccessValidator->canModifyIefEntity($nested_entity)) {
        $this->loggerFactory->get(self::LOGGER_CHANNEL)->warning(
          'Access denied for nested entity in @field',
          ['@field' => $nested_field_name]
        );
        continue;
      }

      $needs_save = $nested_entity_item['needs_save'] ?? TRUE;

      if (method_exists($nested_entity, 'isNew') && $nested_entity->isNew() && $needs_save) {
        try {
          // Set owner to current user for new entities.
          if ($nested_entity->hasField('uid') && empty($nested_entity->get('uid')->target_id)) {
            $nested_entity->set('uid', $this->currentUser->id());
          }

          $nested_entity->save();
        }
        catch (\Exception $e) {
          $this->loggerFactory->get(self::LOGGER_CHANNEL)->error(
            'Failed to save nested entity in @field: @error',
            ['@field' => $nested_field_name, '@error' => $e->getMessage()]
                  );
          continue;
        }
      }

      if (method_exists($nested_entity, 'id') && $nested_entity->id()) {
        $item = ['target_id' => $nested_entity->id()];
        if (method_exists($nested_entity, 'getRevisionId') && $nested_entity->getRevisionId()) {
          $item['target_revision_id'] = $nested_entity->getRevisionId();
        }
        $nested_values[] = $item;
      }
    }

    return $nested_values;
  }

}
