<?php

namespace Drupal\progressive_accessibility_widget\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a block for accessibility tool.
 *
 * @Block(
 *   id = "progressive_accessibility_block",
 *   admin_label = @Translation("Progressive Accessibility Block"),
 * )
 */
class ProgressiveAccessibilityBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The file URL generator service.
   *
   * @var \Drupal\Core\File\FileUrlGenerator
   */
  protected $fileUrlGenerator;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Required for dependency injection.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);

    $instance->fileUrlGenerator = $container->get('file_url_generator');
    $instance->fileSystem = $container->get('file_system');
    $instance->moduleHandler = $container->get('module_handler');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $config = $this->getConfiguration();

    $form['widget_icon'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Widget icon'),
      '#description' => $this->t('Examples: icon.png (for a file in the public filesystem), public://icon.png, or themes/contrib/gin/icon.png.'),
      '#default_value' => $config['widget_icon'] ?? NULL,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->setConfigurationValue('widget_icon', $form_state->getValue('widget_icon'));
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    if ($form_state->getValue('widget_icon')) {
      $path = $this->validatePath($form_state->getValue('widget_icon'));
      if (!$path) {
        $form_state->setErrorByName('widget_icon', $this->t('The icon path is invalid.'));
      }
    }
  }

  /**
   * Helper function for the system_theme_settings form.
   *
   * Attempts to validate normal system paths, paths relative to the
   * public files directory, or stream wrapper URIs.
   * If the given path is any of the above, returns a valid path or URI that
   * the theme system can display.
   *
   * @param string $path
   *   A path relative to the Drupal root or to the public files directory, or
   *   a stream wrapper URI.
   *
   * @return mixed
   *   A valid path that can be displayed through the theme system or FALSE if
   *   the path could not be validated.
   */
  protected function validatePath($path) {
    // Absolute local file paths are invalid.
    if ($this->fileSystem->realpath($path) == $path) {
      return FALSE;
    }
    // A path relative to the Drupal root or a fully qualified URI is valid.
    if (is_file($path)) {
      return $path;
    }
    // Prepend 'public://' for relative file paths within the public filesystem.
    if (StreamWrapperManager::getScheme($path) === FALSE) {
      $path = 'public://' . $path;
    }
    if (is_file($path)) {
      return $path;
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $config = $this->getConfiguration();

    return [
      '#theme' => 'asw_widget',
      '#widget_icon' => !empty($config['widget_icon'])
        // Generate a URL for the icon if it is a valid file path.
        ? $this->fileUrlGenerator->generateString($config['widget_icon'])
        // Otherwise, use the default icon from the module.
        : $this->moduleHandler->getModule('progressive_accessibility_widget')->getPath() . '/icons/access-widget-icon.svg',
      '#attached' => [
        'library' => [
          'progressive_accessibility_widget/progressive_accessibility_widget',
        ],
      ],
    ];
  }

}
