# Extending Prometheus Metrics

This guide explains how to create custom metrics for your Drupal site using the Prometheus Metrics module.

## Overview

The Prometheus Metrics module provides several ways to add custom metrics:

1. Event Subscribers - Listen to Drupal events and record metrics
2. Service Integration - Use the PrometheusMetricsInterface service directly
3. Custom Events - Create and dispatch your own events

## 1. Using Event Subscribers

The module already includes event subscribers for HTTP requests and entity CRUD operations. You can create your own event subscriber to track custom metrics.

Example of creating a custom event subscriber:

```php
<?php

namespace Drupal\your_module\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Custom metrics collector.
 */
class CustomMetricsSubscriber implements EventSubscriberInterface {

  /**
   * The prometheus metrics service.
   *
   * @var \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface
   */
  private $prometheusMetrics;

  /**
   * The namespace for metrics.
   *
   * @var string
   */
  private $namespace;

  /**
   * Constructs the subscriber.
   */
  public function __construct(
    PrometheusMetricsInterface $prometheusMetrics,
    ConfigFactoryInterface $configFactory
  ) {
    $this->prometheusMetrics = $prometheusMetrics;
    $config = $configFactory->get('prometheus_metrics.configuration');
    $this->namespace = $config->get('metrics_namespace') ?: 'drupal';
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::TERMINATE => ['onTerminate', 0],
    ];
  }

  /**
   * Records custom metrics.
   */
  public function onTerminate(TerminateEvent $event) {
    // Example: Track number of active users
    $counter = $this->prometheusMetrics->getCounter(
      $this->namespace,
      'active_users_total',
      'Total number of active users',
      ['status']
    );
    $counter->inc(['active']);

    // Example: Track memory usage
    $gauge = $this->prometheusMetrics->getGauge(
      $this->namespace,
      'memory_usage_bytes',
      'Current memory usage in bytes',
      ['type']
    );
    $gauge->set(memory_get_usage(true), ['peak']);
  }
}
```

Register your service in `your_module.services.yml`:

```yaml
services:
  your_module.custom_metrics:
    class: Drupal\your_module\EventSubscriber\CustomMetricsSubscriber
    arguments: ['@prometheus_metrics.promphp_bridge', '@config.factory']
    tags:
      - { name: 'event_subscriber' }
```

## 2. Direct Service Integration

You can inject the PrometheusMetricsInterface service directly into your services or controllers:

```php
<?php

namespace Drupal\your_module\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;

/**
 * Custom controller with metrics.
 */
class CustomController extends ControllerBase {

  /**
   * The prometheus metrics service.
   *
   * @var \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface
   */
  protected $prometheusMetrics;

  /**
   * Constructs the controller.
   */
  public function __construct(PrometheusMetricsInterface $prometheus_metrics) {
    $this->prometheusMetrics = $prometheus_metrics;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('prometheus_metrics.promphp_bridge')
    );
  }

  /**
   * Example action that records metrics.
   */
  public function customAction() {
    // Record a custom metric
    $counter = $this->prometheusMetrics->getCounter(
      'your_module',
      'custom_action_total',
      'Number of times custom action was performed',
      ['status']
    );
    $counter->inc(['success']);

    return ['#markup' => 'Action completed'];
  }
}
```

## 3. Custom Events

You can create and dispatch your own events to track specific metrics:

```php
<?php

namespace Drupal\your_module\Event;

use Symfony\Contracts\EventDispatcher\Event;

/**
 * Custom event for metrics.
 */
class CustomMetricsEvent extends Event {
  // Add properties and methods as needed
}
```

Dispatch the event:

```php
$event = new CustomMetricsEvent();
\Drupal::service('event_dispatcher')->dispatch('your_module.custom_metrics', $event);
```

Create an event subscriber to handle it:

```php
<?php

namespace Drupal\your_module\EventSubscriber;

use Drupal\your_module\Event\CustomMetricsEvent;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Handles custom metrics events.
 */
class CustomMetricsSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      'your_module.custom_metrics' => ['onCustomMetrics', 0],
    ];
  }

  /**
   * Records metrics from custom event.
   */
  public function onCustomMetrics(CustomMetricsEvent $event) {
    // Record your metrics here
  }
}
```

## Available Metric Types

The module supports the following Prometheus metric types:

1. Counter - For values that only increase
2. Histogram - For tracking distributions of values
3. Gauge - For values that can go up and down

## Best Practices

1. **Namespace Usage**
   - Use a unique namespace for your metrics
   - Follow the pattern: `your_module_metric_name`
   - Avoid using generic names that might conflict with other modules

2. **Labels**
   - Use meaningful labels to categorize your metrics
   - Keep label values consistent
   - Don't use high-cardinality values as labels

3. **Help Text**
   - Provide clear, descriptive help text for each metric
   - Include units in the help text when applicable

4. **Testing**
   - Write unit tests for your metrics collectors
   - Test edge cases and error conditions
   - Verify metric names and labels follow Prometheus conventions

## Example Implementation

Here's a complete example of a custom module that tracks user login attempts:

```php
<?php

namespace Drupal\login_metrics\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Tracks login attempt metrics.
 */
class LoginMetricsSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['onRequest', 0],
    ];
  }

  /**
   * Records login attempt metrics.
   */
  public function onRequest(RequestEvent $event) {
    if ($event->isMainRequest() && $event->getRequest()->get('_route') === 'user.login') {
      $counter = $this->prometheusMetrics->getCounter(
        'login_metrics',
        'login_attempts_total',
        'Total number of login attempts',
        ['status']
      );
      $counter->inc(['attempt']);
    }
  }
}
```

## Troubleshooting

1. **Metrics Not Appearing**
   - Check the metrics endpoint at `/metrics`
   - Verify your service is properly registered
   - Check Drupal logs for errors

2. **Performance Issues**
   - Use appropriate metric types
   - Avoid recording too many unique label combinations
   - Consider using histograms for timing data

3. **Storage Issues**
   - Monitor storage usage if using Redis or APC
   - Consider implementing metric cleanup
   - Use appropriate storage backend for your needs 