<?php

declare(strict_types=1);

namespace Drupal\prometheus_metrics_commerce\EventSubscriber;

use Drupal\commerce_order\Event\OrderEvent;
use Drupal\commerce_order\Event\OrderEvents;
use Drupal\Core\Utility\Error;
use Drupal\prometheus_metrics_commerce\Metrics\OrderRevenue;
use Drupal\prometheus_metrics_commerce\Metrics\OrderTotal;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for tracking order metrics in real-time.
 */
class OrderMetricsSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a new OrderMetricsSubscriber instance.
   *
   * @param \Drupal\prometheus_metrics_commerce\Metrics\OrderTotal $orderTotal
   *   The order total metric.
   * @param \Drupal\prometheus_metrics_commerce\Metrics\OrderRevenue $orderRevenue
   *   The order revenue metric.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected OrderTotal $orderTotal,
    protected OrderRevenue $orderRevenue,
    #[Autowire(service: 'logger.channel.prometheus_metrics')]
    protected LoggerInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      OrderEvents::ORDER_INSERT => ['onOrderInsert'],
      OrderEvents::ORDER_UPDATE => ['onOrderUpdate'],
      OrderEvents::ORDER_DELETE => ['onOrderDelete'],
    ];
  }

  /**
   * Handles order insertion events.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The order event.
   */
  public function onOrderInsert(OrderEvent $event): void {
    try {
      $order = $event->getOrder();
      $this->orderTotal->incrementFromOrder($order);
      $this->orderRevenue->incrementFromOrder($order);
    }
    catch (\Exception $e) {
      Error::logException($this->logger, $e);
    }
  }

  /**
   * Handles order update events.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The order event.
   */
  public function onOrderUpdate(OrderEvent $event): void {
    $order = $event->getOrder();

    // Check if we have access to the original entity for comparison.
    if (!$order->getOriginal()) {
      return;
    }

    try {
      $this->orderTotal->updateFromOrders($order->getOriginal(), $order);
      $this->orderRevenue->updateFromOrders($order->getOriginal(), $order);
    }
    catch (\Exception $e) {
      Error::logException($this->logger, $e);
    }
  }

  /**
   * Handles order delete events.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The order event.
   */
  public function onOrderDelete(OrderEvent $event): void {
    try {
      $order = $event->getOrder();
      $this->orderTotal->decrementFromOrder($order);
      $this->orderRevenue->decrementFromOrder($order);
    }
    catch (\Exception $e) {
      Error::logException($this->logger, $e);
    }
  }

}
