<?php

declare(strict_types=1);

namespace Drupal\prometheus_metrics_commerce\EventSubscriber;

use Drupal\commerce_product\Event\ProductEvent;
use Drupal\commerce_product\Event\ProductEvents;
use Drupal\Core\Utility\Error;
use Drupal\prometheus_metrics_commerce\Metrics\Product;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for tracking product metrics in real-time.
 */
class ProductMetricsSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a new ProductMetricsSubscriber instance.
   *
   * @param \Drupal\prometheus_metrics_commerce\Metrics\Product $product
   *   The product metric.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected Product $product,
    #[Autowire(service: 'logger.channel.prometheus_metrics')]
    protected LoggerInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ProductEvents::PRODUCT_INSERT => ['onProductInsert'],
      ProductEvents::PRODUCT_UPDATE => ['onProductUpdate'],
      ProductEvents::PRODUCT_DELETE => ['onProductDelete'],
    ];
  }

  /**
   * Handles product insertion events.
   *
   * @param \Drupal\commerce_product\Event\ProductEvent $event
   *   The product event.
   */
  public function onProductInsert(ProductEvent $event): void {
    try {
      $this->product->incrementFromProduct($event->getProduct());
    }
    catch (\Exception $e) {
      Error::logException($this->logger, $e);
    }
  }

  /**
   * Handles product update events.
   *
   * @param \Drupal\commerce_product\Event\ProductEvent $event
   *   The product event.
   */
  public function onProductUpdate(ProductEvent $event): void {
    $product = $event->getProduct();

    // Check if we have access to the original entity for comparison.
    if (!$product->getOriginal()) {
      return;
    }

    try {
      $this->product->updateFromProducts($product->getOriginal(), $product);
    }
    catch (\Exception $e) {
      Error::logException($this->logger, $e);
    }
  }

  /**
   * Handles product delete events.
   *
   * @param \Drupal\commerce_product\Event\ProductEvent $event
   *   The product event.
   */
  public function onProductDelete(ProductEvent $event): void {
    try {
      $this->product->decrementFromProduct($event->getProduct());
    }
    catch (\Exception $e) {
      Error::logException($this->logger, $e);
    }
  }

}
