<?php

declare(strict_types=1);

namespace Drupal\prometheus_metrics_commerce\EventSubscriber;

use Drupal\Core\Utility\Error;
use Drupal\prometheus_metrics_commerce\Metrics\OrderRevenue;
use Drupal\prometheus_metrics_commerce\Metrics\OrderTotal;
use Drupal\prometheus_metrics_commerce\Metrics\Product;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\prometheus_metrics\Events\RebuildMetricsEvent;

/**
 * Rebuild metrics when cache is rebuilt.
 *
 * This ensures that if custom update hooks modify order states directly in the
 * database, that metrics are updated accordingly.
 */
class RebuildMetricsEventSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a new RebuildMetricsEventSubscriber instance.
   *
   * @param \Drupal\prometheus_metrics_commerce\Metrics\OrderTotal $orderTotal
   *   The order total metric.
   * @param \Drupal\prometheus_metrics_commerce\Metrics\OrderRevenue $orderRevenue
   *   The order revenue metric.
   * @param \Drupal\prometheus_metrics_commerce\Metrics\Product $product
   *   The product metric.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected OrderTotal $orderTotal,
    protected OrderRevenue $orderRevenue,
    protected Product $product,
    #[Autowire(service: 'logger.channel.prometheus_metrics')]
    protected LoggerInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      RebuildMetricsEvent::class => ['onRebuildMetrics'],
    ];
  }

  /**
   * Handles the metrics rebuild event.
   *
   * @param \Drupal\prometheus_metrics\Events\RebuildMetricsEvent $event
   *   The rebuild metrics event.
   *
   * @throws \Exception
   *   If an exception occurs while rebuilding the metrics.
   */
  public function onRebuildMetrics(RebuildMetricsEvent $event): void {
    try {
      $this->orderTotal->rebuild();
    }
    catch (\Exception $e) {
      // Metrics should never prevent the cache from rebuilding.
      Error::logException($this->logger, $e);
    }
    try {
      $this->orderRevenue->rebuild();
    }
    catch (\Exception $e) {
      // Metrics should never prevent the cache from rebuilding.
      Error::logException($this->logger, $e);
    }
    try {
      $this->product->rebuild();
    }
    catch (\Exception $e) {
      // Metrics should never prevent the cache from rebuilding.
      Error::logException($this->logger, $e);
    }
  }

}
