<?php

declare(strict_types=1);

namespace Drupal\prometheus_metrics_commerce\Metrics;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Drupal\prometheus_metrics\RebuildableMetricInterface;

/**
 * Base class for commerce metrics.
 *
 * Provides common functionality for all commerce metrics including namespace
 * and site name retrieval.
 */
abstract class CommerceMetricBase implements RebuildableMetricInterface {

  /**
   * The cached metrics namespace.
   *
   * @var string
   */
  protected string $namespace = '';

  /**
   * The cached site name.
   *
   * @var string
   */
  protected string $siteName = '';

  /**
   * Constructs a new BaseCommerceMetric instance.
   *
   * @param \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface $metricsService
   *   The Prometheus metrics service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(
    protected PrometheusMetricsInterface $metricsService,
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * Gets the metrics namespace.
   *
   * @return string
   *   The metrics namespace.
   */
  protected function getNamespace(): string {
    if (!$this->namespace) {
      $this->namespace = $this->configFactory->get('prometheus_metrics.configuration')->get('metrics_namespace') ?? 'drupal';
    }
    return $this->namespace;
  }

  /**
   * Gets the site name.
   *
   * @return string
   *   The site name.
   */
  protected function getSiteName(): string {
    if (!$this->siteName) {
      $this->siteName = $this->configFactory->get('system.site')->get('name') ?? 'default';
    }
    return $this->siteName;
  }

}
