<?php

declare(strict_types=1);

namespace Drupal\prometheus_metrics_commerce\Metrics;

use Drupal\address\AddressInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Prometheus\Gauge;

/**
 * Manages the commerce_orders_revenue_total metric.
 */
class OrderRevenue extends CommerceMetricBase {

  /**
   * Constructs a new OrderRevenue instance.
   *
   * @param \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface $metricsService
   *   The Prometheus metrics service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    PrometheusMetricsInterface $metricsService,
    ConfigFactoryInterface $configFactory,
    protected Connection $database,
  ) {
    parent::__construct($metricsService, $configFactory);
  }

  /**
   * {@inheritdoc}
   */
  public function rebuild(): void {
    $namespace = $this->getNamespace();
    $siteName = $this->getSiteName();

    // Use database query with joins to sum revenue by:
    // state, currency, and administrative area.
    // Unfortunately, EFQ doesn't support aggregate queries with joins across
    // entity references, so we need to use the database layer directly.
    $query = $this->database->select('commerce_order', 'o');
    $query->leftJoin('profile', 'p', 'o.billing_profile__target_id = p.profile_id');
    $query->leftJoin('profile__address', 'pa', 'p.profile_id = pa.entity_id');
    $query->addField('o', 'state');
    $query->addField('o', 'total_price__currency_code', 'currency_code');
    $query->addField('pa', 'address_administrative_area');
    $query->addExpression('SUM(o.total_price__number)', 'revenue_sum');
    $query->condition('o.total_price__number', NULL, 'IS NOT NULL');
    $query->groupBy('o.state');
    $query->groupBy('o.total_price__currency_code');
    $query->groupBy('pa.address_administrative_area');

    $results = $query->execute()->fetchAll();

    $revenueGaugeWithArea = $this->metricsService->getGauge(
          $namespace,
          'commerce_orders_revenue_total',
          'Total revenue from commerce orders',
          ['state', 'currency', 'administrative_area', 'site_name']
      );
    assert($revenueGaugeWithArea instanceof Gauge);

    // Set metrics.
    foreach ($results as $result) {
      $state = $result->state ?? NULL;
      $currency = $result->currency_code ?? '';
      $administrativeArea = $result->address_administrative_area ?? NULL;
      $revenue = (float) ($result->revenue_sum ?? 0);

      if ($state !== NULL && $revenue > 0) {
        $revenueGaugeWithArea->set($revenue, [$state, $currency, $administrativeArea, $siteName]);
      }
    }
  }

  /**
   * Increments revenue from an order entity.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   */
  public function incrementFromOrder(OrderInterface $order): void {
    $this->doModifyGauge($order);
  }

  /**
   * Decrements revenue from an order entity.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   */
  public function decrementFromOrder(OrderInterface $order): void {
    $this->doModifyGauge($order, TRUE);
  }

  /**
   * Updates revenue metrics when an order changes.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $oldOrder
   *   The original order entity.
   * @param \Drupal\commerce_order\Entity\OrderInterface $newOrder
   *   The updated order entity.
   */
  public function updateFromOrders(OrderInterface $oldOrder, OrderInterface $newOrder): void {
    $oldPrice = $oldOrder->getTotalPrice();
    $newPrice = $newOrder->getTotalPrice();

    // Track revenue changes based on three scenarios.
    if ($oldPrice && $newPrice) {
      // Both prices exist - check if anything changed.
      $oldState = $oldOrder->getState()->getId();
      $newState = $newOrder->getState()->getId();
      $oldAmount = (float) $oldPrice->getNumber();
      $newAmount = (float) $newPrice->getNumber();
      $oldCurrency = $oldPrice->getCurrencyCode();
      $newCurrency = $newPrice->getCurrencyCode();
      $oldAdministrativeArea = $this->getAdministrativeArea($oldOrder);
      $newAdministrativeArea = $this->getAdministrativeArea($newOrder);

      // Only update if something changed.
      if ($oldState !== $newState || $oldAmount !== $newAmount || $oldCurrency !== $newCurrency || $oldAdministrativeArea !== $newAdministrativeArea) {
        $this->decrementFromOrder($oldOrder);
        $this->incrementFromOrder($newOrder);
      }
    }
    elseif (!$oldPrice && $newPrice) {
      // Price was added.
      $this->incrementFromOrder($newOrder);
    }
    elseif ($oldPrice && !$newPrice) {
      // Price was removed.
      $this->decrementFromOrder($oldOrder);
    }
  }

  /**
   * Gets the administrative area from an order's billing address.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   *
   * @return string|null
   *   The administrative area code, or NULL if not available.
   */
  protected function getAdministrativeArea(OrderInterface $order): ?string {
    $billingProfile = $order->getBillingProfile();
    if (!$billingProfile) {
      return NULL;
    }

    $address = $billingProfile->get('address')->first();
    assert($address instanceof AddressInterface);
    return $address?->getAdministrativeArea();
  }

  /**
   * Gets the revenue gauge with all labels.
   *
   * @return \Prometheus\Gauge
   *   The revenue gauge.
   */
  private function getGauge(): Gauge {
    $gauge = $this->metricsService->getGauge(
          $this->getNamespace(),
          'commerce_orders_revenue_total',
          'Total revenue from commerce orders',
          ['state', 'currency', 'administrative_area', 'site_name']
      );
    assert($gauge instanceof Gauge);
    return $gauge;
  }

  /**
   * Modifies the revenue gauge for an order.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   * @param bool $decrement
   *   Whether to decrement (TRUE) or increment (FALSE).
   */
  public function doModifyGauge(OrderInterface $order, bool $decrement = FALSE): void {
    $totalPrice = $order->getTotalPrice();
    if (!$totalPrice) {
      return;
    }

    $state = $order->getState()->getId();
    $amount = (float) $totalPrice->getNumber();
    $currency = $totalPrice->getCurrencyCode();
    $administrativeArea = $this->getAdministrativeArea($order);
    $siteName = $this->getSiteName();

    $gauge = $this->getGauge();

    if (!$decrement) {
      $gauge->incBy($amount, [$state, $currency, $administrativeArea, $siteName]);
    }
    else {
      $gauge->decBy($amount, [$state, $currency, $administrativeArea, $siteName]);
    }
  }

}
