<?php

declare(strict_types=1);

namespace Drupal\prometheus_metrics_commerce\Metrics;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Prometheus\Gauge;

/**
 * Manages the commerce_orders_total metric.
 */
class OrderTotal extends CommerceMetricBase {

  /**
   * Constructs a new OrderTotal instance.
   *
   * @param \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface $metricsService
   *   The Prometheus metrics service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    PrometheusMetricsInterface $metricsService,
    ConfigFactoryInterface $configFactory,
    protected Connection $database,
  ) {
    parent::__construct($metricsService, $configFactory);
  }

  /**
   * {@inheritdoc}
   */
  public function rebuild(): void {
    $namespace = $this->getNamespace();
    $siteName = $this->getSiteName();

    // Use database query with joins to
    // count orders by state and administrative area.
    // Unfortunately, EFQ doesn't support aggregate queries with joins across
    // entity references, so we need to use the database layer directly.
    $query = $this->database->select('commerce_order', 'o');
    $query->leftJoin('profile', 'p', 'o.billing_profile__target_id = p.profile_id');
    $query->leftJoin('profile__address', 'pa', 'p.profile_id = pa.entity_id');
    $query->addField('o', 'state');
    $query->addField('pa', 'address_administrative_area');
    $query->addExpression('COUNT(o.order_id)', 'order_count');
    $query->groupBy('o.state');
    $query->groupBy('pa.address_administrative_area');

    $results = $query->execute()->fetchAll();

    $gaugeWithArea = $this->metricsService->getGauge(
          $namespace,
          'commerce_orders_total',
          'Total number of commerce orders',
          ['state', 'administrative_area', 'site_name']
      );
    assert($gaugeWithArea instanceof Gauge);

    // Set metrics.
    foreach ($results as $result) {
      $state = $result->state ?? NULL;
      $administrativeArea = $result->address_administrative_area ?? NULL;
      $count = (int) ($result->order_count ?? 0);

      $gaugeWithArea->set($count, [$state, $administrativeArea, $siteName]);
    }
  }

  /**
   * Increments the order count from an order entity.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   */
  public function incrementFromOrder(OrderInterface $order): void {
    $state = $order->getState()->getId();
    $administrativeArea = $this->getAdministrativeArea($order);

    $namespace = $this->getNamespace();
    $siteName = $this->getSiteName();
    $gauge = $this->metricsService->getGauge(
          $namespace,
          'commerce_orders_total',
          'Total number of commerce orders',
          ['state', 'administrative_area', 'site_name']
      );
    assert($gauge instanceof Gauge);
    $gauge->inc([$state, $administrativeArea, $siteName]);
  }

  /**
   * Decrements the order count from an order entity.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   */
  public function decrementFromOrder(OrderInterface $order): void {
    $state = $order->getState()->getId();
    $administrativeArea = $this->getAdministrativeArea($order);

    $namespace = $this->getNamespace();
    $siteName = $this->getSiteName();
    $gauge = $this->metricsService->getGauge(
          $namespace,
          'commerce_orders_total',
          'Total number of commerce orders',
          ['state', 'administrative_area', 'site_name']
      );
    assert($gauge instanceof Gauge);
    $gauge->dec([$state, $administrativeArea, $siteName]);
  }

  /**
   * Updates metrics when an order changes.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $oldOrder
   *   The original order entity.
   * @param \Drupal\commerce_order\Entity\OrderInterface $newOrder
   *   The updated order entity.
   */
  public function updateFromOrders(OrderInterface $oldOrder, OrderInterface $newOrder): void {
    $oldState = $oldOrder->getState()->getId();
    $newState = $newOrder->getState()->getId();
    $oldAdministrativeArea = $this->getAdministrativeArea($oldOrder);
    $newAdministrativeArea = $this->getAdministrativeArea($newOrder);

    // Only update if state or administrative area changed.
    if ($oldState !== $newState || $oldAdministrativeArea !== $newAdministrativeArea) {
      $this->decrementFromOrder($oldOrder);
      $this->incrementFromOrder($newOrder);
    }
  }

  /**
   * Gets the administrative area from an order's billing address.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   *
   * @return string|null
   *   The administrative area code, or NULL if not available.
   */
  protected function getAdministrativeArea(OrderInterface $order): ?string {
    $billingProfile = $order->getBillingProfile();
    if (!$billingProfile) {
      return NULL;
    }

    $address = $billingProfile->get('address')->first();
    return $address?->getAdministrativeArea();
  }

}
