<?php

declare(strict_types=1);

namespace Drupal\prometheus_metrics_commerce\Metrics;

use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Prometheus\Gauge;

/**
 * Manages the commerce_products_total metric.
 */
class Product extends CommerceMetricBase {

  /**
   * Constructs a new Product instance.
   *
   * @param \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface $metricsService
   *   The Prometheus metrics service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    PrometheusMetricsInterface $metricsService,
    ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($metricsService, $configFactory);
  }

  /**
   * {@inheritdoc}
   */
  public function rebuild(): void {
    $namespace = $this->getNamespace();
    $siteName = $this->getSiteName();

    $query = $this->entityTypeManager
      ->getStorage('commerce_product')
      ->getAggregateQuery()
      ->accessCheck(FALSE)
      ->condition('status', 1)
      ->groupBy('type')
      ->aggregate('product_id', 'COUNT');

    $results = $query->execute();

    $gauge = $this->metricsService->getGauge(
          $namespace,
          'commerce_products_total',
          'Total number of published commerce products',
          ['type', 'site_name']
      );
    assert($gauge instanceof Gauge);

    foreach ($results as $result) {
      $type = $result['type'];
      $count = $result['product_id_count'];
      $gauge->set((int) $count, [$type, $siteName]);
    }
  }

  /**
   * Increments the product count from a product entity.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   */
  public function incrementFromProduct(ProductInterface $product): void {
    if (!$product->isPublished()) {
      return;
    }

    $type = $product->bundle();
    $namespace = $this->getNamespace();
    $siteName = $this->getSiteName();

    $gauge = $this->metricsService->getGauge(
          $namespace,
          'commerce_products_total',
          'Total number of published commerce products',
          ['type', 'site_name']
      );
    assert($gauge instanceof Gauge);
    $gauge->inc([$type, $siteName]);
  }

  /**
   * Decrements the product count from a product entity.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   */
  public function decrementFromProduct(ProductInterface $product): void {
    if (!$product->isPublished()) {
      return;
    }

    $type = $product->bundle();
    $namespace = $this->getNamespace();
    $siteName = $this->getSiteName();

    $gauge = $this->metricsService->getGauge(
          $namespace,
          'commerce_products_total',
          'Total number of published commerce products',
          ['type', 'site_name']
      );
    assert($gauge instanceof Gauge);
    $gauge->dec([$type, $siteName]);
  }

  /**
   * Updates metrics when a product changes.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $oldProduct
   *   The original product entity.
   * @param \Drupal\commerce_product\Entity\ProductInterface $newProduct
   *   The updated product entity.
   */
  public function updateFromProducts(ProductInterface $oldProduct, ProductInterface $newProduct): void {
    $wasPublished = $oldProduct->isPublished();
    $isPublished = $newProduct->isPublished();

    // Only update if published status changed.
    if ($wasPublished !== $isPublished) {
      if ($isPublished) {
        $this->incrementFromProduct($newProduct);
      }
      else {
        $this->decrementFromProduct($oldProduct);
      }
    }
  }

}
