<?php

declare(strict_types=1);

namespace Drupal\Tests\prometheus_metrics_commerce\Kernel;

use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Entity\OrderItem;
use Drupal\commerce_price\Price;
use Drupal\profile\Entity\Profile;
use Drupal\prometheus_metrics_commerce\Metrics\OrderRevenue;
use Drupal\Tests\commerce_order\Kernel\OrderKernelTestBase;
use PHPUnit\Framework\Attributes\Group;
use Prometheus\Gauge;

/**
 * Tests the OrderRevenue metric class.
 */
#[Group("prometheus_metrics_commerce")]
class OrderRevenueTest extends OrderKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'prometheus_metrics',
    'prometheus_metrics_commerce',
  ];

  /**
   * The OrderRevenue metric service.
   *
   * @var \Drupal\prometheus_metrics_commerce\Metrics\OrderRevenue
   */
  protected OrderRevenue $orderRevenueMetric;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Import EUR currency for tests.
    $currency_importer = $this->container->get('commerce_price.currency_importer');
    $currency_importer->import('EUR');

    // Set site name for testing.
    $this->config('system.site')->set('name', 'Test Site')->save();

    // Set up prometheus_metrics configuration.
    $this->config('prometheus_metrics.configuration')->set('metrics_namespace', 'drupal')->save();

    $this->orderRevenueMetric = $this->container->get(OrderRevenue::class);
  }

  /**
   * Tests rebuild() sums revenue by state, currency, and administrative area.
   */
  public function testRebuild(): void {
    // Create orders with different states, currencies, and areas.
    $this->createOrder('draft', 'CA', '100.00', 'USD');
    $this->createOrder('draft', 'CA', '50.00', 'USD');
    $this->createOrder('draft', 'NY', '75.00', 'USD');
    $this->createOrder('completed', 'CA', '200.00', 'USD');
    $this->createOrder('completed', NULL, '150.00', 'USD');
    $this->createOrder('draft', 'CA', '50.00', 'EUR');

    // Mock the Prometheus gauge to capture set() calls.
    $gaugeCalls = [];
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->exactly(5))
      ->method('set')
      ->willReturnCallback(
              function ($revenue, $labels) use (&$gaugeCalls) {
                  $gaugeCalls[] = ['revenue' => $revenue, 'labels' => $labels];
              }
          );

    // Mock the Prometheus metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->with(
              'drupal',
              'commerce_orders_revenue_total',
              'Total revenue from commerce orders',
              ['state', 'currency', 'administrative_area', 'site_name']
          )
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setValue($this->orderRevenueMetric, $metricsService);

    // Run rebuild.
    $this->orderRevenueMetric->rebuild();

    // Verify the correct revenue sums were set.
    $this->assertCount(5, $gaugeCalls);

    // Sort for consistent assertion.
    usort(
          $gaugeCalls, function ($a, $b) {
              $stateCompare = strcmp($a['labels'][0], $b['labels'][0]);
            if ($stateCompare !== 0) {
                return $stateCompare;
            }
              $currencyCompare = strcmp($a['labels'][1], $b['labels'][1]);
            if ($currencyCompare !== 0) {
                return $currencyCompare;
            }
              return strcmp($a['labels'][2] ?? '', $b['labels'][2] ?? '');
          }
      );

    // completed, USD, NULL.
    $this->assertEquals(150.00, $gaugeCalls[0]['revenue']);
    $this->assertEquals(['completed', 'USD', NULL, 'Test Site'], $gaugeCalls[0]['labels']);

    // completed, USD, CA.
    $this->assertEquals(200.00, $gaugeCalls[1]['revenue']);
    $this->assertEquals(['completed', 'USD', 'CA', 'Test Site'], $gaugeCalls[1]['labels']);

    // draft, EUR, CA.
    $this->assertEquals(50.00, $gaugeCalls[2]['revenue']);
    $this->assertEquals(['draft', 'EUR', 'CA', 'Test Site'], $gaugeCalls[2]['labels']);

    // draft, USD, CA.
    $this->assertEquals(150.00, $gaugeCalls[3]['revenue']);
    $this->assertEquals(['draft', 'USD', 'CA', 'Test Site'], $gaugeCalls[3]['labels']);

    // draft, USD, NY.
    $this->assertEquals(75.00, $gaugeCalls[4]['revenue']);
    $this->assertEquals(['draft', 'USD', 'NY', 'Test Site'], $gaugeCalls[4]['labels']);
  }

  /**
   * Tests incrementFromOrder() increments revenue.
   */
  public function testIncrementFromOrder(): void {
    $order = $this->createOrder('draft', 'TX', '123.45', 'USD');

    // Mock the Prometheus gauge.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('incBy')
      ->with(123.45, ['draft', 'USD', 'TX', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->incrementFromOrder($order);
  }

  /**
   * Tests incrementFromOrder() handles orders without price.
   */
  public function testIncrementFromOrderWithoutPrice(): void {
    $order = Order::create(
          [
            'type' => 'default',
            'state' => 'draft',
            'store_id' => $this->store->id(),
          ]
      );
    $order->save();

    // Mock the metrics service - should not be called.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->never())
      ->method('getGauge');

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->incrementFromOrder($order);
  }

  /**
   * Tests decrementFromOrder() decrements revenue.
   */
  public function testDecrementFromOrder(): void {
    $order = $this->createOrder('completed', 'FL', '99.99', 'EUR');

    // Mock the Prometheus gauge.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('decBy')
      ->with(99.99, ['completed', 'EUR', 'FL', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->decrementFromOrder($order);
  }

  /**
   * Tests updateFromOrders() when both prices exist and state changes.
   */
  public function testUpdateFromOrdersBothPricesStateChanged(): void {
    $oldOrder = $this->createOrder('draft', 'CA', '100.00', 'USD');
    $newOrder = clone $oldOrder;
    $newOrder->set('state', 'completed');

    // Mock the Prometheus gauge - should dec old and inc new.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('decBy')
      ->with(100.00, ['draft', 'USD', 'CA', 'Test Site']);
    $mockGauge->expects($this->once())
      ->method('incBy')
      ->with(100.00, ['completed', 'USD', 'CA', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->exactly(2))
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Tests updateFromOrders() when price amount changes.
   */
  public function testUpdateFromOrdersPriceAmountChanged(): void {
    $oldOrder = $this->createOrder('draft', 'CA', '100.00', 'USD');
    $newOrder = clone $oldOrder;
    $newOrder->set('total_price', new Price('150.00', 'USD'));

    // Mock the Prometheus gauge - should dec old and inc new.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('decBy')
      ->with(100.00, ['draft', 'USD', 'CA', 'Test Site']);
    $mockGauge->expects($this->once())
      ->method('incBy')
      ->with(150.00, ['draft', 'USD', 'CA', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->exactly(2))
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Tests updateFromOrders() when currency changes.
   */
  public function testUpdateFromOrdersCurrencyChanged(): void {
    $oldOrder = $this->createOrder('draft', 'CA', '100.00', 'USD');
    $newOrder = clone $oldOrder;
    $newOrder->set('total_price', new Price('100.00', 'EUR'));

    // Mock the Prometheus gauge - should dec old and inc new.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('decBy')
      ->with(100.00, ['draft', 'USD', 'CA', 'Test Site']);
    $mockGauge->expects($this->once())
      ->method('incBy')
      ->with(100.00, ['draft', 'EUR', 'CA', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->exactly(2))
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Tests updateFromOrders() when administrative area changes.
   */
  public function testUpdateFromOrdersAdministrativeAreaChanged(): void {
    $oldOrder = $this->createOrder('draft', 'CA', '100.00', 'USD');
    $newOrder = $this->createOrder('draft', 'NY', '100.00', 'USD');

    // Mock the Prometheus gauge - should dec old and inc new.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('decBy')
      ->with(100.00, ['draft', 'USD', 'CA', 'Test Site']);
    $mockGauge->expects($this->once())
      ->method('incBy')
      ->with(100.00, ['draft', 'USD', 'NY', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->exactly(2))
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Tests updateFromOrders() when price is added.
   */
  public function testUpdateFromOrdersPriceAdded(): void {
    $oldOrder = Order::create(
          [
            'type' => 'default',
            'state' => 'draft',
            'store_id' => $this->store->id(),
          ]
      );
    $oldOrder->save();

    $newOrder = clone $oldOrder;
    $newOrder->set('total_price', new Price('100.00', 'USD'));

    // Mock the Prometheus gauge - should only inc new.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('incBy')
      ->with(100.00, ['draft', 'USD', NULL, 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Tests updateFromOrders() when price is removed.
   */
  public function testUpdateFromOrdersPriceRemoved(): void {
    $oldOrder = $this->createOrder('draft', 'CA', '100.00', 'USD');

    $newOrder = clone $oldOrder;
    $newOrder->set('total_price', NULL);

    // Mock the Prometheus gauge - should only dec old.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('decBy')
      ->with(100.00, ['draft', 'USD', 'CA', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Tests updateFromOrders() does nothing when nothing changes.
   */
  public function testUpdateFromOrdersNoChange(): void {
    $oldOrder = $this->createOrder('draft', 'CA', '100.00', 'USD');
    $newOrder = clone $oldOrder;

    // Mock the metrics service - should not be called.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->never())
      ->method('getGauge');

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderRevenueMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderRevenueMetric, $metricsService);

    $this->orderRevenueMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Helper method to create an order with billing profile and price.
   *
   * @param string $state
   *   The order state.
   * @param string|null $administrativeArea
   *   The administrative area code, or NULL.
   * @param string $amount
   *   The price amount.
   * @param string $currency
   *   The currency code.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface
   *   The created order.
   */
  protected function createOrder(string $state, ?string $administrativeArea, string $amount, string $currency) {
    $profile = NULL;
    if ($administrativeArea !== NULL) {
      $profile = Profile::create(
            [
              'type' => 'customer',
              'address' => [
                'country_code' => 'US',
                'administrative_area' => $administrativeArea,
                'locality' => 'Test City',
                'postal_code' => '12345',
                'address_line1' => '123 Test St',
              ],
            ]
        );
      $profile->save();
    }

    // Create an order item with the specified price.
    $order_item = OrderItem::create(
          [
            'type' => 'test',
            'quantity' => 1,
            'unit_price' => new Price($amount, $currency),
          ]
      );
    $order_item->save();

    $order = Order::create(
          [
            'type' => 'default',
            'state' => $state,
            'store_id' => $this->store->id(),
            'billing_profile' => $profile,
            'order_items' => [$order_item],
          ]
      );
    $order->save();

    return $order;
  }

}
