<?php

declare(strict_types=1);

namespace Drupal\Tests\prometheus_metrics_commerce\Kernel;

use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Entity\OrderItem;
use Drupal\commerce_price\Price;
use Drupal\profile\Entity\Profile;
use Drupal\prometheus_metrics_commerce\Metrics\OrderTotal;
use Drupal\Tests\commerce_order\Kernel\OrderKernelTestBase;
use PHPUnit\Framework\Attributes\Group;
use Prometheus\Gauge;

/**
 * Tests the OrderTotal metric class.
 */
#[Group("prometheus_metrics_commerce")]
class OrderTotalTest extends OrderKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'prometheus_metrics',
    'prometheus_metrics_commerce',
  ];

  /**
   * The OrderTotal metric service.
   *
   * @var \Drupal\prometheus_metrics_commerce\Metrics\OrderTotal
   */
  protected OrderTotal $orderTotalMetric;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set site name for testing.
    $this->config('system.site')->set('name', 'Test Site')->save();

    // Set up prometheus_metrics configuration.
    $this->config('prometheus_metrics.configuration')->set('metrics_namespace', 'drupal')->save();

    $this->orderTotalMetric = $this->container->get(OrderTotal::class);
  }

  /**
   * Tests rebuild() counts orders by state and administrative area.
   */
  public function testRebuild(): void {
    // Create orders with different states and administrative areas.
    $this->createOrder('draft', 'CA');
    $this->createOrder('draft', 'CA');
    $this->createOrder('draft', 'NY');
    $this->createOrder('completed', 'CA');
    $this->createOrder('completed', NULL);

    // Mock the Prometheus gauge to capture set() calls.
    $gaugeCalls = [];
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->exactly(4))
      ->method('set')
      ->willReturnCallback(
              function ($count, $labels) use (&$gaugeCalls) {
                  $gaugeCalls[] = ['count' => $count, 'labels' => $labels];
              }
          );

    // Mock the Prometheus metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->with(
              'drupal',
              'commerce_orders_total',
              'Total number of commerce orders',
              ['state', 'administrative_area', 'site_name']
          )
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderTotalMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderTotalMetric, $metricsService);

    // Run rebuild.
    $this->orderTotalMetric->rebuild();

    // Verify the correct counts were set.
    $this->assertCount(4, $gaugeCalls);

    // Sort for consistent assertion.
    usort(
          $gaugeCalls, function ($a, $b) {
              $stateCompare = strcmp($a['labels'][0], $b['labels'][0]);
            if ($stateCompare !== 0) {
                return $stateCompare;
            }
              return strcmp($a['labels'][1] ?? '', $b['labels'][1] ?? '');
          }
      );

    // completed, NULL.
    $this->assertEquals(1, $gaugeCalls[0]['count']);
    $this->assertEquals(['completed', NULL, 'Test Site'], $gaugeCalls[0]['labels']);

    // completed, CA.
    $this->assertEquals(1, $gaugeCalls[1]['count']);
    $this->assertEquals(['completed', 'CA', 'Test Site'], $gaugeCalls[1]['labels']);

    // draft, CA.
    $this->assertEquals(2, $gaugeCalls[2]['count']);
    $this->assertEquals(['draft', 'CA', 'Test Site'], $gaugeCalls[2]['labels']);

    // draft, NY.
    $this->assertEquals(1, $gaugeCalls[3]['count']);
    $this->assertEquals(['draft', 'NY', 'Test Site'], $gaugeCalls[3]['labels']);
  }

  /**
   * Tests incrementFromOrder() increments gauge.
   */
  public function testIncrementFromOrder(): void {
    $order = $this->createOrder('draft', 'TX');

    // Mock the Prometheus gauge.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('inc')
      ->with(['draft', 'TX', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderTotalMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderTotalMetric, $metricsService);

    $this->orderTotalMetric->incrementFromOrder($order);
  }

  /**
   * Tests incrementFromOrder() handles NULL administrative area.
   */
  public function testIncrementFromOrderWithoutAddress(): void {
    $order = $this->createOrder('draft', NULL);

    // Mock the Prometheus gauge.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('inc')
      ->with(['draft', NULL, 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderTotalMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderTotalMetric, $metricsService);

    $this->orderTotalMetric->incrementFromOrder($order);
  }

  /**
   * Tests decrementFromOrder() decrements gauge.
   */
  public function testDecrementFromOrder(): void {
    $order = $this->createOrder('completed', 'FL');

    // Mock the Prometheus gauge.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('dec')
      ->with(['completed', 'FL', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderTotalMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderTotalMetric, $metricsService);

    $this->orderTotalMetric->decrementFromOrder($order);
  }

  /**
   * Tests updateFromOrders() when state changes.
   */
  public function testUpdateFromOrdersStateChanged(): void {
    $oldOrder = $this->createOrder('draft', 'CA');
    $newOrder = clone $oldOrder;
    $newOrder->set('state', 'completed');

    // Mock the Prometheus gauge - should dec old and inc new.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('dec')
      ->with(['draft', 'CA', 'Test Site']);
    $mockGauge->expects($this->once())
      ->method('inc')
      ->with(['completed', 'CA', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->exactly(2))
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderTotalMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderTotalMetric, $metricsService);

    $this->orderTotalMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Tests updateFromOrders() when administrative area changes.
   */
  public function testUpdateFromOrdersAdministrativeAreaChanged(): void {
    $oldOrder = $this->createOrder('draft', 'CA');
    $newOrder = $this->createOrder('draft', 'NY');

    // Mock the Prometheus gauge - should dec old and inc new.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('dec')
      ->with(['draft', 'CA', 'Test Site']);
    $mockGauge->expects($this->once())
      ->method('inc')
      ->with(['draft', 'NY', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->exactly(2))
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderTotalMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderTotalMetric, $metricsService);

    $this->orderTotalMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Tests updateFromOrders() does nothing when nothing changes.
   */
  public function testUpdateFromOrdersNoChange(): void {
    $oldOrder = $this->createOrder('draft', 'CA');
    $newOrder = clone $oldOrder;

    // Mock the metrics service - should not be called.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->never())
      ->method('getGauge');

    // Replace the service.
    $reflection = new \ReflectionClass($this->orderTotalMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->orderTotalMetric, $metricsService);

    $this->orderTotalMetric->updateFromOrders($oldOrder, $newOrder);
  }

  /**
   * Helper method to create an order with billing profile.
   *
   * @param string $state
   *   The order state.
   * @param string|null $administrativeArea
   *   The administrative area code, or NULL.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface
   *   The created order.
   */
  protected function createOrder(string $state, ?string $administrativeArea) {
    $profile = NULL;
    if ($administrativeArea !== NULL) {
      $profile = Profile::create(
            [
              'type' => 'customer',
              'address' => [
                'country_code' => 'US',
                'administrative_area' => $administrativeArea,
                'locality' => 'Test City',
                'postal_code' => '12345',
                'address_line1' => '123 Test St',
              ],
            ]
        );
      $profile->save();
    }

    // Create an order item with a price.
    $order_item = OrderItem::create(
          [
            'type' => 'test',
            'quantity' => 1,
            'unit_price' => new Price('100.00', 'USD'),
          ]
      );
    $order_item->save();

    $order = Order::create(
          [
            'type' => 'default',
            'state' => $state,
            'store_id' => $this->store->id(),
            'billing_profile' => $profile,
            'order_items' => [$order_item],
          ]
      );
    $order->save();

    return $order;
  }

}
